"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LegacyService = void 0;

var _rxjs = require("rxjs");

var _operators = require("rxjs/operators");

var _dev = require("../dev");

var _http = require("../http");

var _plugins = require("./plugins");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function getLegacyRawConfig(config) {
  const rawConfig = config.toRaw(); // Elasticsearch config is solely handled by the core and legacy platform
  // shouldn't have direct access to it.

  if (rawConfig.elasticsearch !== undefined) {
    delete rawConfig.elasticsearch;
  }

  return rawConfig;
}
/**
 * @public
 * @deprecated
 */


/** @internal */
class LegacyService {
  /** Symbol to represent the legacy platform as a fake "plugin". Used by the ContextService */
  constructor(coreContext) {
    this.coreContext = coreContext;

    _defineProperty(this, "legacyId", Symbol());

    _defineProperty(this, "log", void 0);

    _defineProperty(this, "devConfig$", void 0);

    _defineProperty(this, "httpConfig$", void 0);

    _defineProperty(this, "kbnServer", void 0);

    _defineProperty(this, "configSubscription", void 0);

    _defineProperty(this, "setupDeps", void 0);

    _defineProperty(this, "update$", void 0);

    _defineProperty(this, "legacyRawConfig", void 0);

    _defineProperty(this, "legacyPlugins", void 0);

    _defineProperty(this, "settings", void 0);

    this.log = coreContext.logger.get('legacy-service');
    this.devConfig$ = coreContext.configService.atPath('dev').pipe((0, _operators.map)(rawConfig => new _dev.DevConfig(rawConfig)));
    this.httpConfig$ = coreContext.configService.atPath('server').pipe((0, _operators.map)(rawConfig => new _http.HttpConfig(rawConfig, coreContext.env)));
  }

  async setup(setupDeps) {
    this.setupDeps = setupDeps;
    this.update$ = this.coreContext.configService.getConfig$().pipe((0, _operators.tap)(config => {
      if (this.kbnServer !== undefined) {
        this.kbnServer.applyLoggingConfiguration(getLegacyRawConfig(config));
      }
    }), (0, _operators.tap)({
      error: err => this.log.error(err)
    }), (0, _operators.publishReplay)(1));
    this.configSubscription = this.update$.connect();
    this.settings = await this.update$.pipe((0, _operators.first)(), (0, _operators.map)(config => getLegacyRawConfig(config))).toPromise();
    const {
      pluginSpecs,
      pluginExtendedConfig,
      disabledPluginSpecs,
      uiExports
    } = await (0, _plugins.findLegacyPluginSpecs)(this.settings, this.coreContext.logger);
    this.legacyPlugins = {
      pluginSpecs,
      disabledPluginSpecs,
      uiExports
    };
    this.legacyRawConfig = pluginExtendedConfig; // check for unknown uiExport types

    if (uiExports.unknown && uiExports.unknown.length > 0) {
      throw new Error(`Unknown uiExport types: ${uiExports.unknown.map(({
        pluginSpec,
        type
      }) => `${type} from ${pluginSpec.getId()}`).join(', ')}`);
    }

    return {
      pluginSpecs,
      uiExports,
      pluginExtendedConfig
    };
  }

  async start(startDeps) {
    const {
      setupDeps
    } = this;

    if (!setupDeps || !this.legacyRawConfig || !this.legacyPlugins || !this.settings) {
      throw new Error('Legacy service is not setup yet.');
    }

    this.log.debug('starting legacy service'); // Receive initial config and create kbnServer/ClusterManager.

    if (this.coreContext.env.isDevClusterMaster) {
      await this.createClusterManager(this.legacyRawConfig);
    } else {
      this.kbnServer = await this.createKbnServer(this.settings, this.legacyRawConfig, setupDeps, startDeps, this.legacyPlugins);
    }
  }

  async stop() {
    this.log.debug('stopping legacy service');

    if (this.configSubscription !== undefined) {
      this.configSubscription.unsubscribe();
      this.configSubscription = undefined;
    }

    if (this.kbnServer !== undefined) {
      await this.kbnServer.close();
      this.kbnServer = undefined;
    }
  }

  async createClusterManager(config) {
    const basePathProxy$ = this.coreContext.env.cliArgs.basePath ? (0, _rxjs.combineLatest)(this.devConfig$, this.httpConfig$).pipe((0, _operators.first)(), (0, _operators.map)(([devConfig, httpConfig]) => new _http.BasePathProxyServer(this.coreContext.logger.get('server'), httpConfig, devConfig))) : _rxjs.EMPTY;

    require('../../../cli/cluster/cluster_manager').create(this.coreContext.env.cliArgs, config, (await basePathProxy$.toPromise()));
  }

  async createKbnServer(settings, config, setupDeps, startDeps, legacyPlugins) {
    const coreSetup = {
      context: setupDeps.core.context,
      elasticsearch: {
        adminClient$: setupDeps.core.elasticsearch.adminClient$,
        dataClient$: setupDeps.core.elasticsearch.dataClient$,
        createClient: setupDeps.core.elasticsearch.createClient
      },
      http: {
        createCookieSessionStorageFactory: setupDeps.core.http.createCookieSessionStorageFactory,
        registerRouteHandlerContext: setupDeps.core.http.registerRouteHandlerContext.bind(null, this.legacyId),
        createRouter: () => setupDeps.core.http.createRouter('', this.legacyId),
        registerOnPreAuth: setupDeps.core.http.registerOnPreAuth,
        registerAuth: setupDeps.core.http.registerAuth,
        registerOnPostAuth: setupDeps.core.http.registerOnPostAuth,
        basePath: setupDeps.core.http.basePath,
        isTlsEnabled: setupDeps.core.http.isTlsEnabled
      }
    };
    const coreStart = {}; // eslint-disable-next-line @typescript-eslint/no-var-requires

    const KbnServer = require('../../../legacy/server/kbn_server');

    const kbnServer = new KbnServer(settings, config, {
      env: {
        mode: this.coreContext.env.mode,
        packageInfo: this.coreContext.env.packageInfo
      },
      handledConfigPaths: await this.coreContext.configService.getUsedPaths(),
      setupDeps: {
        core: coreSetup,
        plugins: setupDeps.plugins
      },
      startDeps: {
        core: coreStart,
        plugins: startDeps.plugins
      },
      __internals: {
        hapiServer: setupDeps.core.http.server,
        kibanaMigrator: startDeps.core.savedObjects.migrator,
        uiPlugins: setupDeps.core.plugins.uiPlugins,
        elasticsearch: setupDeps.core.elasticsearch,
        uiSettings: setupDeps.core.uiSettings,
        savedObjectsClientProvider: startDeps.core.savedObjects.clientProvider
      },
      logger: this.coreContext.logger
    }, legacyPlugins); // The kbnWorkerType check is necessary to prevent the repl
    // from being started multiple times in different processes.
    // We only want one REPL.

    if (this.coreContext.env.cliArgs.repl && process.env.kbnWorkerType === 'server') {
      require('../../../cli/repl').startRepl(kbnServer);
    }

    const httpConfig = await this.httpConfig$.pipe((0, _operators.first)()).toPromise();

    if (httpConfig.autoListen) {
      try {
        await kbnServer.listen();
      } catch (err) {
        await kbnServer.close();
        throw err;
      }
    } else {
      await kbnServer.ready();
    }

    return kbnServer;
  }

}

exports.LegacyService = LegacyService;