"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const path_1 = require("path");
const ts = require("typescript");
const ast_helpers_1 = require("./ast_helpers");
const insert_import_1 = require("./insert_import");
const interfaces_1 = require("./interfaces");
const make_transform_1 = require("./make_transform");
function replaceServerBootstrap(shouldTransform, getEntryModule, getTypeChecker) {
    const standardTransform = function (sourceFile) {
        const ops = [];
        const entryModule = getEntryModule();
        if (!shouldTransform(sourceFile.fileName) || !entryModule) {
            return ops;
        }
        // Find all identifiers.
        const entryModuleIdentifiers = ast_helpers_1.collectDeepNodes(sourceFile, ts.SyntaxKind.Identifier)
            .filter(identifier => identifier.text === entryModule.className);
        if (entryModuleIdentifiers.length === 0) {
            return [];
        }
        const relativeEntryModulePath = path_1.relative(path_1.dirname(sourceFile.fileName), entryModule.path);
        const normalizedEntryModulePath = `./${relativeEntryModulePath}`.replace(/\\/g, '/');
        const factoryClassName = entryModule.className + 'NgFactory';
        const factoryModulePath = normalizedEntryModulePath + '.ngfactory';
        // Find the bootstrap calls.
        entryModuleIdentifiers.forEach(entryModuleIdentifier => {
            if (!entryModuleIdentifier.parent) {
                return;
            }
            if (entryModuleIdentifier.parent.kind !== ts.SyntaxKind.CallExpression &&
                entryModuleIdentifier.parent.kind !== ts.SyntaxKind.PropertyAssignment) {
                return;
            }
            if (entryModuleIdentifier.parent.kind === ts.SyntaxKind.CallExpression) {
                // Figure out if it's a `platformDynamicServer().bootstrapModule(AppModule)` call.
                const callExpr = entryModuleIdentifier.parent;
                if (callExpr.expression.kind === ts.SyntaxKind.PropertyAccessExpression) {
                    const propAccessExpr = callExpr.expression;
                    if (!(propAccessExpr.name.text === 'bootstrapModule'
                        && propAccessExpr.expression.kind === ts.SyntaxKind.CallExpression)) {
                        return;
                    }
                    const bootstrapModuleIdentifier = propAccessExpr.name;
                    const innerCallExpr = propAccessExpr.expression;
                    if (!(innerCallExpr.expression.kind === ts.SyntaxKind.Identifier
                        && innerCallExpr.expression.text === 'platformDynamicServer')) {
                        return;
                    }
                    const platformDynamicServerIdentifier = innerCallExpr.expression;
                    const idPlatformServer = ts.createUniqueName('__NgCli_bootstrap_');
                    const idNgFactory = ts.createUniqueName('__NgCli_bootstrap_');
                    // Add the transform operations.
                    ops.push(
                    // Replace the entry module import.
                    ...insert_import_1.insertStarImport(sourceFile, idNgFactory, factoryModulePath), new interfaces_1.ReplaceNodeOperation(sourceFile, entryModuleIdentifier, ts.createPropertyAccess(idNgFactory, ts.createIdentifier(factoryClassName))), 
                    // Replace the platformBrowserDynamic import.
                    ...insert_import_1.insertStarImport(sourceFile, idPlatformServer, '@angular/platform-server'), new interfaces_1.ReplaceNodeOperation(sourceFile, platformDynamicServerIdentifier, ts.createPropertyAccess(idPlatformServer, 'platformServer')), new interfaces_1.ReplaceNodeOperation(sourceFile, bootstrapModuleIdentifier, ts.createIdentifier('bootstrapModuleFactory')));
                }
                else if (callExpr.expression.kind === ts.SyntaxKind.Identifier) {
                    // Figure out if it is renderModule
                    const identifierExpr = callExpr.expression;
                    if (identifierExpr.text !== 'renderModule') {
                        return;
                    }
                    const renderModuleIdentifier = identifierExpr;
                    const idPlatformServer = ts.createUniqueName('__NgCli_bootstrap_');
                    const idNgFactory = ts.createUniqueName('__NgCli_bootstrap_');
                    ops.push(
                    // Replace the entry module import.
                    ...insert_import_1.insertStarImport(sourceFile, idNgFactory, factoryModulePath), new interfaces_1.ReplaceNodeOperation(sourceFile, entryModuleIdentifier, ts.createPropertyAccess(idNgFactory, ts.createIdentifier(factoryClassName))), 
                    // Replace the renderModule import.
                    ...insert_import_1.insertStarImport(sourceFile, idPlatformServer, '@angular/platform-server'), new interfaces_1.ReplaceNodeOperation(sourceFile, renderModuleIdentifier, ts.createPropertyAccess(idPlatformServer, 'renderModuleFactory')));
                }
            }
            else if (entryModuleIdentifier.parent.kind === ts.SyntaxKind.PropertyAssignment) {
                // This is for things that accept a module as a property in a config object
                // .ie the express engine
                const idNgFactory = ts.createUniqueName('__NgCli_bootstrap_');
                ops.push(...insert_import_1.insertStarImport(sourceFile, idNgFactory, factoryModulePath), new interfaces_1.ReplaceNodeOperation(sourceFile, entryModuleIdentifier, ts.createPropertyAccess(idNgFactory, ts.createIdentifier(factoryClassName))));
            }
        });
        return ops;
    };
    return make_transform_1.makeTransform(standardTransform, getTypeChecker);
}
exports.replaceServerBootstrap = replaceServerBootstrap;
//# sourceMappingURL=data:application/json;base64,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