"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const exception_1 = require("../exception/exception");
const action_1 = require("./action");
const entry_1 = require("./entry");
const interface_1 = require("./interface");
const recorder_1 = require("./recorder");
class VirtualDirEntry {
    constructor(_tree, _path = core_1.normalize('/')) {
        this._tree = _tree;
        this._path = _path;
        this._subdirs = new Map();
    }
    _createDir(name) {
        return new VirtualDirEntry(this._tree, core_1.join(this._path, name));
    }
    get parent() {
        return this._path == '/' ? null : this._tree.getDir(core_1.dirname(this._path));
    }
    get path() { return this._path; }
    get subdirs() {
        const directChildPartsCount = core_1.split(core_1.normalize(this._path)).length + 1;
        const directories = this._tree.files
            // make sure entries belong to proper subbranch
            .filter(path => path.startsWith(this._path))
            // get all existing directories & prune to direct children
            .map(path => core_1.split(core_1.normalize(path)).slice(0, -1).slice(0, directChildPartsCount))
            // exclude current directory
            .filter(parts => parts.length === directChildPartsCount)
            // get directory name only
            .map(parts => parts[parts.length - 1]);
        // make sure to have a unique set (directories contain multiple files so appear multiple times)
        return Array.from(new Set(directories));
    }
    get subfiles() {
        return this._tree.files
            .filter(path => core_1.dirname(path) === this._path)
            .map(path => core_1.basename(path));
    }
    dir(name) {
        let maybe = this._subdirs.get(name);
        if (!maybe) {
            this._subdirs.set(name, maybe = this._createDir(name));
        }
        return maybe;
    }
    file(name) {
        return this._tree.get(core_1.join(this._path, name));
    }
    visit(visitor) {
        try {
            this.getSubfilesRecursively().forEach(file => visitor(file.path, file));
        }
        catch (e) {
            if (e !== interface_1.FileVisitorCancelToken) {
                throw e;
            }
        }
    }
    getSubfilesRecursively() {
        function _recurse(entry) {
            return entry.subdirs.reduce((files, subdir) => [
                ...files,
                ..._recurse(entry.dir(subdir)),
            ], entry.subfiles.map(subfile => entry.file(subfile)));
        }
        return _recurse(this);
    }
}
exports.VirtualDirEntry = VirtualDirEntry;
/**
 * The root class of most trees.
 */
class VirtualTree {
    constructor() {
        this._actions = new action_1.ActionList();
        this._cacheMap = new Map();
        this._root = new VirtualDirEntry(this);
        this._tree = new Map();
    }
    static isVirtualTree(tree) {
        if (tree instanceof VirtualTree) {
            return true;
        }
        if (typeof tree === 'object' && typeof tree._copyTo === 'function') {
            return true;
        }
        return false;
    }
    /**
     * Normalize the path. Made available to subclasses to overload.
     * @param path The path to normalize.
     * @returns {string} A path that is resolved and normalized.
     */
    _normalizePath(path) {
        return core_1.normalize('/' + path);
    }
    get tree() { return this._tree; }
    get staging() { return this._cacheMap; }
    [interface_1.TreeSymbol]() {
        return this;
    }
    /**
     * A list of file names contained by this Tree.
     * @returns {[string]} File paths.
     */
    get files() {
        return [...new Set([...this.tree.keys(), ...this._cacheMap.keys()]).values()];
    }
    get root() { return this._root; }
    get(path) {
        const normalizedPath = this._normalizePath(path);
        return this._cacheMap.get(normalizedPath) || this.tree.get(normalizedPath) || null;
    }
    has(path) {
        return this.get(path) != null;
    }
    set(entry) {
        return this._cacheMap.set(entry.path, entry);
    }
    exists(path) {
        return this.has(path);
    }
    read(path) {
        const entry = this.get(path);
        return entry ? entry.content : null;
    }
    getDir(path) {
        let dir = this.root;
        core_1.split(this._normalizePath(path)).slice(1).forEach(fragment => {
            dir = dir.dir(fragment);
        });
        return dir;
    }
    visit(visitor) {
        try {
            this.files.forEach(path => visitor(path, this.get(path)));
        }
        catch (e) {
            if (e !== interface_1.FileVisitorCancelToken) {
                throw e;
            }
        }
    }
    beginUpdate(path) {
        const entry = this.get(path);
        if (!entry) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        return recorder_1.UpdateRecorderBase.createFromFileEntry(entry);
    }
    commitUpdate(record) {
        if (record instanceof recorder_1.UpdateRecorderBase) {
            const path = record.path;
            const entry = this.get(path);
            if (!entry) {
                throw new exception_1.ContentHasMutatedException(path);
            }
            else {
                const newContent = record.apply(entry.content);
                this.overwrite(path, newContent);
            }
        }
        else {
            throw new exception_1.InvalidUpdateRecordException();
        }
    }
    overwrite(path, content) {
        const normalizedTo = this._normalizePath(path);
        if (typeof content == 'string') {
            content = Buffer.from(content, 'utf-8');
        }
        const maybeEntry = this.get(normalizedTo);
        if (maybeEntry && maybeEntry.content.equals(content)) {
            return;
        }
        this._overwrite(normalizedTo, content);
    }
    create(path, content) {
        const normalizedTo = this._normalizePath(path);
        if (typeof content == 'string') {
            content = Buffer.from(content);
        }
        this._create(normalizedTo, content);
    }
    rename(path, to) {
        const normalizedPath = this._normalizePath(path);
        const normalizedTo = this._normalizePath(to);
        if (normalizedPath === normalizedTo) {
            // Nothing to do.
            return;
        }
        this._rename(normalizedPath, normalizedTo);
    }
    delete(path) {
        this._delete(this._normalizePath(path));
    }
    _overwrite(path, content, action) {
        if (!this.has(path)) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        // Update the action buffer.
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.overwrite(path, content);
        }
        this.set(new entry_1.SimpleFileEntry(path, content));
    }
    _create(path, content, action) {
        if (this._cacheMap.has(path)) {
            throw new exception_1.FileAlreadyExistException(path);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.create(path, content);
        }
        this.set(new entry_1.SimpleFileEntry(path, content));
    }
    _rename(path, to, action, force = false) {
        const entry = this.get(path);
        if (!entry) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        if (this._cacheMap.has(to) && !force) {
            throw new exception_1.FileAlreadyExistException(to);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.rename(path, to);
        }
        this.set(new entry_1.SimpleFileEntry(to, entry.content));
        this._cacheMap.delete(path);
    }
    _delete(path, action) {
        if (!this.has(path)) {
            throw new exception_1.FileDoesNotExistException(path);
        }
        if (action) {
            this._actions.push(action);
        }
        else {
            this._actions.delete(path);
        }
        this._cacheMap.delete(path);
    }
    apply(action, strategy) {
        if (this._actions.has(action)) {
            return;
        }
        switch (action.kind) {
            case 'o':
                // Update the action buffer.
                this._overwrite(action.path, action.content, action);
                break;
            case 'c':
                if (this._cacheMap.has(action.path)) {
                    switch (strategy) {
                        case interface_1.MergeStrategy.Error: throw new exception_1.MergeConflictException(action.path);
                        case interface_1.MergeStrategy.Overwrite:
                            this._overwrite(action.path, action.content, action);
                            break;
                    }
                }
                else {
                    this._create(action.path, action.content, action);
                }
                break;
            case 'r':
                const force = (strategy & interface_1.MergeStrategy.AllowOverwriteConflict) != 0;
                this._rename(action.path, action.to, action, force);
                break;
            case 'd':
                this._delete(action.path, action);
                break;
            default: throw new action_1.UnknownActionException(action);
        }
    }
    // Returns an ordered list of Action to get this host.
    get actions() {
        return [...this._actions];
    }
    /**
     * Allow subclasses to copy to a tree their own properties.
     * @return {Tree}
     * @private
     */
    _copyTo(tree) {
        tree._tree = new Map(this.tree);
        this._actions.forEach(action => tree._actions.push(action));
        [...this._cacheMap.entries()].forEach(([path, entry]) => {
            tree._cacheMap.set(path, entry);
        });
    }
    branch() {
        const newTree = new VirtualTree();
        this._copyTo(newTree);
        return newTree;
    }
    // Creates a new host from 2 hosts.
    merge(other, strategy = interface_1.MergeStrategy.Default) {
        other.actions.forEach(action => this.apply(action, strategy));
    }
    optimize() {
        // This destroys the history. Hope you know what you're doing.
        this._actions.optimize();
    }
    static branch(tree) {
        return tree.branch();
    }
    static merge(tree, other, strategy = interface_1.MergeStrategy.Default) {
        const newTree = tree.branch();
        newTree.merge(other, strategy);
        return newTree;
    }
    static optimize(tree) {
        const newTree = tree.branch();
        newTree.optimize();
        return newTree;
    }
}
exports.VirtualTree = VirtualTree;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmlydHVhbC5qcyIsInNvdXJjZVJvb3QiOiIuLyIsInNvdXJjZXMiOlsicGFja2FnZXMvYW5ndWxhcl9kZXZraXQvc2NoZW1hdGljcy9zcmMvdHJlZS92aXJ0dWFsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7OztHQU1HO0FBQ0gsK0NBUThCO0FBQzlCLHNEQU1nQztBQUNoQyxxQ0FBc0U7QUFDdEUsbUNBQTBDO0FBQzFDLDJDQVNxQjtBQUNyQix5Q0FBZ0Q7QUFHaEQsTUFBYSxlQUFlO0lBRzFCLFlBQXNCLEtBQWtCLEVBQVksUUFBYyxnQkFBUyxDQUFDLEdBQUcsQ0FBQztRQUExRCxVQUFLLEdBQUwsS0FBSyxDQUFhO1FBQVksVUFBSyxHQUFMLEtBQUssQ0FBdUI7UUFGdEUsYUFBUSxHQUFHLElBQUksR0FBRyxFQUEwQixDQUFDO0lBRTRCLENBQUM7SUFFMUUsVUFBVSxDQUFDLElBQWtCO1FBQ3JDLE9BQU8sSUFBSSxlQUFlLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxXQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRCxJQUFJLE1BQU07UUFDUixPQUFPLElBQUksQ0FBQyxLQUFLLElBQUksR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLGNBQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUMzRSxDQUFDO0lBQ0QsSUFBSSxJQUFJLEtBQUssT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUVqQyxJQUFJLE9BQU87UUFDVCxNQUFNLHFCQUFxQixHQUFHLFlBQUssQ0FBQyxnQkFBUyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7UUFFdEUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLO1lBQ3BDLCtDQUErQzthQUM5QyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUM1QywwREFBMEQ7YUFDekQsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBSyxDQUFDLGdCQUFTLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1lBQ2pGLDRCQUE0QjthQUMzQixNQUFNLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsTUFBTSxLQUFLLHFCQUFxQixDQUFDO1lBQ3hELDBCQUEwQjthQUN6QixHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXZDLCtGQUErRjtRQUMvRixPQUFPLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBQ0QsSUFBSSxRQUFRO1FBQ1YsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUs7YUFDcEIsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLElBQUksQ0FBQyxLQUFLLENBQUM7YUFDNUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVELEdBQUcsQ0FBQyxJQUFrQjtRQUNwQixJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ1YsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLEtBQUssR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7U0FDeEQ7UUFFRCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFDRCxJQUFJLENBQUMsSUFBa0I7UUFDckIsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxXQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRCxLQUFLLENBQUMsT0FBb0I7UUFDeEIsSUFBSTtZQUNGLElBQUksQ0FBQyxzQkFBc0IsRUFBRSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7U0FDekU7UUFBQyxPQUFPLENBQUMsRUFBRTtZQUNWLElBQUksQ0FBQyxLQUFLLGtDQUFzQixFQUFFO2dCQUNoQyxNQUFNLENBQUMsQ0FBQzthQUNUO1NBQ0Y7SUFDSCxDQUFDO0lBRU8sc0JBQXNCO1FBQzVCLFNBQVMsUUFBUSxDQUFDLEtBQWU7WUFDL0IsT0FBTyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBRSxDQUFDO2dCQUM3QyxHQUFHLEtBQUs7Z0JBQ1IsR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQzthQUMvQixFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQWMsQ0FBQyxDQUFDLENBQUM7UUFDdEUsQ0FBQztRQUVELE9BQU8sUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3hCLENBQUM7Q0FDRjtBQXBFRCwwQ0FvRUM7QUFFRDs7R0FFRztBQUNILE1BQWEsV0FBVztJQUF4QjtRQUNZLGFBQVEsR0FBRyxJQUFJLG1CQUFVLEVBQUUsQ0FBQztRQUM1QixjQUFTLEdBQUcsSUFBSSxHQUFHLEVBQW1CLENBQUM7UUFDdkMsVUFBSyxHQUFHLElBQUksZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2xDLFVBQUssR0FBRyxJQUFJLEdBQUcsRUFBbUIsQ0FBQztJQXNSL0MsQ0FBQztJQXBSQyxNQUFNLENBQUMsYUFBYSxDQUFDLElBQVU7UUFDN0IsSUFBSSxJQUFJLFlBQVksV0FBVyxFQUFFO1lBQy9CLE9BQU8sSUFBSSxDQUFDO1NBQ2I7UUFFRCxJQUFJLE9BQU8sSUFBSSxLQUFLLFFBQVEsSUFBSSxPQUFRLElBQW9CLENBQUMsT0FBTyxLQUFLLFVBQVUsRUFBRTtZQUNuRixPQUFPLElBQUksQ0FBQztTQUNiO1FBRUQsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNPLGNBQWMsQ0FBQyxJQUFZO1FBQ25DLE9BQU8sZ0JBQVMsQ0FBQyxHQUFHLEdBQUcsSUFBSSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUNELElBQWMsSUFBSSxLQUFtQyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBQ3pFLElBQUksT0FBTyxLQUFtQyxPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO0lBRXRFLENBQUMsc0JBQVUsQ0FBQztRQUNWLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7T0FHRztJQUNILElBQUksS0FBSztRQUNQLE9BQU8sQ0FBQyxHQUFHLElBQUksR0FBRyxDQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztJQUN0RixDQUFDO0lBRUQsSUFBSSxJQUFJLEtBQWUsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUUzQyxHQUFHLENBQUMsSUFBWTtRQUNkLE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFakQsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxJQUFJLENBQUM7SUFDckYsQ0FBQztJQUNELEdBQUcsQ0FBQyxJQUFZO1FBQ2QsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQztJQUNoQyxDQUFDO0lBQ0QsR0FBRyxDQUFDLEtBQWdCO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQsTUFBTSxDQUFDLElBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3hCLENBQUM7SUFFRCxJQUFJLENBQUMsSUFBWTtRQUNmLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0IsT0FBTyxLQUFLLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztJQUN0QyxDQUFDO0lBRUQsTUFBTSxDQUFDLElBQVk7UUFDakIsSUFBSSxHQUFHLEdBQWEsSUFBSSxDQUFDLElBQUksQ0FBQztRQUM5QixZQUFLLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDM0QsR0FBRyxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDMUIsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFFRCxLQUFLLENBQUMsT0FBb0I7UUFDeEIsSUFBSTtZQUNGLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUMzRDtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsSUFBSSxDQUFDLEtBQUssa0NBQXNCLEVBQUU7Z0JBQ2hDLE1BQU0sQ0FBQyxDQUFDO2FBQ1Q7U0FDRjtJQUNILENBQUM7SUFFRCxXQUFXLENBQUMsSUFBWTtRQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdCLElBQUksQ0FBQyxLQUFLLEVBQUU7WUFDVixNQUFNLElBQUkscUNBQXlCLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDM0M7UUFFRCxPQUFPLDZCQUFrQixDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRCxZQUFZLENBQUMsTUFBc0I7UUFDakMsSUFBSSxNQUFNLFlBQVksNkJBQWtCLEVBQUU7WUFDeEMsTUFBTSxJQUFJLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQztZQUN6QixNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzdCLElBQUksQ0FBQyxLQUFLLEVBQUU7Z0JBQ1YsTUFBTSxJQUFJLHNDQUEwQixDQUFDLElBQUksQ0FBQyxDQUFDO2FBQzVDO2lCQUFNO2dCQUNMLE1BQU0sVUFBVSxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO2dCQUMvQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQzthQUNsQztTQUNGO2FBQU07WUFDTCxNQUFNLElBQUksd0NBQTRCLEVBQUUsQ0FBQztTQUMxQztJQUNILENBQUM7SUFFRCxTQUFTLENBQUMsSUFBWSxFQUFFLE9BQXdCO1FBQzlDLE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDL0MsSUFBSSxPQUFPLE9BQU8sSUFBSSxRQUFRLEVBQUU7WUFDOUIsT0FBTyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1NBQ3pDO1FBQ0QsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUMxQyxJQUFJLFVBQVUsSUFBSSxVQUFVLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUNwRCxPQUFPO1NBQ1I7UUFDRCxJQUFJLENBQUMsVUFBVSxDQUFDLFlBQVksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBQ0QsTUFBTSxDQUFDLElBQVksRUFBRSxPQUF3QjtRQUMzQyxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQy9DLElBQUksT0FBTyxPQUFPLElBQUksUUFBUSxFQUFFO1lBQzlCLE9BQU8sR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQ2hDO1FBQ0QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUNELE1BQU0sQ0FBQyxJQUFZLEVBQUUsRUFBVTtRQUM3QixNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDN0MsSUFBSSxjQUFjLEtBQUssWUFBWSxFQUFFO1lBQ25DLGlCQUFpQjtZQUNqQixPQUFPO1NBQ1I7UUFDRCxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxZQUFZLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQsTUFBTSxDQUFDLElBQVk7UUFDakIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVTLFVBQVUsQ0FBQyxJQUFVLEVBQUUsT0FBZSxFQUFFLE1BQWU7UUFDL0QsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDbkIsTUFBTSxJQUFJLHFDQUF5QixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQzNDO1FBQ0QsNEJBQTRCO1FBQzVCLElBQUksTUFBTSxFQUFFO1lBQ1YsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDNUI7YUFBTTtZQUNMLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztTQUN4QztRQUNELElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSx1QkFBZSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFDUyxPQUFPLENBQUMsSUFBVSxFQUFFLE9BQWUsRUFBRSxNQUFlO1FBQzVELElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDNUIsTUFBTSxJQUFJLHFDQUF5QixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQzNDO1FBRUQsSUFBSSxNQUFNLEVBQUU7WUFDVixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUM1QjthQUFNO1lBQ0wsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1NBQ3JDO1FBQ0QsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLHVCQUFlLENBQUMsSUFBSSxFQUFFLE9BQWlCLENBQUMsQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFDUyxPQUFPLENBQUMsSUFBVSxFQUFFLEVBQVEsRUFBRSxNQUFlLEVBQUUsS0FBSyxHQUFHLEtBQUs7UUFDcEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QixJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ1YsTUFBTSxJQUFJLHFDQUF5QixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQzNDO1FBQ0QsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNwQyxNQUFNLElBQUkscUNBQXlCLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDekM7UUFFRCxJQUFJLE1BQU0sRUFBRTtZQUNWLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQzVCO2FBQU07WUFDTCxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLENBQUM7U0FDaEM7UUFFRCxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksdUJBQWUsQ0FBQyxFQUFFLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUNTLE9BQU8sQ0FBQyxJQUFVLEVBQUUsTUFBZTtRQUMzQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUNuQixNQUFNLElBQUkscUNBQXlCLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDM0M7UUFFRCxJQUFJLE1BQU0sRUFBRTtZQUNWLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQzVCO2FBQU07WUFDTCxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUM1QjtRQUNELElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFHRCxLQUFLLENBQUMsTUFBYyxFQUFFLFFBQXVCO1FBQzNDLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDN0IsT0FBTztTQUNSO1FBQ0QsUUFBUSxNQUFNLENBQUMsSUFBSSxFQUFFO1lBQ25CLEtBQUssR0FBRztnQkFDTiw0QkFBNEI7Z0JBQzVCLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUNyRCxNQUFNO1lBRVIsS0FBSyxHQUFHO2dCQUNOLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUNuQyxRQUFRLFFBQVEsRUFBRTt3QkFDaEIsS0FBSyx5QkFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sSUFBSSxrQ0FBc0IsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7d0JBQ3hFLEtBQUsseUJBQWEsQ0FBQyxTQUFTOzRCQUMxQixJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLE9BQU8sRUFBRSxNQUFNLENBQUMsQ0FBQzs0QkFDckQsTUFBTTtxQkFDVDtpQkFDRjtxQkFBTTtvQkFDTCxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLE9BQU8sRUFBRSxNQUFNLENBQUMsQ0FBQztpQkFDbkQ7Z0JBQ0QsTUFBTTtZQUVSLEtBQUssR0FBRztnQkFDTixNQUFNLEtBQUssR0FBRyxDQUFDLFFBQVEsR0FBRyx5QkFBYSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxDQUFDO2dCQUNyRSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLEVBQUUsRUFBRSxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7Z0JBQ3BELE1BQU07WUFFUixLQUFLLEdBQUc7Z0JBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUFDLE1BQU07WUFFbkQsT0FBTyxDQUFDLENBQUMsTUFBTSxJQUFJLCtCQUFzQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQ25EO0lBQ0gsQ0FBQztJQUVELHNEQUFzRDtJQUN0RCxJQUFJLE9BQU87UUFDVCxPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUVEOzs7O09BSUc7SUFDTyxPQUFPLENBQXdCLElBQU87UUFDOUMsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDaEMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQzVELENBQUMsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRTtZQUN0RCxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDbEMsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsTUFBTTtRQUNKLE1BQU0sT0FBTyxHQUFHLElBQUksV0FBVyxFQUFFLENBQUM7UUFDbEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUV0QixPQUFPLE9BQU8sQ0FBQztJQUNqQixDQUFDO0lBRUQsbUNBQW1DO0lBQ25DLEtBQUssQ0FBQyxLQUFXLEVBQUUsV0FBMEIseUJBQWEsQ0FBQyxPQUFPO1FBQ2hFLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRUQsUUFBUTtRQUNOLDhEQUE4RDtRQUM5RCxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDO0lBQzNCLENBQUM7SUFFRCxNQUFNLENBQUMsTUFBTSxDQUFDLElBQVU7UUFDdEIsT0FBTyxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUM7SUFDdkIsQ0FBQztJQUVELE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBVSxFQUFFLEtBQVcsRUFBRSxXQUEwQix5QkFBYSxDQUFDLE9BQU87UUFDbkYsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQzlCLE9BQU8sQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRS9CLE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFRCxNQUFNLENBQUMsUUFBUSxDQUFDLElBQVU7UUFDeEIsTUFBTSxPQUFPLEdBQUksSUFBb0IsQ0FBQyxNQUFNLEVBQWlCLENBQUM7UUFDOUQsT0FBTyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBRW5CLE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7Q0FDRjtBQTFSRCxrQ0EwUkMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIEluYy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5pbXBvcnQge1xuICBQYXRoLFxuICBQYXRoRnJhZ21lbnQsXG4gIGJhc2VuYW1lLFxuICBkaXJuYW1lLFxuICBqb2luLFxuICBub3JtYWxpemUsXG4gIHNwbGl0LFxufSBmcm9tICdAYW5ndWxhci1kZXZraXQvY29yZSc7XG5pbXBvcnQge1xuICBDb250ZW50SGFzTXV0YXRlZEV4Y2VwdGlvbixcbiAgRmlsZUFscmVhZHlFeGlzdEV4Y2VwdGlvbixcbiAgRmlsZURvZXNOb3RFeGlzdEV4Y2VwdGlvbixcbiAgSW52YWxpZFVwZGF0ZVJlY29yZEV4Y2VwdGlvbixcbiAgTWVyZ2VDb25mbGljdEV4Y2VwdGlvbixcbn0gZnJvbSAnLi4vZXhjZXB0aW9uL2V4Y2VwdGlvbic7XG5pbXBvcnQgeyBBY3Rpb24sIEFjdGlvbkxpc3QsIFVua25vd25BY3Rpb25FeGNlcHRpb24gfSBmcm9tICcuL2FjdGlvbic7XG5pbXBvcnQgeyBTaW1wbGVGaWxlRW50cnkgfSBmcm9tICcuL2VudHJ5JztcbmltcG9ydCB7XG4gIERpckVudHJ5LFxuICBGaWxlRW50cnksXG4gIEZpbGVWaXNpdG9yLFxuICBGaWxlVmlzaXRvckNhbmNlbFRva2VuLFxuICBNZXJnZVN0cmF0ZWd5LFxuICBUcmVlLFxuICBUcmVlU3ltYm9sLFxuICBVcGRhdGVSZWNvcmRlcixcbn0gZnJvbSAnLi9pbnRlcmZhY2UnO1xuaW1wb3J0IHsgVXBkYXRlUmVjb3JkZXJCYXNlIH0gZnJvbSAnLi9yZWNvcmRlcic7XG5cblxuZXhwb3J0IGNsYXNzIFZpcnR1YWxEaXJFbnRyeSBpbXBsZW1lbnRzIERpckVudHJ5IHtcbiAgcHJvdGVjdGVkIF9zdWJkaXJzID0gbmV3IE1hcDxQYXRoRnJhZ21lbnQsIERpckVudHJ5PigpO1xuXG4gIGNvbnN0cnVjdG9yKHByb3RlY3RlZCBfdHJlZTogVmlydHVhbFRyZWUsIHByb3RlY3RlZCBfcGF0aDogUGF0aCA9IG5vcm1hbGl6ZSgnLycpKSB7fVxuXG4gIHByb3RlY3RlZCBfY3JlYXRlRGlyKG5hbWU6IFBhdGhGcmFnbWVudCk6IERpckVudHJ5IHtcbiAgICByZXR1cm4gbmV3IFZpcnR1YWxEaXJFbnRyeSh0aGlzLl90cmVlLCBqb2luKHRoaXMuX3BhdGgsIG5hbWUpKTtcbiAgfVxuXG4gIGdldCBwYXJlbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMuX3BhdGggPT0gJy8nID8gbnVsbCA6IHRoaXMuX3RyZWUuZ2V0RGlyKGRpcm5hbWUodGhpcy5fcGF0aCkpO1xuICB9XG4gIGdldCBwYXRoKCkgeyByZXR1cm4gdGhpcy5fcGF0aDsgfVxuXG4gIGdldCBzdWJkaXJzKCkge1xuICAgIGNvbnN0IGRpcmVjdENoaWxkUGFydHNDb3VudCA9IHNwbGl0KG5vcm1hbGl6ZSh0aGlzLl9wYXRoKSkubGVuZ3RoICsgMTtcblxuICAgIGNvbnN0IGRpcmVjdG9yaWVzID0gdGhpcy5fdHJlZS5maWxlc1xuICAgIC8vIG1ha2Ugc3VyZSBlbnRyaWVzIGJlbG9uZyB0byBwcm9wZXIgc3ViYnJhbmNoXG4gICAgLmZpbHRlcihwYXRoID0+IHBhdGguc3RhcnRzV2l0aCh0aGlzLl9wYXRoKSlcbiAgICAvLyBnZXQgYWxsIGV4aXN0aW5nIGRpcmVjdG9yaWVzICYgcHJ1bmUgdG8gZGlyZWN0IGNoaWxkcmVuXG4gICAgLm1hcChwYXRoID0+IHNwbGl0KG5vcm1hbGl6ZShwYXRoKSkuc2xpY2UoMCwgLTEpLnNsaWNlKDAsIGRpcmVjdENoaWxkUGFydHNDb3VudCkpXG4gICAgLy8gZXhjbHVkZSBjdXJyZW50IGRpcmVjdG9yeVxuICAgIC5maWx0ZXIocGFydHMgPT4gcGFydHMubGVuZ3RoID09PSBkaXJlY3RDaGlsZFBhcnRzQ291bnQpXG4gICAgLy8gZ2V0IGRpcmVjdG9yeSBuYW1lIG9ubHlcbiAgICAubWFwKHBhcnRzID0+IHBhcnRzW3BhcnRzLmxlbmd0aCAtIDFdKTtcblxuICAgIC8vIG1ha2Ugc3VyZSB0byBoYXZlIGEgdW5pcXVlIHNldCAoZGlyZWN0b3JpZXMgY29udGFpbiBtdWx0aXBsZSBmaWxlcyBzbyBhcHBlYXIgbXVsdGlwbGUgdGltZXMpXG4gICAgcmV0dXJuIEFycmF5LmZyb20obmV3IFNldChkaXJlY3RvcmllcykpO1xuICB9XG4gIGdldCBzdWJmaWxlcygpIHtcbiAgICByZXR1cm4gdGhpcy5fdHJlZS5maWxlc1xuICAgICAgLmZpbHRlcihwYXRoID0+IGRpcm5hbWUocGF0aCkgPT09IHRoaXMuX3BhdGgpXG4gICAgICAubWFwKHBhdGggPT4gYmFzZW5hbWUocGF0aCkpO1xuICB9XG5cbiAgZGlyKG5hbWU6IFBhdGhGcmFnbWVudCkge1xuICAgIGxldCBtYXliZSA9IHRoaXMuX3N1YmRpcnMuZ2V0KG5hbWUpO1xuICAgIGlmICghbWF5YmUpIHtcbiAgICAgIHRoaXMuX3N1YmRpcnMuc2V0KG5hbWUsIG1heWJlID0gdGhpcy5fY3JlYXRlRGlyKG5hbWUpKTtcbiAgICB9XG5cbiAgICByZXR1cm4gbWF5YmU7XG4gIH1cbiAgZmlsZShuYW1lOiBQYXRoRnJhZ21lbnQpIHtcbiAgICByZXR1cm4gdGhpcy5fdHJlZS5nZXQoam9pbih0aGlzLl9wYXRoLCBuYW1lKSk7XG4gIH1cblxuICB2aXNpdCh2aXNpdG9yOiBGaWxlVmlzaXRvcik6IHZvaWQge1xuICAgIHRyeSB7XG4gICAgICB0aGlzLmdldFN1YmZpbGVzUmVjdXJzaXZlbHkoKS5mb3JFYWNoKGZpbGUgPT4gdmlzaXRvcihmaWxlLnBhdGgsIGZpbGUpKTtcbiAgICB9IGNhdGNoIChlKSB7XG4gICAgICBpZiAoZSAhPT0gRmlsZVZpc2l0b3JDYW5jZWxUb2tlbikge1xuICAgICAgICB0aHJvdyBlO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgZ2V0U3ViZmlsZXNSZWN1cnNpdmVseSgpIHtcbiAgICBmdW5jdGlvbiBfcmVjdXJzZShlbnRyeTogRGlyRW50cnkpOiBGaWxlRW50cnlbXSB7XG4gICAgICByZXR1cm4gZW50cnkuc3ViZGlycy5yZWR1Y2UoKGZpbGVzLCBzdWJkaXIpID0+IFtcbiAgICAgICAgLi4uZmlsZXMsXG4gICAgICAgIC4uLl9yZWN1cnNlKGVudHJ5LmRpcihzdWJkaXIpKSxcbiAgICAgIF0sIGVudHJ5LnN1YmZpbGVzLm1hcChzdWJmaWxlID0+IGVudHJ5LmZpbGUoc3ViZmlsZSkgYXMgRmlsZUVudHJ5KSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIF9yZWN1cnNlKHRoaXMpO1xuICB9XG59XG5cbi8qKlxuICogVGhlIHJvb3QgY2xhc3Mgb2YgbW9zdCB0cmVlcy5cbiAqL1xuZXhwb3J0IGNsYXNzIFZpcnR1YWxUcmVlIGltcGxlbWVudHMgVHJlZSB7XG4gIHByb3RlY3RlZCBfYWN0aW9ucyA9IG5ldyBBY3Rpb25MaXN0KCk7XG4gIHByb3RlY3RlZCBfY2FjaGVNYXAgPSBuZXcgTWFwPFBhdGgsIEZpbGVFbnRyeT4oKTtcbiAgcHJvdGVjdGVkIF9yb290ID0gbmV3IFZpcnR1YWxEaXJFbnRyeSh0aGlzKTtcbiAgcHJvdGVjdGVkIF90cmVlID0gbmV3IE1hcDxQYXRoLCBGaWxlRW50cnk+KCk7XG5cbiAgc3RhdGljIGlzVmlydHVhbFRyZWUodHJlZTogVHJlZSk6IHRyZWUgaXMgVmlydHVhbFRyZWUge1xuICAgIGlmICh0cmVlIGluc3RhbmNlb2YgVmlydHVhbFRyZWUpIHtcbiAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cblxuICAgIGlmICh0eXBlb2YgdHJlZSA9PT0gJ29iamVjdCcgJiYgdHlwZW9mICh0cmVlIGFzIFZpcnR1YWxUcmVlKS5fY29weVRvID09PSAnZnVuY3Rpb24nKSB7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9XG5cbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cblxuICAvKipcbiAgICogTm9ybWFsaXplIHRoZSBwYXRoLiBNYWRlIGF2YWlsYWJsZSB0byBzdWJjbGFzc2VzIHRvIG92ZXJsb2FkLlxuICAgKiBAcGFyYW0gcGF0aCBUaGUgcGF0aCB0byBub3JtYWxpemUuXG4gICAqIEByZXR1cm5zIHtzdHJpbmd9IEEgcGF0aCB0aGF0IGlzIHJlc29sdmVkIGFuZCBub3JtYWxpemVkLlxuICAgKi9cbiAgcHJvdGVjdGVkIF9ub3JtYWxpemVQYXRoKHBhdGg6IHN0cmluZyk6IFBhdGgge1xuICAgIHJldHVybiBub3JtYWxpemUoJy8nICsgcGF0aCk7XG4gIH1cbiAgcHJvdGVjdGVkIGdldCB0cmVlKCk6IFJlYWRvbmx5TWFwPFBhdGgsIEZpbGVFbnRyeT4geyByZXR1cm4gdGhpcy5fdHJlZTsgfVxuICBnZXQgc3RhZ2luZygpOiBSZWFkb25seU1hcDxQYXRoLCBGaWxlRW50cnk+IHsgcmV0dXJuIHRoaXMuX2NhY2hlTWFwOyB9XG5cbiAgW1RyZWVTeW1ib2xdKCkge1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBmaWxlIG5hbWVzIGNvbnRhaW5lZCBieSB0aGlzIFRyZWUuXG4gICAqIEByZXR1cm5zIHtbc3RyaW5nXX0gRmlsZSBwYXRocy5cbiAgICovXG4gIGdldCBmaWxlcygpOiBQYXRoW10ge1xuICAgIHJldHVybiBbLi4ubmV3IFNldDxQYXRoPihbLi4udGhpcy50cmVlLmtleXMoKSwgLi4udGhpcy5fY2FjaGVNYXAua2V5cygpXSkudmFsdWVzKCldO1xuICB9XG5cbiAgZ2V0IHJvb3QoKTogRGlyRW50cnkgeyByZXR1cm4gdGhpcy5fcm9vdDsgfVxuXG4gIGdldChwYXRoOiBzdHJpbmcpOiBGaWxlRW50cnkgfCBudWxsIHtcbiAgICBjb25zdCBub3JtYWxpemVkUGF0aCA9IHRoaXMuX25vcm1hbGl6ZVBhdGgocGF0aCk7XG5cbiAgICByZXR1cm4gdGhpcy5fY2FjaGVNYXAuZ2V0KG5vcm1hbGl6ZWRQYXRoKSB8fCB0aGlzLnRyZWUuZ2V0KG5vcm1hbGl6ZWRQYXRoKSB8fCBudWxsO1xuICB9XG4gIGhhcyhwYXRoOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5nZXQocGF0aCkgIT0gbnVsbDtcbiAgfVxuICBzZXQoZW50cnk6IEZpbGVFbnRyeSkge1xuICAgIHJldHVybiB0aGlzLl9jYWNoZU1hcC5zZXQoZW50cnkucGF0aCwgZW50cnkpO1xuICB9XG5cbiAgZXhpc3RzKHBhdGg6IHN0cmluZyk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLmhhcyhwYXRoKTtcbiAgfVxuXG4gIHJlYWQocGF0aDogc3RyaW5nKTogQnVmZmVyIHwgbnVsbCB7XG4gICAgY29uc3QgZW50cnkgPSB0aGlzLmdldChwYXRoKTtcblxuICAgIHJldHVybiBlbnRyeSA/IGVudHJ5LmNvbnRlbnQgOiBudWxsO1xuICB9XG5cbiAgZ2V0RGlyKHBhdGg6IHN0cmluZyk6IERpckVudHJ5IHtcbiAgICBsZXQgZGlyOiBEaXJFbnRyeSA9IHRoaXMucm9vdDtcbiAgICBzcGxpdCh0aGlzLl9ub3JtYWxpemVQYXRoKHBhdGgpKS5zbGljZSgxKS5mb3JFYWNoKGZyYWdtZW50ID0+IHtcbiAgICAgIGRpciA9IGRpci5kaXIoZnJhZ21lbnQpO1xuICAgIH0pO1xuXG4gICAgcmV0dXJuIGRpcjtcbiAgfVxuXG4gIHZpc2l0KHZpc2l0b3I6IEZpbGVWaXNpdG9yKSB7XG4gICAgdHJ5IHtcbiAgICAgIHRoaXMuZmlsZXMuZm9yRWFjaChwYXRoID0+IHZpc2l0b3IocGF0aCwgdGhpcy5nZXQocGF0aCkpKTtcbiAgICB9IGNhdGNoIChlKSB7XG4gICAgICBpZiAoZSAhPT0gRmlsZVZpc2l0b3JDYW5jZWxUb2tlbikge1xuICAgICAgICB0aHJvdyBlO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIGJlZ2luVXBkYXRlKHBhdGg6IHN0cmluZyk6IFVwZGF0ZVJlY29yZGVyIHtcbiAgICBjb25zdCBlbnRyeSA9IHRoaXMuZ2V0KHBhdGgpO1xuICAgIGlmICghZW50cnkpIHtcbiAgICAgIHRocm93IG5ldyBGaWxlRG9lc05vdEV4aXN0RXhjZXB0aW9uKHBhdGgpO1xuICAgIH1cblxuICAgIHJldHVybiBVcGRhdGVSZWNvcmRlckJhc2UuY3JlYXRlRnJvbUZpbGVFbnRyeShlbnRyeSk7XG4gIH1cblxuICBjb21taXRVcGRhdGUocmVjb3JkOiBVcGRhdGVSZWNvcmRlcikge1xuICAgIGlmIChyZWNvcmQgaW5zdGFuY2VvZiBVcGRhdGVSZWNvcmRlckJhc2UpIHtcbiAgICAgIGNvbnN0IHBhdGggPSByZWNvcmQucGF0aDtcbiAgICAgIGNvbnN0IGVudHJ5ID0gdGhpcy5nZXQocGF0aCk7XG4gICAgICBpZiAoIWVudHJ5KSB7XG4gICAgICAgIHRocm93IG5ldyBDb250ZW50SGFzTXV0YXRlZEV4Y2VwdGlvbihwYXRoKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGNvbnN0IG5ld0NvbnRlbnQgPSByZWNvcmQuYXBwbHkoZW50cnkuY29udGVudCk7XG4gICAgICAgIHRoaXMub3ZlcndyaXRlKHBhdGgsIG5ld0NvbnRlbnQpO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgSW52YWxpZFVwZGF0ZVJlY29yZEV4Y2VwdGlvbigpO1xuICAgIH1cbiAgfVxuXG4gIG92ZXJ3cml0ZShwYXRoOiBzdHJpbmcsIGNvbnRlbnQ6IEJ1ZmZlciB8IHN0cmluZykge1xuICAgIGNvbnN0IG5vcm1hbGl6ZWRUbyA9IHRoaXMuX25vcm1hbGl6ZVBhdGgocGF0aCk7XG4gICAgaWYgKHR5cGVvZiBjb250ZW50ID09ICdzdHJpbmcnKSB7XG4gICAgICBjb250ZW50ID0gQnVmZmVyLmZyb20oY29udGVudCwgJ3V0Zi04Jyk7XG4gICAgfVxuICAgIGNvbnN0IG1heWJlRW50cnkgPSB0aGlzLmdldChub3JtYWxpemVkVG8pO1xuICAgIGlmIChtYXliZUVudHJ5ICYmIG1heWJlRW50cnkuY29udGVudC5lcXVhbHMoY29udGVudCkpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgdGhpcy5fb3ZlcndyaXRlKG5vcm1hbGl6ZWRUbywgY29udGVudCk7XG4gIH1cbiAgY3JlYXRlKHBhdGg6IHN0cmluZywgY29udGVudDogQnVmZmVyIHwgc3RyaW5nKTogdm9pZCB7XG4gICAgY29uc3Qgbm9ybWFsaXplZFRvID0gdGhpcy5fbm9ybWFsaXplUGF0aChwYXRoKTtcbiAgICBpZiAodHlwZW9mIGNvbnRlbnQgPT0gJ3N0cmluZycpIHtcbiAgICAgIGNvbnRlbnQgPSBCdWZmZXIuZnJvbShjb250ZW50KTtcbiAgICB9XG4gICAgdGhpcy5fY3JlYXRlKG5vcm1hbGl6ZWRUbywgY29udGVudCk7XG4gIH1cbiAgcmVuYW1lKHBhdGg6IHN0cmluZywgdG86IHN0cmluZyk6IHZvaWQge1xuICAgIGNvbnN0IG5vcm1hbGl6ZWRQYXRoID0gdGhpcy5fbm9ybWFsaXplUGF0aChwYXRoKTtcbiAgICBjb25zdCBub3JtYWxpemVkVG8gPSB0aGlzLl9ub3JtYWxpemVQYXRoKHRvKTtcbiAgICBpZiAobm9ybWFsaXplZFBhdGggPT09IG5vcm1hbGl6ZWRUbykge1xuICAgICAgLy8gTm90aGluZyB0byBkby5cbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgdGhpcy5fcmVuYW1lKG5vcm1hbGl6ZWRQYXRoLCBub3JtYWxpemVkVG8pO1xuICB9XG5cbiAgZGVsZXRlKHBhdGg6IHN0cmluZyk6IHZvaWQge1xuICAgIHRoaXMuX2RlbGV0ZSh0aGlzLl9ub3JtYWxpemVQYXRoKHBhdGgpKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBfb3ZlcndyaXRlKHBhdGg6IFBhdGgsIGNvbnRlbnQ6IEJ1ZmZlciwgYWN0aW9uPzogQWN0aW9uKSB7XG4gICAgaWYgKCF0aGlzLmhhcyhwYXRoKSkge1xuICAgICAgdGhyb3cgbmV3IEZpbGVEb2VzTm90RXhpc3RFeGNlcHRpb24ocGF0aCk7XG4gICAgfVxuICAgIC8vIFVwZGF0ZSB0aGUgYWN0aW9uIGJ1ZmZlci5cbiAgICBpZiAoYWN0aW9uKSB7XG4gICAgICB0aGlzLl9hY3Rpb25zLnB1c2goYWN0aW9uKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5fYWN0aW9ucy5vdmVyd3JpdGUocGF0aCwgY29udGVudCk7XG4gICAgfVxuICAgIHRoaXMuc2V0KG5ldyBTaW1wbGVGaWxlRW50cnkocGF0aCwgY29udGVudCkpO1xuICB9XG4gIHByb3RlY3RlZCBfY3JlYXRlKHBhdGg6IFBhdGgsIGNvbnRlbnQ6IEJ1ZmZlciwgYWN0aW9uPzogQWN0aW9uKSB7XG4gICAgaWYgKHRoaXMuX2NhY2hlTWFwLmhhcyhwYXRoKSkge1xuICAgICAgdGhyb3cgbmV3IEZpbGVBbHJlYWR5RXhpc3RFeGNlcHRpb24ocGF0aCk7XG4gICAgfVxuXG4gICAgaWYgKGFjdGlvbikge1xuICAgICAgdGhpcy5fYWN0aW9ucy5wdXNoKGFjdGlvbik7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuX2FjdGlvbnMuY3JlYXRlKHBhdGgsIGNvbnRlbnQpO1xuICAgIH1cbiAgICB0aGlzLnNldChuZXcgU2ltcGxlRmlsZUVudHJ5KHBhdGgsIGNvbnRlbnQgYXMgQnVmZmVyKSk7XG4gIH1cbiAgcHJvdGVjdGVkIF9yZW5hbWUocGF0aDogUGF0aCwgdG86IFBhdGgsIGFjdGlvbj86IEFjdGlvbiwgZm9yY2UgPSBmYWxzZSkge1xuICAgIGNvbnN0IGVudHJ5ID0gdGhpcy5nZXQocGF0aCk7XG4gICAgaWYgKCFlbnRyeSkge1xuICAgICAgdGhyb3cgbmV3IEZpbGVEb2VzTm90RXhpc3RFeGNlcHRpb24ocGF0aCk7XG4gICAgfVxuICAgIGlmICh0aGlzLl9jYWNoZU1hcC5oYXModG8pICYmICFmb3JjZSkge1xuICAgICAgdGhyb3cgbmV3IEZpbGVBbHJlYWR5RXhpc3RFeGNlcHRpb24odG8pO1xuICAgIH1cblxuICAgIGlmIChhY3Rpb24pIHtcbiAgICAgIHRoaXMuX2FjdGlvbnMucHVzaChhY3Rpb24pO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLl9hY3Rpb25zLnJlbmFtZShwYXRoLCB0byk7XG4gICAgfVxuXG4gICAgdGhpcy5zZXQobmV3IFNpbXBsZUZpbGVFbnRyeSh0bywgZW50cnkuY29udGVudCkpO1xuICAgIHRoaXMuX2NhY2hlTWFwLmRlbGV0ZShwYXRoKTtcbiAgfVxuICBwcm90ZWN0ZWQgX2RlbGV0ZShwYXRoOiBQYXRoLCBhY3Rpb24/OiBBY3Rpb24pIHtcbiAgICBpZiAoIXRoaXMuaGFzKHBhdGgpKSB7XG4gICAgICB0aHJvdyBuZXcgRmlsZURvZXNOb3RFeGlzdEV4Y2VwdGlvbihwYXRoKTtcbiAgICB9XG5cbiAgICBpZiAoYWN0aW9uKSB7XG4gICAgICB0aGlzLl9hY3Rpb25zLnB1c2goYWN0aW9uKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5fYWN0aW9ucy5kZWxldGUocGF0aCk7XG4gICAgfVxuICAgIHRoaXMuX2NhY2hlTWFwLmRlbGV0ZShwYXRoKTtcbiAgfVxuXG5cbiAgYXBwbHkoYWN0aW9uOiBBY3Rpb24sIHN0cmF0ZWd5OiBNZXJnZVN0cmF0ZWd5KSB7XG4gICAgaWYgKHRoaXMuX2FjdGlvbnMuaGFzKGFjdGlvbikpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgc3dpdGNoIChhY3Rpb24ua2luZCkge1xuICAgICAgY2FzZSAnbyc6XG4gICAgICAgIC8vIFVwZGF0ZSB0aGUgYWN0aW9uIGJ1ZmZlci5cbiAgICAgICAgdGhpcy5fb3ZlcndyaXRlKGFjdGlvbi5wYXRoLCBhY3Rpb24uY29udGVudCwgYWN0aW9uKTtcbiAgICAgICAgYnJlYWs7XG5cbiAgICAgIGNhc2UgJ2MnOlxuICAgICAgICBpZiAodGhpcy5fY2FjaGVNYXAuaGFzKGFjdGlvbi5wYXRoKSkge1xuICAgICAgICAgIHN3aXRjaCAoc3RyYXRlZ3kpIHtcbiAgICAgICAgICAgIGNhc2UgTWVyZ2VTdHJhdGVneS5FcnJvcjogdGhyb3cgbmV3IE1lcmdlQ29uZmxpY3RFeGNlcHRpb24oYWN0aW9uLnBhdGgpO1xuICAgICAgICAgICAgY2FzZSBNZXJnZVN0cmF0ZWd5Lk92ZXJ3cml0ZTpcbiAgICAgICAgICAgICAgdGhpcy5fb3ZlcndyaXRlKGFjdGlvbi5wYXRoLCBhY3Rpb24uY29udGVudCwgYWN0aW9uKTtcbiAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgfVxuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHRoaXMuX2NyZWF0ZShhY3Rpb24ucGF0aCwgYWN0aW9uLmNvbnRlbnQsIGFjdGlvbik7XG4gICAgICAgIH1cbiAgICAgICAgYnJlYWs7XG5cbiAgICAgIGNhc2UgJ3InOlxuICAgICAgICBjb25zdCBmb3JjZSA9IChzdHJhdGVneSAmIE1lcmdlU3RyYXRlZ3kuQWxsb3dPdmVyd3JpdGVDb25mbGljdCkgIT0gMDtcbiAgICAgICAgdGhpcy5fcmVuYW1lKGFjdGlvbi5wYXRoLCBhY3Rpb24udG8sIGFjdGlvbiwgZm9yY2UpO1xuICAgICAgICBicmVhaztcblxuICAgICAgY2FzZSAnZCc6IHRoaXMuX2RlbGV0ZShhY3Rpb24ucGF0aCwgYWN0aW9uKTsgYnJlYWs7XG5cbiAgICAgIGRlZmF1bHQ6IHRocm93IG5ldyBVbmtub3duQWN0aW9uRXhjZXB0aW9uKGFjdGlvbik7XG4gICAgfVxuICB9XG5cbiAgLy8gUmV0dXJucyBhbiBvcmRlcmVkIGxpc3Qgb2YgQWN0aW9uIHRvIGdldCB0aGlzIGhvc3QuXG4gIGdldCBhY3Rpb25zKCk6IEFjdGlvbltdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX2FjdGlvbnNdO1xuICB9XG5cbiAgLyoqXG4gICAqIEFsbG93IHN1YmNsYXNzZXMgdG8gY29weSB0byBhIHRyZWUgdGhlaXIgb3duIHByb3BlcnRpZXMuXG4gICAqIEByZXR1cm4ge1RyZWV9XG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcm90ZWN0ZWQgX2NvcHlUbzxUIGV4dGVuZHMgVmlydHVhbFRyZWU+KHRyZWU6IFQpOiB2b2lkIHtcbiAgICB0cmVlLl90cmVlID0gbmV3IE1hcCh0aGlzLnRyZWUpO1xuICAgIHRoaXMuX2FjdGlvbnMuZm9yRWFjaChhY3Rpb24gPT4gdHJlZS5fYWN0aW9ucy5wdXNoKGFjdGlvbikpO1xuICAgIFsuLi50aGlzLl9jYWNoZU1hcC5lbnRyaWVzKCldLmZvckVhY2goKFtwYXRoLCBlbnRyeV0pID0+IHtcbiAgICAgIHRyZWUuX2NhY2hlTWFwLnNldChwYXRoLCBlbnRyeSk7XG4gICAgfSk7XG4gIH1cblxuICBicmFuY2goKTogVHJlZSB7XG4gICAgY29uc3QgbmV3VHJlZSA9IG5ldyBWaXJ0dWFsVHJlZSgpO1xuICAgIHRoaXMuX2NvcHlUbyhuZXdUcmVlKTtcblxuICAgIHJldHVybiBuZXdUcmVlO1xuICB9XG5cbiAgLy8gQ3JlYXRlcyBhIG5ldyBob3N0IGZyb20gMiBob3N0cy5cbiAgbWVyZ2Uob3RoZXI6IFRyZWUsIHN0cmF0ZWd5OiBNZXJnZVN0cmF0ZWd5ID0gTWVyZ2VTdHJhdGVneS5EZWZhdWx0KSB7XG4gICAgb3RoZXIuYWN0aW9ucy5mb3JFYWNoKGFjdGlvbiA9PiB0aGlzLmFwcGx5KGFjdGlvbiwgc3RyYXRlZ3kpKTtcbiAgfVxuXG4gIG9wdGltaXplKCkge1xuICAgIC8vIFRoaXMgZGVzdHJveXMgdGhlIGhpc3RvcnkuIEhvcGUgeW91IGtub3cgd2hhdCB5b3UncmUgZG9pbmcuXG4gICAgdGhpcy5fYWN0aW9ucy5vcHRpbWl6ZSgpO1xuICB9XG5cbiAgc3RhdGljIGJyYW5jaCh0cmVlOiBUcmVlKSB7XG4gICAgcmV0dXJuIHRyZWUuYnJhbmNoKCk7XG4gIH1cblxuICBzdGF0aWMgbWVyZ2UodHJlZTogVHJlZSwgb3RoZXI6IFRyZWUsIHN0cmF0ZWd5OiBNZXJnZVN0cmF0ZWd5ID0gTWVyZ2VTdHJhdGVneS5EZWZhdWx0KTogVHJlZSB7XG4gICAgY29uc3QgbmV3VHJlZSA9IHRyZWUuYnJhbmNoKCk7XG4gICAgbmV3VHJlZS5tZXJnZShvdGhlciwgc3RyYXRlZ3kpO1xuXG4gICAgcmV0dXJuIG5ld1RyZWU7XG4gIH1cblxuICBzdGF0aWMgb3B0aW1pemUodHJlZTogVHJlZSkge1xuICAgIGNvbnN0IG5ld1RyZWUgPSAodHJlZSBhcyBWaXJ0dWFsVHJlZSkuYnJhbmNoKCkgYXMgVmlydHVhbFRyZWU7XG4gICAgbmV3VHJlZS5vcHRpbWl6ZSgpO1xuXG4gICAgcmV0dXJuIG5ld1RyZWU7XG4gIH1cbn1cbiJdfQ==