"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

var _repository = _interopRequireDefault(require("./repository"));

var _resolutionProgress = _interopRequireDefault(require("./conflicts/resolution-progress"));

var _fileSystemChangeObserver = _interopRequireDefault(require("./file-system-change-observer"));

var _workspaceChangeObserver = _interopRequireDefault(require("./workspace-change-observer"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

const createRepoSym = Symbol('createRepo');
let absentWorkdirContext;
/*
 * Bundle of model objects associated with a git working directory.
 *
 * Provides synchronous access to each model in the form of a getter method that returns the model or `null` if it
 * has not yet been initialized, and asynchronous access in the form of a Promise generation method that will resolve
 * once the model is available. Initializes the platform-appropriate change observer and proxies select filesystem
 * change events.
 */

class WorkdirContext {
  /*
   * Available options:
   * - `options.window`: Browser window global, used on Linux by the WorkspaceChangeObserver.
   * - `options.workspace`: Atom's workspace singleton, used on Linux by the WorkspaceChangeObserver.
   * - `options.promptCallback`: Callback used to collect information interactively through Atom.
   */
  constructor(directory, options = {}) {
    (0, _helpers.autobind)(this, 'repositoryChangedState');
    this.directory = directory;
    const {
      window: theWindow,
      workspace,
      promptCallback,
      pipelineManager
    } = options;

    this.repository = (options[createRepoSym] || (() => new _repository["default"](directory, null, {
      pipelineManager
    })))();

    this.destroyed = false;
    this.emitter = new _eventKit.Emitter();
    this.subs = new _eventKit.CompositeDisposable();
    this.observer = this.useWorkspaceChangeObserver() ? new _workspaceChangeObserver["default"](theWindow, workspace, this.repository) : new _fileSystemChangeObserver["default"](this.repository);
    this.resolutionProgress = new _resolutionProgress["default"]();

    if (promptCallback) {
      this.repository.setPromptCallback(promptCallback);
    } // Wire up event forwarding among models


    this.subs.add(this.repository.onDidChangeState(this.repositoryChangedState));
    this.subs.add(this.observer.onDidChange(events => {
      this.repository.observeFilesystemChange(events);
    }));
    this.subs.add(this.observer.onDidChangeWorkdirOrHead(() => this.emitter.emit('did-change-workdir-or-head'))); // If a pre-loaded Repository was provided, broadcast an initial state change event.

    this.repositoryChangedState({
      from: null,
      to: this.repository.state
    });
  }

  static absent(options) {
    if (!absentWorkdirContext) {
      absentWorkdirContext = new AbsentWorkdirContext(options);
    }

    return absentWorkdirContext;
  }

  static destroyAbsent() {
    if (absentWorkdirContext) {
      absentWorkdirContext.destroy();
      absentWorkdirContext = null;
    }
  }

  static guess(options, pipelineManager) {
    const projectPathCount = options.projectPathCount || 0;
    const initPathCount = options.initPathCount || 0;
    const createRepo = projectPathCount === 1 || projectPathCount === 0 && initPathCount === 1 ? () => _repository["default"].loadingGuess({
      pipelineManager
    }) : () => _repository["default"].absentGuess({
      pipelineManager
    });
    return new WorkdirContext(null, {
      [createRepoSym]: createRepo
    });
  }
  /**
   * Respond to changes in `Repository` state. Load resolution progress and start the change observer when it becomes
   * present. Stop the change observer when it is destroyed. Re-broadcast the event to context subscribers
   * regardless.
   *
   * The ResolutionProgress will be loaded before the change event is re-broadcast, but change observer modifications
   * will not be complete.
   */


  repositoryChangedState(payload) {
    if (this.destroyed) {
      return;
    }

    if (this.repository.isPresent()) {
      this.observer.start().then(() => this.emitter.emit('did-start-observer'));
    } else if (this.repository.isDestroyed()) {
      this.emitter.emit('did-destroy-repository');
      this.observer.destroy();
    }

    this.emitter.emit('did-change-repository-state', payload);
  }

  isPresent() {
    return true;
  }

  isDestroyed() {
    return this.destroyed;
  }

  useWorkspaceChangeObserver() {
    return !!process.env.ATOM_GITHUB_WORKSPACE_OBSERVER || process.platform === 'linux';
  } // Event subscriptions


  onDidStartObserver(callback) {
    return this.emitter.on('did-start-observer', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  onDidChangeRepositoryState(callback) {
    return this.emitter.on('did-change-repository-state', callback);
  }

  onDidUpdateRepository(callback) {
    return this.emitter.on('did-update-repository', callback);
  }

  onDidDestroyRepository(callback) {
    return this.emitter.on('did-destroy-repository', callback);
  }
  /**
   * Return a Promise that will resolve the next time that a Repository transitions to the requested state. Most
   * useful for test cases; most callers should prefer subscribing to `onDidChangeRepositoryState`.
   */


  getRepositoryStatePromise(stateName) {
    return new Promise(resolve => {
      const sub = this.onDidChangeRepositoryState(() => {
        if (this.repository.isInState(stateName)) {
          resolve();
          sub.dispose();
        }
      });
    });
  }
  /**
   * Return a Promise that will resolve the next time that a ChangeObserver successfully starts. Most useful for
   * test cases.
   */


  getObserverStartedPromise() {
    return new Promise(resolve => {
      const sub = this.onDidStartObserver(() => {
        resolve();
        sub.dispose();
      });
    });
  }

  getWorkingDirectory() {
    return this.directory;
  }

  getRepository() {
    return this.repository;
  }

  getChangeObserver() {
    return this.observer;
  }

  getResolutionProgress() {
    return this.resolutionProgress;
  }
  /*
   * Cleanly destroy any models that need to be cleaned, including stopping the filesystem watcher.
   */


  async destroy() {
    if (this.destroyed) {
      return;
    }

    this.destroyed = true;
    this.subs.dispose();
    this.repository.destroy();
    this.emitter.dispose();
    await this.observer.destroy();
  }

}

exports["default"] = WorkdirContext;

class AbsentWorkdirContext extends WorkdirContext {
  constructor(options) {
    super(null, {
      [createRepoSym]: () => _repository["default"].absent(options)
    });
  }

  isPresent() {
    return false;
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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