"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var electron_1 = require("electron");
var path = require("path");
var locale = require("../locale/locale");
var EnvironmentUtil_1 = require("../runtime/EnvironmentUtil");
var lifecycle_1 = require("../runtime/lifecycle");
var config_1 = require("../settings/config");
var WindowManager_1 = require("../window/WindowManager");
var TrayHandler = (function () {
    function TrayHandler() {
        this.lastUnreadCount = 0;
    }
    TrayHandler.prototype.initTray = function (trayIcon) {
        if (trayIcon === void 0) { trayIcon = new electron_1.Tray(electron_1.nativeImage.createEmpty()); }
        var IMAGE_ROOT = path.join(electron_1.app.getAppPath(), config_1.config.electronDirectory, 'img');
        var trayPng = 'tray.png';
        var trayBadgePng = 'tray.badge.png';
        if (EnvironmentUtil_1.platform.IS_LINUX) {
            trayPng = "tray" + (EnvironmentUtil_1.linuxDesktop.isGnomeX11 ? '.gnome' : '@3x') + ".png";
            trayBadgePng = "tray.badge" + (EnvironmentUtil_1.linuxDesktop.isGnomeX11 ? '.gnome' : '@3x') + ".png";
        }
        var iconPaths = {
            badge: path.join(IMAGE_ROOT, 'taskbar.overlay.png'),
            tray: path.join(IMAGE_ROOT, 'tray-icon/tray', trayPng),
            trayWithBadge: path.join(IMAGE_ROOT, 'tray-icon/tray-with-badge', trayBadgePng),
        };
        this.icons = {
            badge: electron_1.nativeImage.createFromPath(iconPaths.badge),
            tray: electron_1.nativeImage.createFromPath(iconPaths.tray),
            trayWithBadge: electron_1.nativeImage.createFromPath(iconPaths.trayWithBadge),
        };
        this.trayIcon = trayIcon;
        this.trayIcon.setImage(this.icons.tray);
        this.buildTrayMenu();
    };
    TrayHandler.prototype.showUnreadCount = function (win, count) {
        this.updateIcons(win, count);
        this.flashApplicationWindow(win, count);
        this.updateBadgeCount(count);
    };
    TrayHandler.prototype.buildTrayMenu = function () {
        var contextMenu = electron_1.Menu.buildFromTemplate([
            {
                click: function () { return WindowManager_1.WindowManager.showPrimaryWindow(); },
                label: locale.getText('trayOpen'),
            },
            {
                click: function () { return lifecycle_1.quit(); },
                label: locale.getText('trayQuit'),
            },
        ]);
        if (this.trayIcon) {
            this.trayIcon.on('click', function () { return WindowManager_1.WindowManager.showPrimaryWindow(); });
            this.trayIcon.setContextMenu(contextMenu);
            this.trayIcon.setToolTip(config_1.config.name);
        }
    };
    TrayHandler.prototype.flashApplicationWindow = function (win, count) {
        if (win.isFocused() || !count) {
            win.flashFrame(false);
        }
        else if (count > this.lastUnreadCount) {
            win.flashFrame(true);
        }
    };
    TrayHandler.prototype.updateBadgeCount = function (count) {
        if (typeof count !== 'undefined') {
            electron_1.app.setBadgeCount(count);
            this.lastUnreadCount = count;
        }
    };
    TrayHandler.prototype.updateIcons = function (win, count) {
        if (this.icons) {
            var trayImage = count ? this.icons.trayWithBadge : this.icons.tray;
            if (this.trayIcon) {
                this.trayIcon.setImage(trayImage);
            }
            var overlayImage = count ? this.icons.badge : null;
            win.setOverlayIcon(overlayImage, locale.getText('unreadMessages'));
        }
    };
    return TrayHandler;
}());
exports.TrayHandler = TrayHandler;
//# sourceMappingURL=TrayHandler.js.map