/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectorRef } from '../change_detection/change_detection';
import { Injector } from '../di/injector';
import { Type } from '../type';
import { ElementRef } from './element_ref';
import { NgModuleRef } from './ng_module_factory';
import { ViewRef } from './view_ref';
/**
 * Represents a component created by a `ComponentFactory`.
 * Provides access to the component instance and related objects,
 * and provides the means of destroying the instance.
 *
 * @publicApi
 */
export declare abstract class ComponentRef<C> {
    /**
     * The host or anchor [element](guide/glossary#element) for this component instance.
     */
    abstract readonly location: ElementRef;
    /**
     * The [dependency injector](guide/glossary#injector) for this component instance.
     */
    abstract readonly injector: Injector;
    /**
     * This component instance.
     */
    abstract readonly instance: C;
    /**
     * The [host view](guide/glossary#view-tree) defined by the template
     * for this component instance.
     */
    abstract readonly hostView: ViewRef;
    /**
     * The change detector for this component instance.
     */
    abstract readonly changeDetectorRef: ChangeDetectorRef;
    /**
     * The component type.
     */
    abstract readonly componentType: Type<any>;
    /**
     * Destroys the component instance and all of the data structures associated with it.
     */
    abstract destroy(): void;
    /**
     * A lifecycle hook that provides additional developer-defined cleanup
     * functionality for the component.
     * @param callback A handler function that cleans up developer-defined data
     * associated with this component. Called when the `destroy()` method is invoked.
     */
    abstract onDestroy(callback: Function): void;
}
/**
 * @publicApi
 */
export declare abstract class ComponentFactory<C> {
    /**
     * The component's HTML selector.
     */
    abstract readonly selector: string;
    /**
     * The component's type
     */
    abstract readonly componentType: Type<any>;
    /**
     * Selector for all <ng-content> elements in the component.
     */
    abstract readonly ngContentSelectors: string[];
    /**
     * The inputs of the component.
     */
    abstract readonly inputs: {
        propName: string;
        templateName: string;
    }[];
    /**
     * The outputs of the component.
     */
    abstract readonly outputs: {
        propName: string;
        templateName: string;
    }[];
    /**
     * Creates a new component.
     */
    abstract create(injector: Injector, projectableNodes?: any[][], rootSelectorOrNode?: string | any, ngModule?: NgModuleRef<any>): ComponentRef<C>;
}
