import { Appservice, MatrixClient } from "..";
import { IAppserviceOptions } from "./Appservice";
/**
 * An Intent is an intelligent client that tracks things like the user's membership
 * in rooms to ensure the action being performed is possible. This is very similar
 * to how Intents work in the matrix-js-sdk in that the Intent will ensure that the
 * user is joined to the room before posting a message, for example.
 */
export declare class Intent {
    private impersonateUserId;
    private appservice;
    private readonly client;
    private readonly storage;
    private knownJoinedRooms;
    /**
     * Creates a new intent. Intended to be created by application services.
     * @param {IAppserviceOptions} options The options for the application service.
     * @param {string} impersonateUserId The user ID to impersonate.
     * @param {Appservice} appservice The application service itself.
     */
    constructor(options: IAppserviceOptions, impersonateUserId: string, appservice: Appservice);
    /**
     * Gets the user ID this intent is for.
     */
    readonly userId: string;
    /**
     * Gets the underlying MatrixClient that powers this Intent.
     */
    readonly underlyingClient: MatrixClient;
    /**
     * Gets the joined rooms for the intent. Note that by working around
     * the intent to join rooms may yield inaccurate results.
     * @returns {Promise<string[]>} Resolves to an array of room IDs where
     * the intent is joined.
     */
    getJoinedRooms(): Promise<string[]>;
    /**
     * Leaves the given room.
     * @param {string} roomId The room ID to leave
     * @returns {Promise<*>} Resolves when the room has been left.
     */
    leaveRoom(roomId: string): Promise<any>;
    /**
     * Joins the given room
     * @param {string} roomIdOrAlias the room ID or alias to join
     * @returns {Promise<string>} resolves to the joined room ID
     */
    joinRoom(roomIdOrAlias: string): Promise<string>;
    /**
     * Sends a text message to a room.
     * @param {string} roomId The room ID to send text to.
     * @param {string} body The message body to send.
     * @param {"m.text" | "m.emote" | "m.notice"} msgtype The message type to send.
     * @returns {Promise<string>} Resolves to the event ID of the sent message.
     */
    sendText(roomId: string, body: string, msgtype?: "m.text" | "m.emote" | "m.notice"): Promise<string>;
    /**
     * Sends an event to a room.
     * @param {string} roomId The room ID to send the event to.
     * @param {*} content The content of the event.
     * @returns {Promise<string>} Resolves to the event ID of the sent event.
     */
    sendEvent(roomId: string, content: any): Promise<string>;
    /**
     * Ensures the user is registered and joined to the given room.
     * @param {string} roomId The room ID to join
     * @returns {Promise<*>} Resolves when complete
     */
    ensureRegisteredAndJoined(roomId: string): Promise<void>;
    /**
     * Ensures the user is joined to the given room
     * @param {string} roomId The room ID to join
     * @returns {Promise<*>} Resolves when complete
     */
    ensureJoined(roomId: string): Promise<string>;
    /**
     * Refreshes which rooms the user is joined to, potentially saving time on
     * calls like ensureJoined()
     * @returns {Promise<string[]>} Resolves to the joined room IDs for the user.
     */
    refreshJoinedRooms(): Promise<string[]>;
    /**
     * Ensures the user is registered
     * @returns {Promise<*>} Resolves when complete
     */
    ensureRegistered(): Promise<any>;
}
