import { toString } from './function';
export const URI = 'Free';
export class Pure {
    constructor(value) {
        this.value = value;
        this._tag = 'Pure';
    }
    map(f) {
        return new Pure(f(this.value));
    }
    ap(fab) {
        return fab.chain(f => this.map(f)); // <- derived
    }
    /**
     * Flipped version of `ap`
     */
    ap_(fb) {
        return fb.ap(this);
    }
    chain(f) {
        return f(this.value);
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `new Pure(${toString(this.value)})`;
    }
    isPure() {
        return true;
    }
    isImpure() {
        return false;
    }
}
export class Impure {
    constructor(fx, f) {
        this.fx = fx;
        this.f = f;
        this._tag = 'Impure';
    }
    map(f) {
        return new Impure(this.fx, x => this.f(x).map(f));
    }
    ap(fab) {
        return fab.chain(f => this.map(f)); // <- derived
    }
    ap_(fb) {
        return fb.ap(this);
    }
    chain(f) {
        return new Impure(this.fx, x => this.f(x).chain(f));
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `new Impure(${(toString(this.fx), toString(this.f))})`;
    }
    isPure() {
        return false;
    }
    isImpure() {
        return true;
    }
}
/**
 * @since 1.0.0
 */
export const of = (a) => {
    return new Pure(a);
};
/**
 * Lift an impure value described by the generating type constructor `F` into the free monad
 *
 * @since 1.0.0
 */
export const liftF = (fa) => {
    return new Impure(fa, a => of(a));
};
const substFree = (f) => {
    function go(fa) {
        switch (fa._tag) {
            case 'Pure':
                return of(fa.value);
            case 'Impure':
                return f(fa.fx).chain(x => go(fa.f(x)));
        }
    }
    return go;
};
export function hoistFree(nt) {
    return substFree(fa => liftF(nt(fa)));
}
export function foldFree(M) {
    return (nt, fa) => {
        if (fa.isPure()) {
            return M.of(fa.value);
        }
        else {
            return M.chain(nt(fa.fx), x => foldFree(M)(nt, fa.f(x)));
        }
    };
}
