/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { NgModuleFactory, NgModuleRef, Type } from '@angular/core';
import { Observable } from 'rxjs';
import { ActivatedRouteSnapshot } from './router_state';
import { UrlSegment, UrlSegmentGroup } from './url_tree';
/**
 * @description
 *
 * Represents router configuration.
 *
 * `Routes` is an array of route configurations. Each one has the following properties:
 *
 * - `path` is a string that uses the route matcher DSL.
 * - `pathMatch` is a string that specifies the matching strategy. Options are `prefix` (default)
 *   and `full`. See [Matching Strategy](#matching-strategy) below for more information.
 * - `matcher` defines a custom strategy for path matching and supersedes `path` and `pathMatch`.
 * - `component` is a component type.
 * - `redirectTo` is the url fragment which will replace the current matched segment.
 * - `outlet` is the name of the outlet the component should be placed into.
 * - `canActivate` is an array of DI tokens used to look up CanActivate handlers. See
 *   `CanActivate` for more info.
 * - `canActivateChild` is an array of DI tokens used to look up CanActivateChild handlers. See
 *   `CanActivateChild` for more info.
 * - `canDeactivate` is an array of DI tokens used to look up CanDeactivate handlers. See
 *   `CanDeactivate` for more info.
 * - `canLoad` is an array of DI tokens used to look up CanLoad handlers. See
 *   `CanLoad` for more info.
 * - `data` is additional data provided to the component via `ActivatedRoute`.
 * - `resolve` is a map of DI tokens used to look up data resolvers. See `Resolve` for more
 *   info.
 * - `runGuardsAndResolvers` defines when guards and resolvers will be run. By default they run only
 *    when the matrix parameters of the route change. Options include:
 *    - `paramsChange` (default) - Run guards and resolvers when path or matrix params change. This
 *      mode ignores query param changes.
 *    - `paramsOrQueryParamsChange` - Guards and resolvers will run when any parameters change. This
 *      includes path, matrix, and query params.
 *    - `pathParamsChange` - Run guards and resolvers path or any path params change. This mode is
 *      useful if you want to ignore changes to all optional parameters such as query *and* matrix
 *      params.
 *    - `pathParamsOrQueryParamsChange` - Same as `pathParamsChange`, but also rerun when any query
 *      param changes
 *    - `always` - Run guards and resolvers on every navigation.
 *    - (from: ActivatedRouteSnapshot, to: ActivatedRouteSnapshot) => boolean - Use a predicate
 *      function when none of the pre-configured modes fit the needs of the application. An example
 *      might be when you need to ignore updates to a param such as `sortDirection`, but need to
 *      reload guards and resolvers when changing the `searchRoot` param.
 * - `children` is an array of child route definitions.
 * - `loadChildren` is a reference to lazy loaded child routes. See `LoadChildren` for more
 *   info.
 *
 * @usageNotes
 * ### Simple Configuration
 *
 * ```
 * [{
 *   path: 'team/:id',
  *  component: Team,
 *   children: [{
 *     path: 'user/:name',
 *     component: User
 *   }]
 * }]
 * ```
 *
 * When navigating to `/team/11/user/bob`, the router will create the team component with the user
 * component in it.
 *
 * ### Multiple Outlets
 *
 * ```
 * [{
 *   path: 'team/:id',
 *   component: Team
 * }, {
 *   path: 'chat/:user',
 *   component: Chat
 *   outlet: 'aux'
 * }]
 * ```
 *
 * When navigating to `/team/11(aux:chat/jim)`, the router will create the team component next to
 * the chat component. The chat component will be placed into the aux outlet.
 *
 * ### Wild Cards
 *
 * ```
 * [{
 *   path: '**',
 *   component: Sink
 * }]
 * ```
 *
 * Regardless of where you navigate to, the router will instantiate the sink component.
 *
 * ### Redirects
 *
 * ```
 * [{
 *   path: 'team/:id',
 *   component: Team,
 *   children: [{
 *     path: 'legacy/user/:name',
 *     redirectTo: 'user/:name'
 *   }, {
 *     path: 'user/:name',
 *     component: User
 *   }]
 * }]
 * ```
 *
 * When navigating to '/team/11/legacy/user/jim', the router will change the url to
 * '/team/11/user/jim', and then will instantiate the team component with the user component
 * in it.
 *
 * If the `redirectTo` value starts with a '/', then it is an absolute redirect. E.g., if in the
 * example above we change the `redirectTo` to `/user/:name`, the result url will be '/user/jim'.
 *
 * ### Empty Path
 *
 * Empty-path route configurations can be used to instantiate components that do not 'consume'
 * any url segments. Let's look at the following configuration:
 *
 * ```
 * [{
 *   path: 'team/:id',
 *   component: Team,
 *   children: [{
 *     path: '',
 *     component: AllUsers
 *   }, {
 *     path: 'user/:name',
 *     component: User
 *   }]
 * }]
 * ```
 *
 * When navigating to `/team/11`, the router will instantiate the AllUsers component.
 *
 * Empty-path routes can have children.
 *
 * ```
 * [{
 *   path: 'team/:id',
 *   component: Team,
 *   children: [{
 *     path: '',
 *     component: WrapperCmp,
 *     children: [{
 *       path: 'user/:name',
 *       component: User
 *     }]
 *   }]
 * }]
 * ```
 *
 * When navigating to `/team/11/user/jim`, the router will instantiate the wrapper component with
 * the user component in it.
 *
 * An empty path route inherits its parent's params and data. This is because it cannot have its
 * own params, and, as a result, it often uses its parent's params and data as its own.
 *
 * ### Matching Strategy
 *
 * By default the router will look at what is left in the url, and check if it starts with
 * the specified path (e.g., `/team/11/user` starts with `team/:id`).
 *
 * We can change the matching strategy to make sure that the path covers the whole unconsumed url,
 * which is akin to `unconsumedUrl === path` or `$` regular expressions.
 *
 * This is particularly important when redirecting empty-path routes.
 *
 * ```
 * [{
 *   path: '',
 *   pathMatch: 'prefix', //default
 *   redirectTo: 'main'
 * }, {
 *   path: 'main',
 *   component: Main
 * }]
 * ```
 *
 * Since an empty path is a prefix of any url, even when navigating to '/main', the router will
 * still apply the redirect.
 *
 * If `pathMatch: full` is provided, the router will apply the redirect if and only if navigating to
 * '/'.
 *
 * ```
 * [{
 *   path: '',
 *   pathMatch: 'full',
 *   redirectTo: 'main'
 * }, {
 *   path: 'main',
 *   component: Main
 * }]
 * ```
 *
 * ### Componentless Routes
 *
 * It is useful at times to have the ability to share parameters between sibling components.
 *
 * Say we have two components--ChildCmp and AuxCmp--that we want to put next to each other and both
 * of them require some id parameter.
 *
 * One way to do that would be to have a bogus parent component, so both the siblings can get the id
 * parameter from it. This is not ideal. Instead, you can use a componentless route.
 *
 * ```
 * [{
 *    path: 'parent/:id',
 *    children: [
 *      { path: 'a', component: MainChild },
 *      { path: 'b', component: AuxChild, outlet: 'aux' }
 *    ]
 * }]
 * ```
 *
 * So when navigating to `parent/10/(a//aux:b)`, the route will instantiate the main child and aux
 * child components next to each other. In this example, the application component
 * has to have the primary and aux outlets defined.
 *
 * The router will also merge the `params`, `data`, and `resolve` of the componentless parent into
 * the `params`, `data`, and `resolve` of the children. This is done because there is no component
 * that can inject the activated route of the componentless parent.
 *
 * This is especially useful when child components are defined as follows:
 *
 * ```
 * [{
 *    path: 'parent/:id',
 *    children: [
 *      { path: '', component: MainChild },
 *      { path: '', component: AuxChild, outlet: 'aux' }
 *    ]
 * }]
 * ```
 *
 * With this configuration in place, navigating to '/parent/10' will create the main child and aux
 * components.
 *
 * ### Lazy Loading
 *
 * Lazy loading speeds up our application load time by splitting it into multiple bundles, and
 * loading them on demand. The router is designed to make lazy loading simple and easy. Instead of
 * providing the children property, you can provide the `loadChildren` property, as follows:
 *
 * ```
 * [{
 *   path: 'team/:id',
 *   component: Team,
 *   loadChildren: 'team'
 * }]
 * ```
 *
 * The router will use registered NgModuleFactoryLoader to fetch an NgModule associated with 'team'.
 * Then it will extract the set of routes defined in that NgModule, and will transparently add
 * those routes to the main configuration.
 *
 * @publicApi
 */
export declare type Routes = Route[];
/**
 * @description Represents the results of the URL matching.
 *
 * * `consumed` is an array of the consumed URL segments.
 * * `posParams` is a map of positional parameters.
 *
 * @publicApi
 */
export declare type UrlMatchResult = {
    consumed: UrlSegment[];
    posParams?: {
        [name: string]: UrlSegment;
    };
};
/**
 * @description
 *
 * A function matching URLs
 *
 * A custom URL matcher can be provided when a combination of `path` and `pathMatch` isn't
 * expressive enough.
 *
 * For instance, the following matcher matches html files.
 *
 * ```
 * export function htmlFiles(url: UrlSegment[]) {
 *   return url.length === 1 && url[0].path.endsWith('.html') ? ({consumed: url}) : null;
 * }
 *
 * export const routes = [{ matcher: htmlFiles, component: AnyComponent }];
 * ```
 *
 * @publicApi
 */
export declare type UrlMatcher = (segments: UrlSegment[], group: UrlSegmentGroup, route: Route) => UrlMatchResult;
/**
 * @description
 *
 * Represents the static data associated with a particular route.
 *
 * See `Routes` for more details.
 *
 * @publicApi
 */
export declare type Data = {
    [name: string]: any;
};
/**
 * @description
 *
 * Represents the resolved data associated with a particular route.
 *
 * See `Routes` for more details.
 *
 * @publicApi
 */
export declare type ResolveData = {
    [name: string]: any;
};
/**
 * @description
 *
 * The type of `loadChildren`.
 *
 * See `Routes` for more details.
 *
 * @publicApi
 */
export declare type LoadChildrenCallback = () => Type<any> | NgModuleFactory<any> | Promise<Type<any>> | Observable<Type<any>>;
/**
 * @description
 *
 * The type of `loadChildren`.
 *
 * See `Routes` for more details.
 *
 * @publicApi
 */
export declare type LoadChildren = string | LoadChildrenCallback;
/**
 * @description
 *
 * The type of `queryParamsHandling`.
 *
 * See `RouterLink` for more details.
 *
 */
export declare type QueryParamsHandling = 'merge' | 'preserve' | '';
/**
 * @description
 *
 * The type of `runGuardsAndResolvers`.
 *
 * See `Routes` for more details.
 * @publicApi
 */
export declare type RunGuardsAndResolvers = 'pathParamsChange' | 'pathParamsOrQueryParamsChange' | 'paramsChange' | 'paramsOrQueryParamsChange' | 'always' | ((from: ActivatedRouteSnapshot, to: ActivatedRouteSnapshot) => boolean);
/**
 * See `Routes` for more details.
 *
 * @publicApi
 */
export interface Route {
    path?: string;
    pathMatch?: string;
    matcher?: UrlMatcher;
    component?: Type<any>;
    redirectTo?: string;
    outlet?: string;
    canActivate?: any[];
    canActivateChild?: any[];
    canDeactivate?: any[];
    canLoad?: any[];
    data?: Data;
    resolve?: ResolveData;
    children?: Routes;
    loadChildren?: LoadChildren;
    runGuardsAndResolvers?: RunGuardsAndResolvers;
}
export declare class LoadedRouterConfig {
    routes: Route[];
    module: NgModuleRef<any>;
    constructor(routes: Route[], module: NgModuleRef<any>);
}
export declare function validateConfig(config: Routes, parentPath?: string): void;
/**
 * Makes a copy of the config and adds any default required properties.
 */
export declare function standardizeConfig(r: Route): Route;
