(function() {
  var AtomReporter, SpecResultView, SuiteResultView, _, formatStackTrace, grim, ipcHelpers, listen, marked, path, process;

  path = require('path');

  process = require('process');

  _ = require('underscore-plus');

  grim = require('grim');

  marked = require('marked');

  listen = require('../src/delegated-listener');

  ipcHelpers = require('../src/ipc-helpers');

  formatStackTrace = function(spec, message, stackTrace) {
    var errorMatch, firstJasmineLinePattern, i, jasminePattern, len, line, lines, ref, ref1, ref2;
    if (message == null) {
      message = '';
    }
    if (!stackTrace) {
      return stackTrace;
    }
    jasminePattern = /^\s*at\s+.*\(?.*[\/\\]jasmine(-[^\/\\]*)?\.js:\d+:\d+\)?\s*$/;
    firstJasmineLinePattern = /^\s*at\s+jasmine\.[A-Z][^\s]*\s+\(?.*[\/\\]jasmine(-[^\/\\]*)?\.js:\d+:\d+\)?\s*$/;
    lines = [];
    ref = stackTrace.split('\n');
    for (i = 0, len = ref.length; i < len; i++) {
      line = ref[i];
      if (firstJasmineLinePattern.test(line)) {
        break;
      }
      if (!jasminePattern.test(line)) {
        lines.push(line);
      }
    }
    errorMatch = (ref1 = lines[0]) != null ? ref1.match(/^Error: (.*)/) : void 0;
    if (message.trim() === (errorMatch != null ? (ref2 = errorMatch[1]) != null ? ref2.trim() : void 0 : void 0)) {
      lines.shift();
    }
    lines = lines.map(function(line) {
      if (/^\s*at\s/.test(line)) {
        if (process.platform === 'win32' && /file:\/\/\//.test(line)) {
          line = line.replace('file:///', '').replace(RegExp("" + path.posix.sep, "g"), path.win32.sep);
        }
        line = line.trim().replace(/^at jasmine\.Spec\.<anonymous> \(([^)]+)\)/, 'at $1').replace(/^at jasmine\.Spec\.f*it \(([^)]+)\)/, 'at $1').replace(/^at f*it \(([^)]+)\)/, 'at $1').replace(spec.specDirectory + path.sep, '');
      }
      return line;
    });
    return lines.join('\n').trim();
  };

  module.exports = AtomReporter = (function() {
    function AtomReporter() {
      var element, i, len, ref;
      this.element = document.createElement('div');
      this.element.classList.add('spec-reporter-container');
      this.element.innerHTML = "<div class=\"spec-reporter\">\n  <div class=\"padded pull-right\">\n    <button outlet=\"reloadButton\" class=\"btn btn-small reload-button\">Reload Specs</button>\n  </div>\n  <div outlet=\"coreArea\" class=\"symbol-area\">\n    <div outlet=\"coreHeader\" class=\"symbol-header\"></div>\n    <ul outlet=\"coreSummary\"class=\"symbol-summary list-unstyled\"></ul>\n  </div>\n  <div outlet=\"bundledArea\" class=\"symbol-area\">\n    <div outlet=\"bundledHeader\" class=\"symbol-header\"></div>\n    <ul outlet=\"bundledSummary\"class=\"symbol-summary list-unstyled\"></ul>\n  </div>\n  <div outlet=\"userArea\" class=\"symbol-area\">\n    <div outlet=\"userHeader\" class=\"symbol-header\"></div>\n    <ul outlet=\"userSummary\"class=\"symbol-summary list-unstyled\"></ul>\n  </div>\n  <div outlet=\"status\" class=\"status alert alert-info\">\n    <div outlet=\"time\" class=\"time\"></div>\n    <div outlet=\"specCount\" class=\"spec-count\"></div>\n    <div outlet=\"message\" class=\"message\"></div>\n  </div>\n  <div outlet=\"results\" class=\"results\"></div>\n  <div outlet=\"deprecations\" class=\"status alert alert-warning\" style=\"display: none\">\n    <span outlet=\"deprecationStatus\">0 deprecations</span>\n    <div class=\"deprecation-toggle\"></div>\n  </div>\n  <div outlet=\"deprecationList\" class=\"deprecation-list\"></div>\n</div>";
      ref = this.element.querySelectorAll('[outlet]');
      for (i = 0, len = ref.length; i < len; i++) {
        element = ref[i];
        this[element.getAttribute('outlet')] = element;
      }
    }

    AtomReporter.prototype.startedAt = null;

    AtomReporter.prototype.runningSpecCount = 0;

    AtomReporter.prototype.completeSpecCount = 0;

    AtomReporter.prototype.passedCount = 0;

    AtomReporter.prototype.failedCount = 0;

    AtomReporter.prototype.skippedCount = 0;

    AtomReporter.prototype.totalSpecCount = 0;

    AtomReporter.prototype.deprecationCount = 0;

    AtomReporter.timeoutId = 0;

    AtomReporter.prototype.reportRunnerStarting = function(runner) {
      var specs;
      this.handleEvents();
      this.startedAt = Date.now();
      specs = runner.specs();
      this.totalSpecCount = specs.length;
      this.addSpecs(specs);
      return document.body.appendChild(this.element);
    };

    AtomReporter.prototype.reportRunnerResults = function(runner) {
      this.updateSpecCounts();
      if (this.failedCount === 0) {
        this.status.classList.add('alert-success');
        this.status.classList.remove('alert-info');
      }
      if (this.failedCount === 1) {
        return this.message.textContent = this.failedCount + " failure";
      } else {
        return this.message.textContent = this.failedCount + " failures";
      }
    };

    AtomReporter.prototype.reportSuiteResults = function(suite) {};

    AtomReporter.prototype.reportSpecResults = function(spec) {
      this.completeSpecCount++;
      spec.endedAt = Date.now();
      this.specComplete(spec);
      return this.updateStatusView(spec);
    };

    AtomReporter.prototype.reportSpecStarting = function(spec) {
      return this.specStarted(spec);
    };

    AtomReporter.prototype.handleEvents = function() {
      listen(document, 'click', '.spec-toggle', function(event) {
        var specFailures;
        specFailures = event.currentTarget.parentElement.querySelector('.spec-failures');
        if (specFailures.style.display === 'none') {
          specFailures.style.display = '';
          event.currentTarget.classList.remove('folded');
        } else {
          specFailures.style.display = 'none';
          event.currentTarget.classList.add('folded');
        }
        return event.preventDefault();
      });
      listen(document, 'click', '.deprecation-list', function(event) {
        var deprecationList;
        deprecationList = event.currentTarget.parentElement.querySelector('.deprecation-list');
        if (deprecationList.style.display === 'none') {
          deprecationList.style.display = '';
          event.currentTarget.classList.remove('folded');
        } else {
          deprecationList.style.display = 'none';
          event.currentTarget.classList.add('folded');
        }
        return event.preventDefault();
      });
      listen(document, 'click', '.stack-trace', function(event) {
        return event.currentTarget.classList.toggle('expanded');
      });
      return this.reloadButton.addEventListener('click', function() {
        return ipcHelpers.call('window-method', 'reload');
      });
    };

    AtomReporter.prototype.updateSpecCounts = function() {
      var specCount;
      if (this.skippedCount) {
        specCount = (this.completeSpecCount - this.skippedCount) + "/" + (this.totalSpecCount - this.skippedCount) + " (" + this.skippedCount + " skipped)";
      } else {
        specCount = this.completeSpecCount + "/" + this.totalSpecCount;
      }
      return this.specCount.textContent = specCount;
    };

    AtomReporter.prototype.updateStatusView = function(spec) {
      var rootSuite, time;
      if (this.failedCount > 0) {
        this.status.classList.add('alert-danger');
        this.status.classList.remove('alert-info');
      }
      this.updateSpecCounts();
      rootSuite = spec.suite;
      while (rootSuite.parentSuite) {
        rootSuite = rootSuite.parentSuite;
      }
      this.message.textContent = rootSuite.description;
      time = "" + (Math.round((spec.endedAt - this.startedAt) / 10));
      if (time.length < 3) {
        time = "0" + time;
      }
      return this.time.textContent = time.slice(0, -2) + "." + time.slice(-2) + "s";
    };

    AtomReporter.prototype.specTitle = function(spec) {
      var desc, i, indent, len, parentDescs, s, suiteString;
      parentDescs = [];
      s = spec.suite;
      while (s) {
        parentDescs.unshift(s.description);
        s = s.parentSuite;
      }
      suiteString = "";
      indent = "";
      for (i = 0, len = parentDescs.length; i < len; i++) {
        desc = parentDescs[i];
        suiteString += indent + desc + "\n";
        indent += "  ";
      }
      return suiteString + " " + indent + " it " + spec.description;
    };

    AtomReporter.prototype.addSpecs = function(specs) {
      var bundledPackageSpecs, coreSpecs, i, len, packageFolderName, packageName, spec, specDirectory, symbol, userPackageSpecs;
      coreSpecs = 0;
      bundledPackageSpecs = 0;
      userPackageSpecs = 0;
      for (i = 0, len = specs.length; i < len; i++) {
        spec = specs[i];
        symbol = document.createElement('li');
        symbol.setAttribute('id', "spec-summary-" + spec.id);
        symbol.setAttribute('title', this.specTitle(spec));
        symbol.className = "spec-summary pending";
        switch (spec.specType) {
          case 'core':
            coreSpecs++;
            this.coreSummary.appendChild(symbol);
            break;
          case 'bundled':
            bundledPackageSpecs++;
            this.bundledSummary.appendChild(symbol);
            break;
          case 'user':
            userPackageSpecs++;
            this.userSummary.appendChild(symbol);
        }
      }
      if (coreSpecs > 0) {
        this.coreHeader.textContent = "Core Specs (" + coreSpecs + ")";
      } else {
        this.coreArea.style.display = 'none';
      }
      if (bundledPackageSpecs > 0) {
        this.bundledHeader.textContent = "Bundled Package Specs (" + bundledPackageSpecs + ")";
      } else {
        this.bundledArea.style.display = 'none';
      }
      if (userPackageSpecs > 0) {
        if (coreSpecs === 0 && bundledPackageSpecs === 0) {
          specDirectory = specs[0].specDirectory;
          packageFolderName = path.basename(path.dirname(specDirectory));
          packageName = _.undasherize(_.uncamelcase(packageFolderName));
          return this.userHeader.textContent = packageName + " Specs";
        } else {
          return this.userHeader.textContent = "User Package Specs (" + userPackageSpecs + ")";
        }
      } else {
        return this.userArea.style.display = 'none';
      }
    };

    AtomReporter.prototype.specStarted = function(spec) {
      return this.runningSpecCount++;
    };

    AtomReporter.prototype.specComplete = function(spec) {
      var results, specSummaryElement, specView;
      specSummaryElement = document.getElementById("spec-summary-" + spec.id);
      specSummaryElement.classList.remove('pending');
      results = spec.results();
      if (results.skipped) {
        specSummaryElement.classList.add("skipped");
        return this.skippedCount++;
      } else if (results.passed()) {
        specSummaryElement.classList.add("passed");
        return this.passedCount++;
      } else {
        specSummaryElement.classList.add("failed");
        specView = new SpecResultView(spec);
        specView.attach();
        return this.failedCount++;
      }
    };

    return AtomReporter;

  })();

  SuiteResultView = (function() {
    function SuiteResultView(suite1) {
      this.suite = suite1;
      this.element = document.createElement('div');
      this.element.className = 'suite';
      this.element.setAttribute('id', "suite-view-" + this.suite.id);
      this.description = document.createElement('div');
      this.description.className = 'description';
      this.description.textContent = this.suite.description;
      this.element.appendChild(this.description);
    }

    SuiteResultView.prototype.attach = function() {
      return (this.parentSuiteView() || document.querySelector('.results')).appendChild(this.element);
    };

    SuiteResultView.prototype.parentSuiteView = function() {
      var suiteView, suiteViewElement;
      if (!this.suite.parentSuite) {
        return;
      }
      if (!(suiteViewElement = document.querySelector("#suite-view-" + this.suite.parentSuite.id))) {
        suiteView = new SuiteResultView(this.suite.parentSuite);
        suiteView.attach();
        suiteViewElement = suiteView.element;
      }
      return suiteViewElement;
    };

    return SuiteResultView;

  })();

  SpecResultView = (function() {
    function SpecResultView(spec1) {
      var description, i, len, ref, result, resultElement, stackTrace, traceElement;
      this.spec = spec1;
      this.element = document.createElement('div');
      this.element.className = 'spec';
      this.element.innerHTML = "<div class='spec-toggle'></div>\n<div outlet='description' class='description'></div>\n<div outlet='specFailures' class='spec-failures'></div>";
      this.description = this.element.querySelector('[outlet="description"]');
      this.specFailures = this.element.querySelector('[outlet="specFailures"]');
      this.element.classList.add("spec-view-" + this.spec.id);
      description = this.spec.description;
      if (description.indexOf('it ') !== 0) {
        description = "it " + description;
      }
      this.description.textContent = description;
      ref = this.spec.results().getItems();
      for (i = 0, len = ref.length; i < len; i++) {
        result = ref[i];
        if (!(!result.passed())) {
          continue;
        }
        stackTrace = formatStackTrace(this.spec, result.message, result.trace.stack);
        resultElement = document.createElement('div');
        resultElement.className = 'result-message fail';
        resultElement.textContent = result.message;
        this.specFailures.appendChild(resultElement);
        if (stackTrace) {
          traceElement = document.createElement('pre');
          traceElement.className = 'stack-trace padded';
          traceElement.textContent = stackTrace;
          this.specFailures.appendChild(traceElement);
        }
      }
    }

    SpecResultView.prototype.attach = function() {
      return this.parentSuiteView().appendChild(this.element);
    };

    SpecResultView.prototype.parentSuiteView = function() {
      var suiteView, suiteViewElement;
      if (!(suiteViewElement = document.querySelector("#suite-view-" + this.spec.suite.id))) {
        suiteView = new SuiteResultView(this.spec.suite);
        suiteView.attach();
        suiteViewElement = suiteView.element;
      }
      return suiteViewElement;
    };

    return SpecResultView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
