"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _compositeGitStrategy = _interopRequireDefault(require("../composite-git-strategy"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

/**
 * Locate the nearest git working directory above a given starting point, caching results.
 */
class WorkdirCache {
  constructor(maxSize = 1000) {
    this.maxSize = maxSize;
    this.known = new Map();
  }

  async find(startPath) {
    const cached = this.known.get(startPath);

    if (cached !== undefined) {
      return cached;
    }

    const workDir = await this.revParse(startPath);

    if (this.known.size >= this.maxSize) {
      this.known.clear();
    }

    this.known.set(startPath, workDir);
    return workDir;
  }

  invalidate() {
    this.known.clear();
  }

  async revParse(startPath) {
    try {
      const startDir = (await _fsExtra["default"].stat(startPath)).isDirectory() ? startPath : _path["default"].dirname(startPath); // Within a git worktree, return a non-empty string containing the path to the worktree root.
      // Within a gitdir or outside of a worktree, return an empty string.
      // Throw if startDir does not exist.

      const topLevel = await _compositeGitStrategy["default"].create(startDir).exec(['rev-parse', '--show-toplevel']);

      if (/\S/.test(topLevel)) {
        return (0, _helpers.toNativePathSep)(topLevel.trim());
      } // Within a gitdir, return the absolute path to the gitdir.
      // Outside of a gitdir or worktree, throw.


      const gitDir = await _compositeGitStrategy["default"].create(startDir).exec(['rev-parse', '--absolute-git-dir']);
      return this.revParse(_path["default"].resolve(gitDir, '..'));
    } catch (e) {
      /* istanbul ignore if */
      if (atom.config.get('github.reportCannotLocateWorkspaceError')) {
        // eslint-disable-next-line no-console
        console.error(`Unable to locate git workspace root for ${startPath}. Expected if ${startPath} is not in a git repository.`, e);
      }

      return null;
    }
  }

}

exports["default"] = WorkdirCache;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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