(function() {
  var SpellCheckerManager, manager;

  SpellCheckerManager = (function() {
    function SpellCheckerManager() {}

    SpellCheckerManager.prototype.checkers = [];

    SpellCheckerManager.prototype.checkerPaths = [];

    SpellCheckerManager.prototype.locales = [];

    SpellCheckerManager.prototype.localePaths = [];

    SpellCheckerManager.prototype.useLocales = false;

    SpellCheckerManager.prototype.localeCheckers = null;

    SpellCheckerManager.prototype.knownWords = [];

    SpellCheckerManager.prototype.addKnownWords = false;

    SpellCheckerManager.prototype.knownWordsChecker = null;

    SpellCheckerManager.prototype.setGlobalArgs = function(data) {
      var _, checker, checkers, i, len, ref, removeKnownWordsChecker, removeLocaleCheckers;
      _ = require("underscore-plus");
      removeLocaleCheckers = false;
      removeKnownWordsChecker = false;
      if (!_.isEqual(this.locales, data.locales)) {
        if (!this.localeCheckers || ((ref = data.locales) != null ? ref.length : void 0) !== 0) {
          this.locales = data.locales;
          removeLocaleCheckers = true;
        }
      }
      if (!_.isEqual(this.localePaths, data.localePaths)) {
        this.localePaths = data.localePaths;
        removeLocaleCheckers = true;
      }
      if (this.useLocales !== data.useLocales) {
        this.useLocales = data.useLocales;
        removeLocaleCheckers = true;
      }
      if (!_.isEqual(this.knownWords, data.knownWords)) {
        this.knownWords = data.knownWords;
        removeKnownWordsChecker = true;
      }
      if (this.addKnownWords !== data.addKnownWords) {
        this.addKnownWords = data.addKnownWords;
        removeKnownWordsChecker = true;
      }
      if (removeLocaleCheckers && this.localeCheckers) {
        checkers = this.localeCheckers;
        for (i = 0, len = checkers.length; i < len; i++) {
          checker = checkers[i];
          this.removeSpellChecker(checker);
        }
        this.localeCheckers = null;
      }
      if (removeKnownWordsChecker && this.knownWordsChecker) {
        this.removeSpellChecker(this.knownWordsChecker);
        return this.knownWordsChecker = null;
      }
    };

    SpellCheckerManager.prototype.addCheckerPath = function(checkerPath) {
      var checker;
      checker = require(checkerPath);
      if (checker["default"]) {
        checker = new checker["default"]();
      }
      return this.addPluginChecker(checker);
    };

    SpellCheckerManager.prototype.addPluginChecker = function(checker) {
      return this.addSpellChecker(checker);
    };

    SpellCheckerManager.prototype.addSpellChecker = function(checker) {
      return this.checkers.push(checker);
    };

    SpellCheckerManager.prototype.removeSpellChecker = function(spellChecker) {
      return this.checkers = this.checkers.filter(function(plugin) {
        return plugin !== spellChecker;
      });
    };

    SpellCheckerManager.prototype.check = function(args, text) {
      var checker, correct, i, incorrects, len, multirange, promises, ref;
      this.init();
      multirange = require('multi-integer-range');
      correct = new multirange.MultiRange([]);
      incorrects = [];
      promises = [];
      ref = this.checkers;
      for (i = 0, len = ref.length; i < len; i++) {
        checker = ref[i];
        if (!checker.isEnabled() || !checker.providesSpelling(args)) {
          continue;
        }
        promises.push(Promise.resolve(checker.check(args, text)));
      }
      return Promise.all(promises).then((function(_this) {
        return function(allResults) {
          var incorrect, intersection, invertedCorrect, j, k, l, len1, len2, len3, len4, len5, lineBeginIndex, lineEndIndex, lineRange, m, misspellings, n, newIncorrect, range, rangeIndex, rangeRange, ref1, ref2, ref3, removeRange, results, row;
          for (j = 0, len1 = allResults.length; j < len1; j++) {
            results = allResults[j];
            if (results.invertIncorrectAsCorrect && results.incorrect) {
              invertedCorrect = new multirange.MultiRange([[0, text.length]]);
              removeRange = new multirange.MultiRange([]);
              ref1 = results.incorrect;
              for (k = 0, len2 = ref1.length; k < len2; k++) {
                range = ref1[k];
                removeRange.appendRange(range.start, range.end);
              }
              invertedCorrect.subtract(removeRange);
              correct.append(invertedCorrect);
            } else if (results.correct) {
              ref2 = results.correct;
              for (l = 0, len3 = ref2.length; l < len3; l++) {
                range = ref2[l];
                correct.appendRange(range.start, range.end);
              }
            }
            if (results.incorrect) {
              newIncorrect = new multirange.MultiRange([]);
              incorrects.push(newIncorrect);
              ref3 = results.incorrect;
              for (m = 0, len4 = ref3.length; m < len4; m++) {
                range = ref3[m];
                newIncorrect.appendRange(range.start, range.end);
              }
            }
          }
          if (incorrects.length === 0) {
            return {
              misspellings: []
            };
          }
          intersection = null;
          for (n = 0, len5 = incorrects.length; n < len5; n++) {
            incorrect = incorrects[n];
            if (intersection === null) {
              intersection = incorrect;
            } else {
              intersection.append(incorrect);
            }
          }
          if (intersection.length === 0) {
            return {
              misspellings: []
            };
          }
          if (correct.ranges.length > 0) {
            intersection.subtract(correct);
          }
          row = 0;
          rangeIndex = 0;
          lineBeginIndex = 0;
          misspellings = [];
          while (lineBeginIndex < text.length && rangeIndex < intersection.ranges.length) {
            lineEndIndex = text.indexOf('\n', lineBeginIndex);
            if (lineEndIndex === -1) {
              lineEndIndex = 2e308;
            }
            while (true) {
              range = intersection.ranges[rangeIndex];
              if (range && range[0] < lineEndIndex) {
                lineRange = new multirange.MultiRange([]).appendRange(lineBeginIndex, lineEndIndex);
                rangeRange = new multirange.MultiRange([]).appendRange(range[0], range[1]);
                lineRange.intersect(rangeRange);
                _this.addMisspellings(misspellings, row, lineRange.ranges[0], lineBeginIndex, text);
                if (lineEndIndex >= range[1]) {
                  rangeIndex++;
                } else {
                  break;
                }
              } else {
                break;
              }
            }
            lineBeginIndex = lineEndIndex + 1;
            row++;
          }
          return {
            misspellings: misspellings
          };
        };
      })(this));
    };

    SpellCheckerManager.prototype.suggest = function(args, word) {
      var checker, i, index, j, k, key, keys, l, len, len1, len2, len3, len4, m, priority, ref, ref1, results, s, seen, suggestion, suggestions, target, targets, that;
      this.init();
      suggestions = [];
      ref = this.checkers;
      for (i = 0, len = ref.length; i < len; i++) {
        checker = ref[i];
        if (!checker.isEnabled() || !checker.providesSuggestions(args)) {
          continue;
        }
        index = 0;
        priority = checker.getPriority();
        ref1 = checker.suggest(args, word);
        for (j = 0, len1 = ref1.length; j < len1; j++) {
          suggestion = ref1[j];
          suggestions.push({
            isSuggestion: true,
            priority: priority,
            index: index++,
            suggestion: suggestion,
            label: suggestion
          });
        }
      }
      keys = Object.keys(suggestions).sort(function(key1, key2) {
        var value1, value2, weight1, weight2;
        value1 = suggestions[key1];
        value2 = suggestions[key2];
        weight1 = value1.priority + value1.index;
        weight2 = value2.priority + value2.index;
        if (weight1 !== weight2) {
          return weight1 - weight2;
        }
        return value1.suggestion.localeCompare(value2.suggestion);
      });
      results = [];
      seen = [];
      for (k = 0, len2 = keys.length; k < len2; k++) {
        key = keys[k];
        s = suggestions[key];
        if (seen.hasOwnProperty(s.suggestion)) {
          continue;
        }
        results.push(s);
        seen[s.suggestion] = 1;
      }
      that = this;
      keys = Object.keys(this.checkers).sort(function(key1, key2) {
        var value1, value2;
        value1 = that.checkers[key1];
        value2 = that.checkers[key2];
        return value1.getPriority() - value2.getPriority();
      });
      for (l = 0, len3 = keys.length; l < len3; l++) {
        key = keys[l];
        checker = this.checkers[key];
        if (!checker.isEnabled() || !checker.providesAdding(args)) {
          continue;
        }
        targets = checker.getAddingTargets(args);
        for (m = 0, len4 = targets.length; m < len4; m++) {
          target = targets[m];
          target.plugin = checker;
          target.word = word;
          target.isSuggestion = false;
          results.push(target);
        }
      }
      return results;
    };

    SpellCheckerManager.prototype.addMisspellings = function(misspellings, row, range, lineBeginIndex, text) {
      var i, len, markBeginIndex, markEndIndex, part, parts, substring, substringIndex;
      substring = text.substring(range[0], range[1]);
      if (/\s+/.test(substring)) {
        parts = substring.split(/(\s+)/);
        substringIndex = 0;
        for (i = 0, len = parts.length; i < len; i++) {
          part = parts[i];
          if (!/\s+/.test(part)) {
            markBeginIndex = range[0] - lineBeginIndex + substringIndex;
            markEndIndex = markBeginIndex + part.length;
            misspellings.push([[row, markBeginIndex], [row, markEndIndex]]);
          }
          substringIndex += part.length;
        }
        return;
      }
      return misspellings.push([[row, range[0] - lineBeginIndex], [row, range[1] - lineBeginIndex]]);
    };

    SpellCheckerManager.prototype.init = function() {
      var KnownWordsChecker, SystemChecker, checker, defaultLocale, i, len, locale, ref, separatorChar;
      if (this.localeCheckers === null) {
        this.localeCheckers = [];
        if (this.useLocales) {
          if (!this.locales.length) {
            defaultLocale = process.env.LANG;
            if (defaultLocale) {
              this.locales = [defaultLocale.split('.')[0]];
            }
          }
          if (!this.locales.length) {
            defaultLocale = navigator.language;
            if (defaultLocale && defaultLocale.length === 5) {
              separatorChar = defaultLocale.charAt(2);
              if (separatorChar === '_' || separatorChar === '-') {
                this.locales = [defaultLocale];
              }
            }
          }
          if (!this.locales.length) {
            this.locales = ['en_US'];
          }
          SystemChecker = require("./system-checker");
          ref = this.locales;
          for (i = 0, len = ref.length; i < len; i++) {
            locale = ref[i];
            checker = new SystemChecker(locale, this.localePaths);
            this.addSpellChecker(checker);
            this.localeCheckers.push(checker);
          }
        }
      }
      if (this.knownWordsChecker === null) {
        KnownWordsChecker = require('./known-words-checker');
        this.knownWordsChecker = new KnownWordsChecker(this.knownWords);
        this.knownWordsChecker.enableAdd = this.addKnownWords;
        return this.addSpellChecker(this.knownWordsChecker);
      }
    };

    SpellCheckerManager.prototype.deactivate = function() {
      this.checkers = [];
      this.locales = [];
      this.localePaths = [];
      this.useLocales = false;
      this.localeCheckers = null;
      this.knownWords = [];
      this.addKnownWords = false;
      return this.knownWordsChecker = null;
    };

    SpellCheckerManager.prototype.reloadLocales = function() {
      var i, len, localeChecker, ref;
      if (this.localeCheckers) {
        ref = this.localeCheckers;
        for (i = 0, len = ref.length; i < len; i++) {
          localeChecker = ref[i];
          this.removeSpellChecker(localeChecker);
        }
        return this.localeCheckers = null;
      }
    };

    SpellCheckerManager.prototype.reloadKnownWords = function() {
      if (this.knownWordsChecker) {
        this.removeSpellChecker(this.knownWordsChecker);
        return this.knownWordsChecker = null;
      }
    };

    return SpellCheckerManager;

  })();

  manager = new SpellCheckerManager;

  module.exports = manager;

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
