"use strict";

/* eslint comma-dangle: ["error", {
    "arrays": "never",
    "objects": "never",
    "imports": "never",
    "exports": "never",
    "functions": "never"
  }] */
const {
  execFile
} = require('child_process');

const fs = require('fs');

if (typeof atom === 'undefined') {
  global.atom = {
    inSpecMode() {
      return !!process.env.ATOM_GITHUB_SPEC_MODE;
    },

    inDevMode() {
      return false;
    }

  };
} // No token available in your OS keychain.


const UNAUTHENTICATED = Symbol('UNAUTHENTICATED'); // The token in your keychain isn't granted all of the required OAuth scopes.

const INSUFFICIENT = Symbol('INSUFFICIENT'); // The token in your keychain is not accepted by GitHub.

const UNAUTHORIZED = Symbol('UNAUTHORIZED');

class KeytarStrategy {
  static get keytar() {
    return require('keytar');
  }

  static async isValid() {
    // Allow for disabling Keytar on problematic CI environments
    if (process.env.ATOM_GITHUB_DISABLE_KEYTAR) {
      return false;
    }

    const keytar = this.keytar;

    try {
      const rand = Math.floor(Math.random() * 10e20).toString(16);
      await keytar.setPassword('atom-test-service', rand, rand);
      const pass = await keytar.getPassword('atom-test-service', rand);
      const success = pass === rand;
      keytar.deletePassword('atom-test-service', rand);
      return success;
    } catch (err) {
      return false;
    }
  }

  async getPassword(service, account) {
    const password = await this.constructor.keytar.getPassword(service, account);
    return password !== null ? password : UNAUTHENTICATED;
  }

  replacePassword(service, account, password) {
    return this.constructor.keytar.setPassword(service, account, password);
  }

  deletePassword(service, account) {
    return this.constructor.keytar.deletePassword(service, account);
  }

}

class SecurityBinaryStrategy {
  static isValid() {
    return process.platform === 'darwin';
  }

  async getPassword(service, account) {
    try {
      const password = await this.exec(['find-generic-password', '-s', service, '-a', account, '-w']);
      return password.trim() || UNAUTHENTICATED;
    } catch (err) {
      return UNAUTHENTICATED;
    }
  }

  replacePassword(service, account, newPassword) {
    return this.exec(['add-generic-password', '-s', service, '-a', account, '-w', newPassword, '-U']);
  }

  deletePassword(service, account) {
    return this.exec(['delete-generic-password', '-s', service, '-a', account]);
  }

  exec(securityArgs, {
    binary
  } = {
    binary: 'security'
  }) {
    return new Promise((resolve, reject) => {
      execFile(binary, securityArgs, (error, stdout) => {
        if (error) {
          return reject(error);
        }

        return resolve(stdout);
      });
    });
  }

}

class InMemoryStrategy {
  static isValid() {
    return true;
  }

  constructor() {
    if (!atom.inSpecMode()) {
      // eslint-disable-next-line no-console
      console.warn('Using an InMemoryStrategy strategy for storing tokens. ' + 'The tokens will only be stored for the current window.');
    }

    this.passwordsByService = new Map();
  }

  getPassword(service, account) {
    const passwords = this.passwordsByService.get(service) || new Map();
    const password = passwords.get(account);
    return password || UNAUTHENTICATED;
  }

  replacePassword(service, account, newPassword) {
    const passwords = this.passwordsByService.get(service) || new Map();
    passwords.set(account, newPassword);
    this.passwordsByService.set(service, passwords);
  }

  deletePassword(service, account) {
    const passwords = this.passwordsByService.get(service);

    if (passwords) {
      passwords["delete"](account);
    }
  }

}

class FileStrategy {
  static isValid() {
    if (!atom.inSpecMode() && !atom.inDevMode()) {
      return false;
    }

    return Boolean(process.env.ATOM_GITHUB_KEYTAR_FILE);
  }

  constructor() {
    this.filePath = process.env.ATOM_GITHUB_KEYTAR_FILE;

    if (!atom.inSpecMode()) {
      // eslint-disable-next-line no-console
      console.warn('Using a FileStrategy strategy for storing tokens. ' + 'The tokens will be stored %cin the clear%c in a file at %s. ' + "You probably shouldn't use real credentials while this strategy is in use. " + 'Unset ATOM_GITHUB_KEYTAR_FILE_STRATEGY to disable it.', 'color: red; font-weight: bold; font-style: italic', 'color: black; font-weight: normal; font-style: normal', this.filePath);
    }
  }

  async getPassword(service, account) {
    const payload = await this.load();
    const forService = payload[service];

    if (forService === undefined) {
      return UNAUTHENTICATED;
    }

    const passwd = forService[account];

    if (passwd === undefined) {
      return UNAUTHENTICATED;
    }

    return passwd;
  }

  replacePassword(service, account, password) {
    return this.modify(payload => {
      let forService = payload[service];

      if (forService === undefined) {
        forService = {};
        payload[service] = forService;
      }

      forService[account] = password;
    });
  }

  deletePassword(service, account) {
    return this.modify(payload => {
      const forService = payload[service];

      if (forService === undefined) {
        return;
      }

      delete forService[account];

      if (Object.keys(forService).length === 0) {
        delete payload[service];
      }
    });
  }

  load() {
    return new Promise((resolve, reject) => {
      fs.readFile(this.filePath, 'utf8', (err, content) => {
        if (err && err.code === 'ENOENT') {
          return resolve({});
        }

        if (err) {
          return reject(err);
        }

        return resolve(JSON.parse(content));
      });
    });
  }

  save(payload) {
    return new Promise((resolve, reject) => {
      fs.writeFile(this.filePath, JSON.stringify(payload), 'utf8', err => {
        if (err) {
          reject(err);
        } else {
          resolve();
        }
      });
    });
  }

  async modify(callback) {
    const payload = await this.load();
    callback(payload);
    await this.save(payload);
  }

}

const strategies = [FileStrategy, KeytarStrategy, SecurityBinaryStrategy, InMemoryStrategy];
let ValidStrategy = null;

async function createStrategy() {
  if (ValidStrategy) {
    return new ValidStrategy();
  }

  for (let i = 0; i < strategies.length; i++) {
    const strat = strategies[i];
    const isValid = await strat.isValid();

    if (isValid) {
      ValidStrategy = strat;
      break;
    }
  }

  if (!ValidStrategy) {
    throw new Error('None of the listed keytar strategies returned true for `isValid`');
  }

  return new ValidStrategy();
}

module.exports = {
  UNAUTHENTICATED,
  INSUFFICIENT,
  UNAUTHORIZED,
  KeytarStrategy,
  SecurityBinaryStrategy,
  InMemoryStrategy,
  FileStrategy,
  createStrategy
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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