/**
 * @file The `Ring` class is for types that support addition, multiplication, and subtraction operations.
 *
 * Instances must satisfy the following law in addition to the `Semiring` laws:
 *
 * - Additive inverse: `a - a = (zero - a) + a = zero`
 *
 * Adapted from https://github.com/purescript/purescript-prelude/blob/master/src/Data/Ring.purs
 */
import { getFunctionSemiring } from './Semiring';
/**
 * @since 1.0.0
 */
export const getFunctionRing = (ring) => {
    return Object.assign({}, getFunctionSemiring(ring), { sub: (f, g) => x => ring.sub(f(x), g(x)) });
};
/**
 * `negate x` can be used as a shorthand for `zero - x`
 *
 * @since 1.0.0
 */
export const negate = (ring) => (a) => {
    return ring.sub(ring.zero, a);
};
/**
 * Given a tuple of `Ring`s returns a `Ring` for the tuple
 *
 * @example
 * import { getTupleRing } from 'fp-ts/lib/Ring'
 * import { fieldNumber } from 'fp-ts/lib/Field'
 *
 * const R = getTupleRing(fieldNumber, fieldNumber, fieldNumber)
 * assert.deepStrictEqual(R.add([1, 2, 3], [4, 5, 6]), [5, 7, 9])
 * assert.deepStrictEqual(R.mul([1, 2, 3], [4, 5, 6]), [4, 10, 18])
 * assert.deepStrictEqual(R.one, [1, 1, 1])
 * assert.deepStrictEqual(R.sub([1, 2, 3], [4, 5, 6]), [-3, -3, -3])
 * assert.deepStrictEqual(R.zero, [0, 0, 0])
 *
 * @since 1.14.3
 */
export const getTupleRing = (...rings) => {
    return {
        add: (x, y) => rings.map((R, i) => R.add(x[i], y[i])),
        zero: rings.map(R => R.zero),
        mul: (x, y) => rings.map((R, i) => R.mul(x[i], y[i])),
        one: rings.map(R => R.one),
        sub: (x, y) => rings.map((R, i) => R.sub(x[i], y[i]))
    };
};
/**
 * Use `getTupleRing` instead
 * @since 1.0.0
 * @deprecated
 */
export const getProductRing = (RA, RB) => {
    return getTupleRing(RA, RB);
};
