"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
class UnknownActionException extends core_1.BaseException {
    constructor(action) { super(`Unknown action: "${action.kind}".`); }
}
exports.UnknownActionException = UnknownActionException;
let _id = 1;
class ActionList {
    constructor() {
        this._actions = [];
    }
    _action(action) {
        this._actions.push(Object.assign({
            id: _id++,
            parent: this._actions[this._actions.length - 1] || 0,
        }, action));
    }
    create(path, content) {
        this._action({ kind: 'c', path, content });
    }
    overwrite(path, content) {
        this._action({ kind: 'o', path, content });
    }
    rename(path, to) {
        this._action({ kind: 'r', path, to });
    }
    delete(path) {
        this._action({ kind: 'd', path });
    }
    optimize() {
        const toCreate = new Map();
        const toRename = new Map();
        const toOverwrite = new Map();
        const toDelete = new Set();
        for (const action of this._actions) {
            switch (action.kind) {
                case 'c':
                    toCreate.set(action.path, action.content);
                    break;
                case 'o':
                    if (toCreate.has(action.path)) {
                        toCreate.set(action.path, action.content);
                    }
                    else {
                        toOverwrite.set(action.path, action.content);
                    }
                    break;
                case 'd':
                    toDelete.add(action.path);
                    break;
                case 'r':
                    const maybeCreate = toCreate.get(action.path);
                    const maybeOverwrite = toOverwrite.get(action.path);
                    if (maybeCreate) {
                        toCreate.delete(action.path);
                        toCreate.set(action.to, maybeCreate);
                    }
                    if (maybeOverwrite) {
                        toOverwrite.delete(action.path);
                        toOverwrite.set(action.to, maybeOverwrite);
                    }
                    let maybeRename = undefined;
                    for (const [from, to] of toRename.entries()) {
                        if (to == action.path) {
                            maybeRename = from;
                            break;
                        }
                    }
                    if (maybeRename) {
                        toRename.set(maybeRename, action.to);
                    }
                    if (!maybeCreate && !maybeOverwrite && !maybeRename) {
                        toRename.set(action.path, action.to);
                    }
                    break;
            }
        }
        this._actions = [];
        toDelete.forEach(x => {
            this.delete(x);
        });
        toRename.forEach((to, from) => {
            this.rename(from, to);
        });
        toCreate.forEach((content, path) => {
            this.create(path, content);
        });
        toOverwrite.forEach((content, path) => {
            this.overwrite(path, content);
        });
    }
    push(action) { this._actions.push(action); }
    get(i) { return this._actions[i]; }
    has(action) {
        for (let i = 0; i < this._actions.length; i++) {
            const a = this._actions[i];
            if (a.id == action.id) {
                return true;
            }
            if (a.id > action.id) {
                return false;
            }
        }
        return false;
    }
    find(predicate) {
        return this._actions.find(predicate) || null;
    }
    forEach(fn, thisArg) {
        this._actions.forEach(fn, thisArg);
    }
    get length() { return this._actions.length; }
    [Symbol.iterator]() { return this._actions[Symbol.iterator](); }
}
exports.ActionList = ActionList;
function isContentAction(action) {
    return action.kind == 'c' || action.kind == 'o';
}
exports.isContentAction = isContentAction;
function isAction(action) {
    const kind = action && action.kind;
    return action !== null
        && typeof action.id == 'number'
        && typeof action.path == 'string'
        && (kind == 'c' || kind == 'o' || kind == 'r' || kind == 'd');
}
exports.isAction = isAction;
//# sourceMappingURL=data:application/json;base64,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