"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _path = _interopRequireDefault(require("path"));

var _electron = require("electron");

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _eventKit = require("event-kit");

var _statusBar = _interopRequireDefault(require("../atom/status-bar"));

var _panel = _interopRequireDefault(require("../atom/panel"));

var _paneItem = _interopRequireDefault(require("../atom/pane-item"));

var _cloneDialog = _interopRequireDefault(require("../views/clone-dialog"));

var _openIssueishDialog = _interopRequireDefault(require("../views/open-issueish-dialog"));

var _openCommitDialog = _interopRequireDefault(require("../views/open-commit-dialog"));

var _initDialog = _interopRequireDefault(require("../views/init-dialog"));

var _credentialDialog = _interopRequireDefault(require("../views/credential-dialog"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _commitDetailItem = _interopRequireDefault(require("../items/commit-detail-item"));

var _commitPreviewItem = _interopRequireDefault(require("../items/commit-preview-item"));

var _gitTabItem = _interopRequireDefault(require("../items/git-tab-item"));

var _githubTabItem = _interopRequireDefault(require("../items/github-tab-item"));

var _reviewsItem = _interopRequireDefault(require("../items/reviews-item"));

var _commentDecorationsContainer = _interopRequireDefault(require("../containers/comment-decorations-container"));

var _statusBarTileController = _interopRequireDefault(require("./status-bar-tile-controller"));

var _repositoryConflictController = _interopRequireDefault(require("./repository-conflict-controller"));

var _gitCacheView = _interopRequireDefault(require("../views/git-cache-view"));

var _gitTimingsView = _interopRequireDefault(require("../views/git-timings-view"));

var _conflict = _interopRequireDefault(require("../models/conflicts/conflict"));

var _switchboard = _interopRequireDefault(require("../switchboard"));

var _propTypes2 = require("../prop-types");

var _helpers = require("../helpers");

var _gitShellOutStrategy = require("../git-shell-out-strategy");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class RootController extends _react["default"].Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "toggleCommitPreviewItem", () => {
      const workdir = this.props.repository.getWorkingDirectoryPath();
      return this.props.workspace.toggle(_commitPreviewItem["default"].buildURI(workdir));
    });

    _defineProperty(this, "showOpenCommitDialog", () => {
      this.setState({
        openCommitDialogActive: true
      });
    });

    _defineProperty(this, "isValidCommit", async ref => {
      try {
        await this.props.repository.getCommit(ref);
        return true;
      } catch (error) {
        if (error instanceof _gitShellOutStrategy.GitError && error.code === 128) {
          return false;
        } else {
          throw error;
        }
      }
    });

    _defineProperty(this, "acceptOpenCommit", ({
      ref
    }) => {
      const workdir = this.props.repository.getWorkingDirectoryPath();

      const uri = _commitDetailItem["default"].buildURI(workdir, ref);

      this.setState({
        openCommitDialogActive: false
      });
      this.props.workspace.open(uri).then(() => {
        (0, _reporterProxy.addEvent)('open-commit-in-pane', {
          "package": 'github',
          from: _openCommitDialog["default"].name
        });
      });
    });

    _defineProperty(this, "cancelOpenCommit", () => {
      this.setState({
        openCommitDialogActive: false
      });
    });

    _defineProperty(this, "surfaceFromFileAtPath", (filePath, stagingStatus) => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectStagingItem(filePath, stagingStatus);
    });

    _defineProperty(this, "surfaceToCommitPreviewButton", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectCommitPreviewButton();
    });

    _defineProperty(this, "surfaceToRecentCommit", () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectRecentCommit();
    });

    _defineProperty(this, "reportRelayError", (friendlyMessage, err) => {
      const opts = {
        dismissable: true
      };

      if (err.network) {
        // Offline
        opts.icon = 'alignment-unalign';
        opts.description = "It looks like you're offline right now.";
      } else if (err.responseText) {
        // Transient error like a 500 from the API
        opts.description = 'The GitHub API reported a problem.';
        opts.detail = err.responseText;
      } else if (err.errors) {
        // GraphQL errors
        opts.detail = err.errors.map(e => e.message).join('\n');
      } else {
        opts.detail = err.stack;
      }

      this.props.notificationManager.addError(friendlyMessage, opts);
    });

    (0, _helpers.autobind)(this, 'installReactDevTools', 'clearGithubToken', 'initializeRepo', 'showOpenIssueishDialog', 'showWaterfallDiagnostics', 'showCacheDiagnostics', 'acceptClone', 'cancelClone', 'acceptInit', 'cancelInit', 'acceptOpenIssueish', 'cancelOpenIssueish', 'destroyFilePatchPaneItems', 'destroyEmptyFilePatchPaneItems', 'openCloneDialog', 'quietlySelectItem', 'viewUnstagedChangesForCurrentFile', 'viewStagedChangesForCurrentFile', 'openFiles', 'getUnsavedFiles', 'ensureNoUnsavedFiles', 'discardWorkDirChangesForPaths', 'discardLines', 'undoLastDiscard', 'refreshResolutionProgress');
    this.state = {
      cloneDialogActive: false,
      cloneDialogInProgress: false,
      initDialogActive: false,
      initDialogPath: null,
      initDialogResolve: null,
      credentialDialogQuery: null
    };
    this.gitTabTracker = new TabTracker('git', {
      uri: _gitTabItem["default"].buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.githubTabTracker = new TabTracker('github', {
      uri: _githubTabItem["default"].buildURI(),
      getWorkspace: () => this.props.workspace
    });
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(this.gitTabTracker.ensureVisible));
    this.props.commandRegistry.onDidDispatch(event => {
      if (event.type && event.type.startsWith('github:') && event.detail && event.detail[0] && event.detail[0].contextCommand) {
        (0, _reporterProxy.addEvent)('context-menu-action', {
          "package": 'github',
          command: event.type
        });
      }
    });
  }

  componentDidMount() {
    this.openTabs();
  }

  render() {
    return _react["default"].createElement(_react.Fragment, null, this.renderCommands(), this.renderStatusBarTile(), this.renderPaneItems(), this.renderDialogs(), this.renderConflictResolver(), this.renderCommentDecorations());
  }

  renderCommands() {
    const devMode = global.atom && global.atom.inDevMode();
    return _react["default"].createElement(_commands["default"], {
      registry: this.props.commandRegistry,
      target: "atom-workspace"
    }, devMode && _react["default"].createElement(_commands.Command, {
      command: "github:install-react-dev-tools",
      callback: this.installReactDevTools
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-commit-preview",
      callback: this.toggleCommitPreviewItem
    }), _react["default"].createElement(_commands.Command, {
      command: "github:logout",
      callback: this.clearGithubToken
    }), _react["default"].createElement(_commands.Command, {
      command: "github:show-waterfall-diagnostics",
      callback: this.showWaterfallDiagnostics
    }), _react["default"].createElement(_commands.Command, {
      command: "github:show-cache-diagnostics",
      callback: this.showCacheDiagnostics
    }), _react["default"].createElement(_commands.Command, {
      command: "github:open-issue-or-pull-request",
      callback: this.showOpenIssueishDialog
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-git-tab",
      callback: this.gitTabTracker.toggle
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-git-tab-focus",
      callback: this.gitTabTracker.toggleFocus
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-github-tab",
      callback: this.githubTabTracker.toggle
    }), _react["default"].createElement(_commands.Command, {
      command: "github:toggle-github-tab-focus",
      callback: this.githubTabTracker.toggleFocus
    }), _react["default"].createElement(_commands.Command, {
      command: "github:clone",
      callback: this.openCloneDialog
    }), _react["default"].createElement(_commands.Command, {
      command: "github:open-commit",
      callback: this.showOpenCommitDialog
    }), _react["default"].createElement(_commands.Command, {
      command: "github:view-unstaged-changes-for-current-file",
      callback: this.viewUnstagedChangesForCurrentFile
    }), _react["default"].createElement(_commands.Command, {
      command: "github:view-staged-changes-for-current-file",
      callback: this.viewStagedChangesForCurrentFile
    }), _react["default"].createElement(_commands.Command, {
      command: "github:close-all-diff-views",
      callback: this.destroyFilePatchPaneItems
    }), _react["default"].createElement(_commands.Command, {
      command: "github:close-empty-diff-views",
      callback: this.destroyEmptyFilePatchPaneItems
    }));
  }

  renderStatusBarTile() {
    return _react["default"].createElement(_statusBar["default"], {
      statusBar: this.props.statusBar,
      onConsumeStatusBar: sb => this.onConsumeStatusBar(sb),
      className: "github-StatusBarTileController"
    }, _react["default"].createElement(_statusBarTileController["default"], {
      pipelineManager: this.props.pipelineManager,
      workspace: this.props.workspace,
      repository: this.props.repository,
      commandRegistry: this.props.commandRegistry,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      confirm: this.props.confirm,
      toggleGitTab: this.gitTabTracker.toggle,
      toggleGithubTab: this.githubTabTracker.toggle
    }));
  }

  renderDialogs() {
    return _react["default"].createElement(_react.Fragment, null, this.renderInitDialog(), this.renderCloneDialog(), this.renderCredentialDialog(), this.renderOpenIssueishDialog(), this.renderOpenCommitDialog());
  }

  renderInitDialog() {
    if (!this.state.initDialogActive) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_initDialog["default"], {
      config: this.props.config,
      commandRegistry: this.props.commandRegistry,
      initPath: this.state.initDialogPath,
      didAccept: this.acceptInit,
      didCancel: this.cancelInit
    }));
  }

  renderCloneDialog() {
    if (!this.state.cloneDialogActive) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_cloneDialog["default"], {
      config: this.props.config,
      commandRegistry: this.props.commandRegistry,
      didAccept: this.acceptClone,
      didCancel: this.cancelClone,
      inProgress: this.state.cloneDialogInProgress
    }));
  }

  renderOpenIssueishDialog() {
    if (!this.state.openIssueishDialogActive) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_openIssueishDialog["default"], {
      commandRegistry: this.props.commandRegistry,
      didAccept: this.acceptOpenIssueish,
      didCancel: this.cancelOpenIssueish
    }));
  }

  renderOpenCommitDialog() {
    if (!this.state.openCommitDialogActive) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_openCommitDialog["default"], {
      commandRegistry: this.props.commandRegistry,
      didAccept: this.acceptOpenCommit,
      didCancel: this.cancelOpenCommit,
      isValidEntry: this.isValidCommit
    }));
  }

  renderCredentialDialog() {
    if (this.state.credentialDialogQuery === null) {
      return null;
    }

    return _react["default"].createElement(_panel["default"], {
      workspace: this.props.workspace,
      location: "modal"
    }, _react["default"].createElement(_credentialDialog["default"], _extends({
      commandRegistry: this.props.commandRegistry
    }, this.state.credentialDialogQuery)));
  }

  renderCommentDecorations() {
    if (!this.props.repository) {
      return null;
    }

    return _react["default"].createElement(_commentDecorationsContainer["default"], {
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      localRepository: this.props.repository,
      loginModel: this.props.loginModel,
      reportRelayError: this.reportRelayError
    });
  }

  renderConflictResolver() {
    if (!this.props.repository) {
      return null;
    }

    return _react["default"].createElement(_repositoryConflictController["default"], {
      workspace: this.props.workspace,
      config: this.props.config,
      repository: this.props.repository,
      resolutionProgress: this.props.resolutionProgress,
      refreshResolutionProgress: this.refreshResolutionProgress,
      commandRegistry: this.props.commandRegistry
    });
  }

  renderPaneItems() {
    return _react["default"].createElement(_react.Fragment, null, _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _gitTabItem["default"].uriPattern,
      className: "github-Git-root"
    }, ({
      itemHolder
    }) => _react["default"].createElement(_gitTabItem["default"], {
      ref: itemHolder.setter,
      workspace: this.props.workspace,
      commandRegistry: this.props.commandRegistry,
      notificationManager: this.props.notificationManager,
      tooltips: this.props.tooltips,
      grammars: this.props.grammars,
      project: this.props.project,
      confirm: this.props.confirm,
      config: this.props.config,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      initializeRepo: this.initializeRepo,
      resolutionProgress: this.props.resolutionProgress,
      ensureGitTab: this.gitTabTracker.ensureVisible,
      openFiles: this.openFiles,
      discardWorkDirChangesForPaths: this.discardWorkDirChangesForPaths,
      undoLastDiscard: this.undoLastDiscard,
      refreshResolutionProgress: this.refreshResolutionProgress
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _githubTabItem["default"].uriPattern,
      className: "github-GitHub-root"
    }, ({
      itemHolder
    }) => _react["default"].createElement(_githubTabItem["default"], {
      ref: itemHolder.setter,
      repository: this.props.repository,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _changedFileItem["default"].uriPattern
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_changedFileItem["default"], {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      relPath: _path["default"].join(...params.relPath),
      workingDirectory: params.workingDirectory,
      stagingStatus: params.stagingStatus,
      tooltips: this.props.tooltips,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      workspace: this.props.workspace,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceFileAtPath: this.surfaceFromFileAtPath
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _commitPreviewItem["default"].uriPattern,
      className: "github-CommitPreview-root"
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_commitPreviewItem["default"], {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      discardLines: this.discardLines,
      undoLastDiscard: this.undoLastDiscard,
      surfaceToCommitPreviewButton: this.surfaceToCommitPreviewButton
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _commitDetailItem["default"].uriPattern,
      className: "github-CommitDetail-root"
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_commitDetailItem["default"], {
      ref: itemHolder.setter,
      workdirContextPool: this.props.workdirContextPool,
      workingDirectory: params.workingDirectory,
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      sha: params.sha,
      surfaceCommit: this.surfaceToRecentCommit
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _issueishDetailItem["default"].uriPattern
    }, ({
      itemHolder,
      params,
      deserialized
    }) => _react["default"].createElement(_issueishDetailItem["default"], {
      ref: itemHolder.setter,
      host: params.host,
      owner: params.owner,
      repo: params.repo,
      issueishNumber: parseInt(params.issueishNumber, 10),
      workingDirectory: params.workingDirectory,
      workdirContextPool: this.props.workdirContextPool,
      loginModel: this.props.loginModel,
      initSelectedTab: deserialized.initSelectedTab,
      workspace: this.props.workspace,
      commands: this.props.commandRegistry,
      keymaps: this.props.keymaps,
      tooltips: this.props.tooltips,
      config: this.props.config,
      reportRelayError: this.reportRelayError
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _reviewsItem["default"].uriPattern
    }, ({
      itemHolder,
      params
    }) => _react["default"].createElement(_reviewsItem["default"], {
      ref: itemHolder.setter,
      host: params.host,
      owner: params.owner,
      repo: params.repo,
      number: parseInt(params.number, 10),
      workdir: params.workdir,
      workdirContextPool: this.props.workdirContextPool,
      loginModel: this.props.loginModel,
      workspace: this.props.workspace,
      tooltips: this.props.tooltips,
      config: this.props.config,
      commands: this.props.commandRegistry,
      confirm: this.props.confirm,
      reportRelayError: this.reportRelayError
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _gitTimingsView["default"].uriPattern
    }, ({
      itemHolder
    }) => _react["default"].createElement(_gitTimingsView["default"], {
      ref: itemHolder.setter
    })), _react["default"].createElement(_paneItem["default"], {
      workspace: this.props.workspace,
      uriPattern: _gitCacheView["default"].uriPattern
    }, ({
      itemHolder
    }) => _react["default"].createElement(_gitCacheView["default"], {
      ref: itemHolder.setter,
      repository: this.props.repository
    })));
  }

  async openTabs() {
    if (this.props.startOpen) {
      await Promise.all([this.gitTabTracker.ensureRendered(false), this.githubTabTracker.ensureRendered(false)]);
    }

    if (this.props.startRevealed) {
      const docks = new Set([_gitTabItem["default"].buildURI(), _githubTabItem["default"].buildURI()].map(uri => this.props.workspace.paneContainerForURI(uri)).filter(container => container && typeof container.show === 'function'));

      for (const dock of docks) {
        dock.show();
      }
    }
  }

  async installReactDevTools() {
    // Prevent electron-link from attempting to descend into electron-devtools-installer, which is not available
    // when we're bundled in Atom.
    const devToolsName = 'electron-devtools-installer';

    const devTools = require(devToolsName);

    await Promise.all([this.installExtension(devTools.REACT_DEVELOPER_TOOLS.id), // relay developer tools extension id
    this.installExtension('ncedobpgnmkhcmnnkcimnobpfepidadl')]);
    this.props.notificationManager.addSuccess('🌈 Reload your window to start using the React/Relay dev tools!');
  }

  async installExtension(id) {
    const devToolsName = 'electron-devtools-installer';

    const devTools = require(devToolsName);

    const crossUnzipName = 'cross-unzip';

    const unzip = require(crossUnzipName);

    const url = 'https://clients2.google.com/service/update2/crx?' + `response=redirect&x=id%3D${id}%26uc&prodversion=32`;

    const extensionFolder = _path["default"].resolve(_electron.remote.app.getPath('userData'), `extensions/${id}`);

    const extensionFile = `${extensionFolder}.crx`;
    await _fsExtra["default"].ensureDir(_path["default"].dirname(extensionFile));
    const response = await fetch(url, {
      method: 'GET'
    });
    const body = Buffer.from((await response.arrayBuffer()));
    await _fsExtra["default"].writeFile(extensionFile, body);
    await new Promise((resolve, reject) => {
      unzip(extensionFile, extensionFolder, async err => {
        if (err && !(await _fsExtra["default"].exists(_path["default"].join(extensionFolder, 'manifest.json')))) {
          reject(err);
        }

        resolve();
      });
    });
    await _fsExtra["default"].ensureDir(extensionFolder, 0o755);
    await devTools["default"](id);
  }

  componentWillUnmount() {
    this.subscription.dispose();
  }

  componentDidUpdate() {
    this.subscription.dispose();
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onPullError(() => this.gitTabTracker.ensureVisible()));
  }

  onConsumeStatusBar(statusBar) {
    if (statusBar.disableGitInfoTile) {
      statusBar.disableGitInfoTile();
    }
  }

  clearGithubToken() {
    return this.props.loginModel.removeToken('https://api.github.com');
  }

  initializeRepo(initDialogPath) {
    if (this.state.initDialogActive) {
      return null;
    }

    if (!initDialogPath) {
      initDialogPath = this.props.repository.getWorkingDirectoryPath();
    }

    return new Promise(resolve => {
      this.setState({
        initDialogActive: true,
        initDialogPath,
        initDialogResolve: resolve
      });
    });
  }

  showOpenIssueishDialog() {
    this.setState({
      openIssueishDialogActive: true
    });
  }

  showWaterfallDiagnostics() {
    this.props.workspace.open(_gitTimingsView["default"].buildURI());
  }

  showCacheDiagnostics() {
    this.props.workspace.open(_gitCacheView["default"].buildURI());
  }

  async acceptClone(remoteUrl, projectPath) {
    this.setState({
      cloneDialogInProgress: true
    });

    try {
      await this.props.cloneRepositoryForProjectPath(remoteUrl, projectPath);
      (0, _reporterProxy.addEvent)('clone-repo', {
        "package": 'github'
      });
    } catch (e) {
      this.props.notificationManager.addError(`Unable to clone ${remoteUrl}`, {
        detail: e.stdErr,
        dismissable: true
      });
    } finally {
      this.setState({
        cloneDialogInProgress: false,
        cloneDialogActive: false
      });
    }
  }

  cancelClone() {
    this.setState({
      cloneDialogActive: false
    });
  }

  async acceptInit(projectPath) {
    try {
      await this.props.createRepositoryForProjectPath(projectPath);

      if (this.state.initDialogResolve) {
        this.state.initDialogResolve(projectPath);
      }
    } catch (e) {
      this.props.notificationManager.addError(`Unable to initialize git repository in ${projectPath}`, {
        detail: e.stdErr,
        dismissable: true
      });
    } finally {
      this.setState({
        initDialogActive: false,
        initDialogPath: null,
        initDialogResolve: null
      });
    }
  }

  cancelInit() {
    if (this.state.initDialogResolve) {
      this.state.initDialogResolve(false);
    }

    this.setState({
      initDialogActive: false,
      initDialogPath: null,
      initDialogResolve: null
    });
  }

  acceptOpenIssueish({
    repoOwner,
    repoName,
    issueishNumber
  }) {
    const uri = _issueishDetailItem["default"].buildURI({
      host: 'github.com',
      owner: repoOwner,
      repo: repoName,
      number: issueishNumber
    });

    this.setState({
      openIssueishDialogActive: false
    });
    this.props.workspace.open(uri).then(() => {
      (0, _reporterProxy.addEvent)('open-issueish-in-pane', {
        "package": 'github',
        from: 'dialog'
      });
    });
  }

  cancelOpenIssueish() {
    this.setState({
      openIssueishDialogActive: false
    });
  }

  destroyFilePatchPaneItems() {
    (0, _helpers.destroyFilePatchPaneItems)({
      onlyStaged: false
    }, this.props.workspace);
  }

  destroyEmptyFilePatchPaneItems() {
    (0, _helpers.destroyEmptyFilePatchPaneItems)(this.props.workspace);
  }

  openCloneDialog() {
    this.setState({
      cloneDialogActive: true
    });
  }

  quietlySelectItem(filePath, stagingStatus) {
    const gitTab = this.gitTabTracker.getComponent();
    return gitTab && gitTab.quietlySelectItem(filePath, stagingStatus);
  }

  async viewChangesForCurrentFile(stagingStatus) {
    const editor = this.props.workspace.getActiveTextEditor();

    if (!editor.getPath()) {
      return;
    }

    const absFilePath = await _fsExtra["default"].realpath(editor.getPath());
    const repoPath = this.props.repository.getWorkingDirectoryPath();

    if (repoPath === null) {
      const [projectPath] = this.props.project.relativizePath(editor.getPath());
      const notification = this.props.notificationManager.addInfo("Hmm, there's nothing to compare this file to", {
        description: 'You can create a Git repository to track changes to the files in your project.',
        dismissable: true,
        buttons: [{
          className: 'btn btn-primary',
          text: 'Create a repository now',
          onDidClick: async () => {
            notification.dismiss();
            const createdPath = await this.initializeRepo(projectPath); // If the user confirmed repository creation for this project path,
            // retry the operation that got them here in the first place

            if (createdPath === projectPath) {
              this.viewChangesForCurrentFile(stagingStatus);
            }
          }
        }]
      });
      return;
    }

    if (absFilePath.startsWith(repoPath)) {
      const filePath = absFilePath.slice(repoPath.length + 1);
      this.quietlySelectItem(filePath, stagingStatus);
      const splitDirection = this.props.config.get('github.viewChangesForCurrentFileDiffPaneSplitDirection');
      const pane = this.props.workspace.getActivePane();

      if (splitDirection === 'right') {
        pane.splitRight();
      } else if (splitDirection === 'down') {
        pane.splitDown();
      }

      const lineNum = editor.getCursorBufferPosition().row + 1;
      const item = await this.props.workspace.open(_changedFileItem["default"].buildURI(filePath, repoPath, stagingStatus), {
        pending: true,
        activatePane: true,
        activateItem: true
      });
      await item.getRealItemPromise();
      await item.getFilePatchLoadedPromise();
      item.goToDiffLine(lineNum);
      item.focus();
    } else {
      throw new Error(`${absFilePath} does not belong to repo ${repoPath}`);
    }
  }

  viewUnstagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('unstaged');
  }

  viewStagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('staged');
  }

  openFiles(filePaths, repository = this.props.repository) {
    return Promise.all(filePaths.map(filePath => {
      const absolutePath = _path["default"].join(repository.getWorkingDirectoryPath(), filePath);

      return this.props.workspace.open(absolutePath, {
        pending: filePaths.length === 1
      });
    }));
  }

  getUnsavedFiles(filePaths, workdirPath) {
    const isModifiedByPath = new Map();
    this.props.workspace.getTextEditors().forEach(editor => {
      isModifiedByPath.set(editor.getPath(), editor.isModified());
    });
    return filePaths.filter(filePath => {
      const absFilePath = _path["default"].join(workdirPath, filePath);

      return isModifiedByPath.get(absFilePath);
    });
  }

  ensureNoUnsavedFiles(filePaths, message, workdirPath = this.props.repository.getWorkingDirectoryPath()) {
    const unsavedFiles = this.getUnsavedFiles(filePaths, workdirPath).map(filePath => `\`${filePath}\``).join('<br>');

    if (unsavedFiles.length) {
      this.props.notificationManager.addError(message, {
        description: `You have unsaved changes in:<br>${unsavedFiles}.`,
        dismissable: true
      });
      return false;
    } else {
      return true;
    }
  }

  async discardWorkDirChangesForPaths(filePaths) {
    const destructiveAction = () => {
      return this.props.repository.discardWorkDirChangesForPaths(filePaths);
    };

    return await this.props.repository.storeBeforeAndAfterBlobs(filePaths, () => this.ensureNoUnsavedFiles(filePaths, 'Cannot discard changes in selected files.'), destructiveAction);
  }

  async discardLines(multiFilePatch, lines, repository = this.props.repository) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    if (multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    const filePath = multiFilePatch.getFilePatches()[0].getPath();

    const destructiveAction = async () => {
      const discardFilePatch = multiFilePatch.getUnstagePatchForLines(lines);
      await repository.applyPatchToWorkdir(discardFilePatch);
    };

    return await repository.storeBeforeAndAfterBlobs([filePath], () => this.ensureNoUnsavedFiles([filePath], 'Cannot discard lines.', repository.getWorkingDirectoryPath()), destructiveAction, filePath);
  }

  getFilePathsForLastDiscard(partialDiscardFilePath = null) {
    let lastSnapshots = this.props.repository.getLastHistorySnapshots(partialDiscardFilePath);

    if (partialDiscardFilePath) {
      lastSnapshots = lastSnapshots ? [lastSnapshots] : [];
    }

    return lastSnapshots.map(snapshot => snapshot.filePath);
  }

  async undoLastDiscard(partialDiscardFilePath = null, repository = this.props.repository) {
    const filePaths = this.getFilePathsForLastDiscard(partialDiscardFilePath);

    try {
      const results = await repository.restoreLastDiscardInTempFiles(() => this.ensureNoUnsavedFiles(filePaths, 'Cannot undo last discard.'), partialDiscardFilePath);

      if (results.length === 0) {
        return;
      }

      await this.proceedOrPromptBasedOnResults(results, partialDiscardFilePath);
    } catch (e) {
      if (e instanceof _gitShellOutStrategy.GitError && e.stdErr.match(/fatal: Not a valid object name/)) {
        this.cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath);
      } else {
        // eslint-disable-next-line no-console
        console.error(e);
      }
    }
  }

  async proceedOrPromptBasedOnResults(results, partialDiscardFilePath = null) {
    const conflicts = results.filter(({
      conflict
    }) => conflict);

    if (conflicts.length === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else {
      await this.promptAboutConflicts(results, conflicts, partialDiscardFilePath);
    }
  }

  async promptAboutConflicts(results, conflicts, partialDiscardFilePath = null) {
    const conflictedFiles = conflicts.map(({
      filePath
    }) => `\t${filePath}`).join('\n');
    const choice = this.props.confirm({
      message: 'Undoing will result in conflicts...',
      detailedMessage: `for the following files:\n${conflictedFiles}\n` + 'Would you like to apply the changes with merge conflict markers, ' + 'or open the text with merge conflict markers in a new file?',
      buttons: ['Merge with conflict markers', 'Open in new file', 'Cancel']
    });

    if (choice === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else if (choice === 1) {
      await this.openConflictsInNewEditors(conflicts.map(({
        resultPath
      }) => resultPath));
    }
  }

  cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath = null) {
    this.props.repository.clearDiscardHistory(partialDiscardFilePath);
    const filePathsStr = filePaths.map(filePath => `\`${filePath}\``).join('<br>');
    this.props.notificationManager.addError('Discard history has expired.', {
      description: `Cannot undo discard for<br>${filePathsStr}<br>Stale discard history has been deleted.`,
      dismissable: true
    });
  }

  async proceedWithLastDiscardUndo(results, partialDiscardFilePath = null) {
    const promises = results.map(async result => {
      const {
        filePath,
        resultPath,
        deleted,
        conflict,
        theirsSha,
        commonBaseSha,
        currentSha
      } = result;

      const absFilePath = _path["default"].join(this.props.repository.getWorkingDirectoryPath(), filePath);

      if (deleted && resultPath === null) {
        await _fsExtra["default"].remove(absFilePath);
      } else {
        await _fsExtra["default"].copy(resultPath, absFilePath);
      }

      if (conflict) {
        await this.props.repository.writeMergeConflictToIndex(filePath, commonBaseSha, currentSha, theirsSha);
      }
    });
    await Promise.all(promises);
    await this.props.repository.popDiscardHistory(partialDiscardFilePath);
  }

  async openConflictsInNewEditors(resultPaths) {
    const editorPromises = resultPaths.map(resultPath => {
      return this.props.workspace.open(resultPath);
    });
    return await Promise.all(editorPromises);
  }

  /*
   * Asynchronously count the conflict markers present in a file specified by full path.
   */
  refreshResolutionProgress(fullPath) {
    const readStream = _fsExtra["default"].createReadStream(fullPath, {
      encoding: 'utf8'
    });

    return new Promise(resolve => {
      _conflict["default"].countFromStream(readStream).then(count => {
        this.props.resolutionProgress.reportMarkerCount(fullPath, count);
      });
    });
  }
  /*
   * Display the credential entry dialog. Return a Promise that will resolve with the provided credentials on accept
   * or reject on cancel.
   */


  promptForCredentials(query) {
    return new Promise((resolve, reject) => {
      this.setState({
        credentialDialogQuery: _objectSpread({}, query, {
          onSubmit: response => this.setState({
            credentialDialogQuery: null
          }, () => resolve(response)),
          onCancel: () => this.setState({
            credentialDialogQuery: null
          }, reject)
        })
      });
    });
  }

}

exports["default"] = RootController;

_defineProperty(RootController, "propTypes", {
  workspace: _propTypes["default"].object.isRequired,
  commandRegistry: _propTypes["default"].object.isRequired,
  deserializers: _propTypes["default"].object.isRequired,
  notificationManager: _propTypes["default"].object.isRequired,
  tooltips: _propTypes["default"].object.isRequired,
  keymaps: _propTypes["default"].object.isRequired,
  grammars: _propTypes["default"].object.isRequired,
  config: _propTypes["default"].object.isRequired,
  project: _propTypes["default"].object.isRequired,
  loginModel: _propTypes["default"].object.isRequired,
  confirm: _propTypes["default"].func.isRequired,
  createRepositoryForProjectPath: _propTypes["default"].func,
  cloneRepositoryForProjectPath: _propTypes["default"].func,
  repository: _propTypes["default"].object.isRequired,
  resolutionProgress: _propTypes["default"].object.isRequired,
  statusBar: _propTypes["default"].object,
  switchboard: _propTypes["default"].instanceOf(_switchboard["default"]),
  startOpen: _propTypes["default"].bool,
  startRevealed: _propTypes["default"].bool,
  pipelineManager: _propTypes["default"].object,
  workdirContextPool: _propTypes2.WorkdirContextPoolPropType.isRequired
});

_defineProperty(RootController, "defaultProps", {
  switchboard: new _switchboard["default"](),
  startOpen: false,
  startRevealed: false
});

class TabTracker {
  constructor(name, {
    getWorkspace,
    uri
  }) {
    (0, _helpers.autobind)(this, 'toggle', 'toggleFocus', 'ensureVisible');
    this.name = name;
    this.getWorkspace = getWorkspace;
    this.uri = uri;
  }

  async toggle() {
    const focusToRestore = document.activeElement;
    let shouldRestoreFocus = false; // Rendered => the dock item is being rendered, whether or not the dock is visible or the item
    //   is visible within its dock.
    // Visible => the item is active and the dock item is active within its dock.

    const wasRendered = this.isRendered();
    const wasVisible = this.isVisible();

    if (!wasRendered || !wasVisible) {
      // Not rendered, or rendered but not an active item in a visible dock.
      await this.reveal();
      shouldRestoreFocus = true;
    } else {
      // Rendered and an active item within a visible dock.
      await this.hide();
      shouldRestoreFocus = false;
    }

    if (shouldRestoreFocus) {
      process.nextTick(() => focusToRestore.focus());
    }
  }

  async toggleFocus() {
    const hadFocus = this.hasFocus();
    await this.ensureVisible();

    if (hadFocus) {
      let workspace = this.getWorkspace();

      if (workspace.getCenter) {
        workspace = workspace.getCenter();
      }

      workspace.getActivePane().activate();
    } else {
      this.focus();
    }
  }

  async ensureVisible() {
    if (!this.isVisible()) {
      await this.reveal();
      return true;
    }

    return false;
  }

  ensureRendered() {
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: false,
      activatePane: false
    });
  }

  reveal() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-open`);
    return this.getWorkspace().open(this.uri, {
      searchAllPanes: true,
      activateItem: true,
      activatePane: true
    });
  }

  hide() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-close`);
    return this.getWorkspace().hide(this.uri);
  }

  focus() {
    this.getComponent().restoreFocus();
  }

  getItem() {
    const pane = this.getWorkspace().paneForURI(this.uri);

    if (!pane) {
      return null;
    }

    const paneItem = pane.itemForURI(this.uri);

    if (!paneItem) {
      return null;
    }

    return paneItem;
  }

  getComponent() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getRealItem !== 'function') {
      return null;
    }

    return paneItem.getRealItem();
  }

  getDOMElement() {
    const paneItem = this.getItem();

    if (!paneItem) {
      return null;
    }

    if (typeof paneItem.getElement !== 'function') {
      return null;
    }

    return paneItem.getElement();
  }

  isRendered() {
    return !!this.getWorkspace().paneForURI(this.uri);
  }

  isVisible() {
    const workspace = this.getWorkspace();
    return workspace.getPaneContainers().filter(container => container === workspace.getCenter() || container.isVisible()).some(container => container.getPanes().some(pane => {
      const item = pane.getActiveItem();
      return item && item.getURI && item.getURI() === this.uri;
    }));
  }

  hasFocus() {
    const root = this.getDOMElement();
    return root && root.contains(document.activeElement);
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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