/**
 * @file The `Ord` type class represents types which support comparisons with a _total order_.
 *
 * Instances should satisfy the laws of total orderings:
 *
 * 1. Reflexivity: `S.compare(a, a) <= 0`
 * 2. Antisymmetry: if `S.compare(a, b) <= 0` and `S.compare(b, a) <= 0` then `a <-> b`
 * 3. Transitivity: if `S.compare(a, b) <= 0` and `S.compare(b, c) <= 0` then `S.compare(a, c) <= 0`
 *
 * See [Getting started with fp-ts: Ord](https://dev.to/gcanti/getting-started-with-fp-ts-ord-5f1e)
 */
import { semigroupOrdering } from './Ordering';
import { setoidBoolean, setoidNumber, setoidString } from './Setoid';
import { on } from './function';
/**
 * @since 1.0.0
 */
export const unsafeCompare = (x, y) => {
    return x < y ? -1 : x > y ? 1 : 0;
};
/**
 * @since 1.0.0
 */
export const ordString = Object.assign({}, setoidString, { compare: unsafeCompare });
/**
 * @since 1.0.0
 */
export const ordNumber = Object.assign({}, setoidNumber, { compare: unsafeCompare });
/**
 * @since 1.0.0
 */
export const ordBoolean = Object.assign({}, setoidBoolean, { compare: unsafeCompare });
/**
 * Test whether one value is _strictly less than_ another
 *
 * @since 1.0.0
 */
export const lessThan = (O) => (x, y) => {
    return O.compare(x, y) === -1;
};
/**
 * Test whether one value is _strictly greater than_ another
 *
 * @since 1.0.0
 */
export const greaterThan = (O) => (x, y) => {
    return O.compare(x, y) === 1;
};
/**
 * Test whether one value is _non-strictly less than_ another
 *
 * @since 1.0.0
 */
export const lessThanOrEq = (O) => (x, y) => {
    return O.compare(x, y) !== 1;
};
/**
 * Test whether one value is _non-strictly greater than_ another
 *
 * @since 1.0.0
 */
export const greaterThanOrEq = (O) => (x, y) => {
    return O.compare(x, y) !== -1;
};
/**
 * Take the minimum of two values. If they are considered equal, the first argument is chosen
 *
 * @since 1.0.0
 */
export const min = (O) => (x, y) => {
    return O.compare(x, y) === 1 ? y : x;
};
/**
 * Take the maximum of two values. If they are considered equal, the first argument is chosen
 *
 * @since 1.0.0
 */
export const max = (O) => (x, y) => {
    return O.compare(x, y) === -1 ? y : x;
};
/**
 * Clamp a value between a minimum and a maximum
 *
 * @since 1.0.0
 */
export const clamp = (O) => {
    const minO = min(O);
    const maxO = max(O);
    return (low, hi) => x => maxO(minO(x, hi), low);
};
/**
 * Test whether a value is between a minimum and a maximum (inclusive)
 *
 * @since 1.0.0
 */
export const between = (O) => {
    const lessThanO = lessThan(O);
    const greaterThanO = greaterThan(O);
    return (low, hi) => x => (lessThanO(x, low) || greaterThanO(x, hi) ? false : true);
};
/**
 * @since 1.0.0
 */
export const fromCompare = (compare) => {
    const optimizedCompare = (x, y) => (x === y ? 0 : compare(x, y));
    return {
        equals: (x, y) => optimizedCompare(x, y) === 0,
        compare: optimizedCompare
    };
};
/**
 * @since 1.0.0
 */
export const contramap = (f, fa) => {
    return fromCompare(on(fa.compare)(f));
};
/**
 * @since 1.0.0
 */
export const getSemigroup = () => {
    return {
        concat: (x, y) => fromCompare((a, b) => semigroupOrdering.concat(x.compare(a, b), y.compare(a, b)))
    };
};
/**
 * Given a tuple of `Ord`s returns an `Ord` for the tuple
 *
 * @example
 * import { getTupleOrd, ordString, ordNumber, ordBoolean } from 'fp-ts/lib/Ord'
 *
 * const O = getTupleOrd(ordString, ordNumber, ordBoolean)
 * assert.strictEqual(O.compare(['a', 1, true], ['b', 2, true]), -1)
 * assert.strictEqual(O.compare(['a', 1, true], ['a', 2, true]), -1)
 * assert.strictEqual(O.compare(['a', 1, true], ['a', 1, false]), 1)
 *
 * @since 1.14.3
 */
export const getTupleOrd = (...ords) => {
    const len = ords.length;
    return fromCompare((x, y) => {
        let i = 0;
        for (; i < len - 1; i++) {
            const r = ords[i].compare(x[i], y[i]);
            if (r !== 0) {
                return r;
            }
        }
        return ords[i].compare(x[i], y[i]);
    });
};
/**
 * Use `getTupleOrd` instead
 * @since 1.0.0
 * @deprecated
 */
export const getProductOrd = (OA, OB) => {
    return getTupleOrd(OA, OB);
};
/**
 * @since 1.3.0
 */
export const getDualOrd = (O) => {
    return fromCompare((x, y) => O.compare(y, x));
};
/**
 * @since 1.4.0
 */
export const ordDate = contramap(date => date.valueOf(), ordNumber);
