"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
function calculateSizes(budget, compilation) {
    const calculatorMap = {
        all: AllCalculator,
        allScript: AllScriptCalculator,
        any: AnyCalculator,
        anyScript: AnyScriptCalculator,
        bundle: BundleCalculator,
        initial: InitialCalculator,
    };
    const ctor = calculatorMap[budget.type];
    const calculator = new ctor(budget, compilation);
    return calculator.calculate();
}
exports.calculateSizes = calculateSizes;
class Calculator {
    constructor(budget, compilation) {
        this.budget = budget;
        this.compilation = compilation;
    }
}
exports.Calculator = Calculator;
/**
 * A named bundle.
 */
class BundleCalculator extends Calculator {
    calculate() {
        const size = this.compilation.chunks
            .filter(chunk => chunk.name === this.budget.name)
            .reduce((files, chunk) => [...files, ...chunk.files], [])
            .map((file) => this.compilation.assets[file].size())
            .reduce((total, size) => total + size, 0);
        return [{ size, label: this.budget.name }];
    }
}
/**
 * The sum of all initial chunks (marked as initial by webpack).
 */
class InitialCalculator extends Calculator {
    calculate() {
        const initialChunks = this.compilation.chunks.filter(chunk => chunk.isOnlyInitial());
        const size = initialChunks
            .reduce((files, chunk) => [...files, ...chunk.files], [])
            .filter((file) => !file.endsWith('.map'))
            .map((file) => this.compilation.assets[file].size())
            .reduce((total, size) => total + size, 0);
        return [{ size, label: 'initial' }];
    }
}
/**
 * The sum of all the scripts portions.
 */
class AllScriptCalculator extends Calculator {
    calculate() {
        const size = Object.keys(this.compilation.assets)
            .filter(key => key.endsWith('.js'))
            .map(key => this.compilation.assets[key])
            .map(asset => asset.size())
            .reduce((total, size) => total + size, 0);
        return [{ size, label: 'total scripts' }];
    }
}
/**
 * All scripts and assets added together.
 */
class AllCalculator extends Calculator {
    calculate() {
        const size = Object.keys(this.compilation.assets)
            .filter(key => !key.endsWith('.map'))
            .map(key => this.compilation.assets[key].size())
            .reduce((total, size) => total + size, 0);
        return [{ size, label: 'total' }];
    }
}
/**
 * Any script, individually.
 */
class AnyScriptCalculator extends Calculator {
    calculate() {
        return Object.keys(this.compilation.assets)
            .filter(key => key.endsWith('.js'))
            .map(key => {
            const asset = this.compilation.assets[key];
            return {
                size: asset.size(),
                label: key,
            };
        });
    }
}
/**
 * Any script or asset (images, css, etc).
 */
class AnyCalculator extends Calculator {
    calculate() {
        return Object.keys(this.compilation.assets)
            .filter(key => !key.endsWith('.map'))
            .map(key => {
            const asset = this.compilation.assets[key];
            return {
                size: asset.size(),
                label: key,
            };
        });
    }
}
/**
 * Calculate the bytes given a string value.
 */
function calculateBytes(input, baseline, factor = 1) {
    const matches = input.match(/^\s*(\d+(?:\.\d+)?)\s*(%|(?:[mM]|[kK]|[gG])?[bB])?\s*$/);
    if (!matches) {
        return NaN;
    }
    const baselineBytes = baseline && calculateBytes(baseline) || 0;
    let value = Number(matches[1]);
    switch (matches[2] && matches[2].toLowerCase()) {
        case '%':
            value = baselineBytes * value / 100;
            break;
        case 'kb':
            value *= 1024;
            break;
        case 'mb':
            value *= 1024 * 1024;
            break;
        case 'gb':
            value *= 1024 * 1024 * 1024;
            break;
    }
    if (baselineBytes === 0) {
        return value;
    }
    return baselineBytes + value * factor;
}
exports.calculateBytes = calculateBytes;
//# sourceMappingURL=data:application/json;base64,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