(function() {
  var CSON, Disposable, MenuHelpers, MenuManager, _, fs, ipcRenderer, path, platformMenu, ref, ref1;

  path = require('path');

  _ = require('underscore-plus');

  ipcRenderer = require('electron').ipcRenderer;

  CSON = require('season');

  fs = require('fs-plus');

  Disposable = require('event-kit').Disposable;

  MenuHelpers = require('./menu-helpers');

  platformMenu = (ref = require('../package.json')) != null ? (ref1 = ref._atomMenu) != null ? ref1.menu : void 0 : void 0;

  module.exports = MenuManager = (function() {
    function MenuManager(arg) {
      this.resourcePath = arg.resourcePath, this.keymapManager = arg.keymapManager, this.packageManager = arg.packageManager;
      this.initialized = false;
      this.pendingUpdateOperation = null;
      this.template = [];
      this.keymapManager.onDidLoadBundledKeymaps((function(_this) {
        return function() {
          return _this.loadPlatformItems();
        };
      })(this));
      this.packageManager.onDidActivateInitialPackages((function(_this) {
        return function() {
          return _this.sortPackagesMenu();
        };
      })(this));
    }

    MenuManager.prototype.initialize = function(arg) {
      this.resourcePath = arg.resourcePath;
      this.keymapManager.onDidReloadKeymap((function(_this) {
        return function() {
          return _this.update();
        };
      })(this));
      this.update();
      return this.initialized = true;
    };

    MenuManager.prototype.add = function(items) {
      var i, item, len;
      items = _.deepClone(items);
      for (i = 0, len = items.length; i < len; i++) {
        item = items[i];
        if (item.label == null) {
          continue;
        }
        this.merge(this.template, item);
      }
      this.update();
      return new Disposable((function(_this) {
        return function() {
          return _this.remove(items);
        };
      })(this));
    };

    MenuManager.prototype.remove = function(items) {
      var i, item, len;
      for (i = 0, len = items.length; i < len; i++) {
        item = items[i];
        this.unmerge(this.template, item);
      }
      return this.update();
    };

    MenuManager.prototype.clear = function() {
      this.template = [];
      return this.update();
    };

    MenuManager.prototype.includeSelector = function(selector) {
      var element, error, ref2, ref3, testBody, testDocument, testWorkspace, workspaceClasses;
      try {
        if (document.body.webkitMatchesSelector(selector)) {
          return true;
        }
      } catch (error1) {
        error = error1;
        return false;
      }
      if (this.testEditor == null) {
        testDocument = document.implementation.createDocument(document.namespaceURI, 'html');
        testBody = testDocument.createElement('body');
        (ref2 = testBody.classList).add.apply(ref2, this.classesForElement(document.body));
        testWorkspace = testDocument.createElement('atom-workspace');
        workspaceClasses = this.classesForElement(document.body.querySelector('atom-workspace'));
        if (workspaceClasses.length === 0) {
          workspaceClasses = ['workspace'];
        }
        (ref3 = testWorkspace.classList).add.apply(ref3, workspaceClasses);
        testBody.appendChild(testWorkspace);
        this.testEditor = testDocument.createElement('atom-text-editor');
        this.testEditor.classList.add('editor');
        testWorkspace.appendChild(this.testEditor);
      }
      element = this.testEditor;
      while (element) {
        if (element.webkitMatchesSelector(selector)) {
          return true;
        }
        element = element.parentElement;
      }
      return false;
    };

    MenuManager.prototype.update = function() {
      if (!this.initialized) {
        return;
      }
      if (this.pendingUpdateOperation != null) {
        clearTimeout(this.pendingUpdateOperation);
      }
      return this.pendingUpdateOperation = setTimeout((function(_this) {
        return function() {
          var binding, i, j, keystrokesByCommand, len, len1, name, ref2, ref3, unsetKeystrokes;
          unsetKeystrokes = new Set;
          ref2 = _this.keymapManager.getKeyBindings();
          for (i = 0, len = ref2.length; i < len; i++) {
            binding = ref2[i];
            if (binding.command === 'unset!') {
              unsetKeystrokes.add(binding.keystrokes);
            }
          }
          keystrokesByCommand = {};
          ref3 = _this.keymapManager.getKeyBindings();
          for (j = 0, len1 = ref3.length; j < len1; j++) {
            binding = ref3[j];
            if (!_this.includeSelector(binding.selector)) {
              continue;
            }
            if (unsetKeystrokes.has(binding.keystrokes)) {
              continue;
            }
            if (process.platform === 'darwin' && /^alt-(shift-)?.$/.test(binding.keystrokes)) {
              continue;
            }
            if (process.platform === 'win32' && /^ctrl-alt-(shift-)?.$/.test(binding.keystrokes)) {
              continue;
            }
            if (keystrokesByCommand[name = binding.command] == null) {
              keystrokesByCommand[name] = [];
            }
            keystrokesByCommand[binding.command].unshift(binding.keystrokes);
          }
          return _this.sendToBrowserProcess(_this.template, keystrokesByCommand);
        };
      })(this), 1);
    };

    MenuManager.prototype.loadPlatformItems = function() {
      var menu, menusDirPath, platformMenuPath;
      if (platformMenu != null) {
        return this.add(platformMenu);
      } else {
        menusDirPath = path.join(this.resourcePath, 'menus');
        platformMenuPath = fs.resolve(menusDirPath, process.platform, ['cson', 'json']);
        menu = CSON.readFileSync(platformMenuPath).menu;
        return this.add(menu);
      }
    };

    MenuManager.prototype.merge = function(menu, item) {
      return MenuHelpers.merge(menu, item);
    };

    MenuManager.prototype.unmerge = function(menu, item) {
      return MenuHelpers.unmerge(menu, item);
    };

    MenuManager.prototype.sendToBrowserProcess = function(template, keystrokesByCommand) {
      return ipcRenderer.send('update-application-menu', template, keystrokesByCommand);
    };

    MenuManager.prototype.classesForElement = function(element) {
      var classList;
      if (classList = element != null ? element.classList : void 0) {
        return Array.prototype.slice.apply(classList);
      } else {
        return [];
      }
    };

    MenuManager.prototype.sortPackagesMenu = function() {
      var packagesMenu;
      packagesMenu = _.find(this.template, function(arg) {
        var label;
        label = arg.label;
        return MenuHelpers.normalizeLabel(label) === 'Packages';
      });
      if ((packagesMenu != null ? packagesMenu.submenu : void 0) == null) {
        return;
      }
      packagesMenu.submenu.sort(function(item1, item2) {
        if (item1.label && item2.label) {
          return MenuHelpers.normalizeLabel(item1.label).localeCompare(MenuHelpers.normalizeLabel(item2.label));
        } else {
          return 0;
        }
      });
      return this.update();
    };

    return MenuManager;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
