"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _eventKit = require("event-kit");

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _state = _interopRequireDefault(require("./state"));

var _gitShellOutStrategy = require("../../git-shell-out-strategy");

var _workspaceChangeObserver = require("../workspace-change-observer");

var _patch = require("../patch");

var _discardHistory = _interopRequireDefault(require("../discard-history"));

var _branch = _interopRequireWildcard(require("../branch"));

var _author = _interopRequireDefault(require("../author"));

var _branchSet = _interopRequireDefault(require("../branch-set"));

var _remote = _interopRequireDefault(require("../remote"));

var _remoteSet = _interopRequireDefault(require("../remote-set"));

var _commit = _interopRequireDefault(require("../commit"));

var _operationStates = _interopRequireDefault(require("../operation-states"));

var _reporterProxy = require("../../reporter-proxy");

var _helpers = require("../../helpers");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * State used when the working directory contains a valid git repository and can be interacted with. Performs
 * actual git operations, caching the results, and broadcasts `onDidUpdate` events when write actions are
 * performed.
 */
class Present extends _state["default"] {
  constructor(repository, history) {
    super(repository);
    this.cache = new Cache();
    this.discardHistory = new _discardHistory["default"](this.createBlob.bind(this), this.expandBlobToFile.bind(this), this.mergeFile.bind(this), this.workdir(), {
      maxHistoryLength: 60
    });
    this.operationStates = new _operationStates["default"]({
      didUpdate: this.didUpdate.bind(this)
    });
    this.commitMessage = '';
    this.commitMessageTemplate = null;
    this.fetchInitialMessage();
    /* istanbul ignore else */

    if (history) {
      this.discardHistory.updateHistory(history);
    }
  }

  setCommitMessage(message, {
    suppressUpdate
  } = {
    suppressUpdate: false
  }) {
    this.commitMessage = message;

    if (!suppressUpdate) {
      this.didUpdate();
    }
  }

  setCommitMessageTemplate(template) {
    this.commitMessageTemplate = template;
  }

  async fetchInitialMessage() {
    const mergeMessage = await this.repository.getMergeMessage();
    const template = await this.fetchCommitMessageTemplate();

    if (template) {
      this.commitMessageTemplate = template;
    }

    if (mergeMessage) {
      this.setCommitMessage(mergeMessage);
    } else if (template) {
      this.setCommitMessage(template);
    }
  }

  getCommitMessage() {
    return this.commitMessage;
  }

  fetchCommitMessageTemplate() {
    return this.git().fetchCommitMessageTemplate();
  }

  getOperationStates() {
    return this.operationStates;
  }

  isPresent() {
    return true;
  }

  destroy() {
    this.cache.destroy();
    super.destroy();
  }

  showStatusBarTiles() {
    return true;
  }

  acceptInvalidation(spec) {
    this.cache.invalidate(spec());
    this.didUpdate();
  }

  invalidateCacheAfterFilesystemChange(events) {
    const paths = events.map(e => e.special || e.path);
    const keys = new Set();

    for (let i = 0; i < paths.length; i++) {
      const fullPath = paths[i];

      if (fullPath === _workspaceChangeObserver.FOCUS) {
        keys.add(Keys.statusBundle);

        for (const k of Keys.filePatch.eachWithOpts({
          staged: false
        })) {
          keys.add(k);
        }

        continue;
      }

      const includes = (...segments) => fullPath.includes(_path["default"].join(...segments));

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'index')) {
        keys.add(Keys.stagedChanges);
        keys.add(Keys.filePatch.all);
        keys.add(Keys.index.all);
        keys.add(Keys.statusBundle);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'HEAD')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'heads')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'remotes')) {
        keys.add(Keys.remotes);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'config')) {
        keys.add(Keys.remotes);
        keys.add(Keys.config.all);
        keys.add(Keys.statusBundle);
        continue;
      } // File change within the working directory


      const relativePath = _path["default"].relative(this.workdir(), fullPath);

      for (const key of Keys.filePatch.eachWithFileOpts([relativePath], [{
        staged: false
      }])) {
        keys.add(key);
      }

      keys.add(Keys.statusBundle);
    }
    /* istanbul ignore else */


    if (keys.size > 0) {
      this.cache.invalidate(Array.from(keys));
      this.didUpdate();
    }
  }

  isCommitMessageClean() {
    if (this.commitMessage.trim() === '') {
      return true;
    } else if (this.commitMessageTemplate) {
      return this.commitMessage === this.commitMessageTemplate;
    }

    return false;
  }

  async updateCommitMessageAfterFileSystemChange(events) {
    for (let i = 0; i < events.length; i++) {
      const event = events[i];

      if (!event.path) {
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'MERGE_HEAD')) {
        if (event.action === 'created') {
          if (this.isCommitMessageClean()) {
            this.setCommitMessage((await this.repository.getMergeMessage()));
          }
        } else if (event.action === 'deleted') {
          this.setCommitMessage(this.commitMessageTemplate || '');
        }
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'config')) {
        // this won't catch changes made to the template file itself...
        const template = await this.fetchCommitMessageTemplate();

        if (template === null) {
          this.setCommitMessage('');
        } else if (this.commitMessageTemplate !== template) {
          this.setCommitMessage(template);
        }

        this.setCommitMessageTemplate(template);
      }
    }
  }

  observeFilesystemChange(events) {
    this.invalidateCacheAfterFilesystemChange(events);
    this.updateCommitMessageAfterFileSystemChange(events);
  }

  refresh() {
    this.cache.clear();
    this.didUpdate();
  }

  init() {
    return super.init()["catch"](e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  }

  clone() {
    return super.clone()["catch"](e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  } // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // Staging and unstaging


  stageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().stageFiles(paths));
  }

  unstageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths));
  }

  stageFilesFromParentCommit(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths, 'HEAD~'));
  }

  stageFileModeChange(filePath, fileMode) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileModeChange(filePath, fileMode));
  }

  stageFileSymlinkChange(filePath) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileSymlinkChange(filePath));
  }

  applyPatchToIndex(multiFilePatch) {
    return this.invalidate(() => Keys.cacheOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr, {
        index: true
      });
    });
  }

  applyPatchToWorkdir(multiFilePatch) {
    return this.invalidate(() => Keys.workdirOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr);
    });
  } // Committing


  commit(message, options) {
    return this.invalidate(Keys.headOperationKeys, // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('COMMIT', async (message, options = {}) => {
      const coAuthors = options.coAuthors;
      const opts = !coAuthors ? options : _objectSpread({}, options, {
        coAuthors: coAuthors.map(author => {
          return {
            email: author.getEmail(),
            name: author.getFullName()
          };
        })
      });
      await this.git().commit(message, opts); // Collect commit metadata metrics
      // note: in GitShellOutStrategy we have counters for all git commands, including `commit`, but here we have
      //       access to additional metadata (unstaged file count) so it makes sense to collect commit events here

      const {
        unstagedFiles,
        mergeConflictFiles
      } = await this.getStatusesForChangedFiles();
      const unstagedCount = Object.keys(_objectSpread({}, unstagedFiles, mergeConflictFiles)).length;
      (0, _reporterProxy.addEvent)('commit', {
        "package": 'github',
        partial: unstagedCount > 0,
        amend: !!options.amend,
        coAuthorCount: coAuthors ? coAuthors.length : 0
      });
    }, message, options));
  } // Merging


  merge(branchName) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription], () => this.git().merge(branchName));
  }

  abortMerge() {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, Keys.filePatch.all, Keys.index.all], async () => {
      await this.git().abortMerge();
      this.setCommitMessage(this.commitMessageTemplate || '');
    });
  }

  checkoutSide(side, paths) {
    return this.git().checkoutSide(side, paths);
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return this.git().mergeFile(oursPath, commonBasePath, theirsPath, resultPath);
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...Keys.filePatch.eachWithFileOpts([filePath], [{
      staged: false
    }, {
      staged: true
    }]), Keys.index.oneWith(filePath)], () => this.git().writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha));
  } // Checkout


  checkout(revision, options = {}) {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({
      staged: true
    }), Keys.filePatch.allAgainstNonHead, Keys.headDescription, Keys.branches], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('CHECKOUT', (revision, options) => {
      return this.git().checkout(revision, options);
    }, revision, options));
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...paths.map(fileName => Keys.index.oneWith(fileName)), ...Keys.filePatch.eachWithFileOpts(paths, [{
      staged: true
    }]), ...Keys.filePatch.eachNonHeadWithFiles(paths)], () => this.git().checkoutFiles(paths, revision));
  } // Reset


  undoLastCommit() {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({
      staged: true
    }), Keys.headDescription], async () => {
      try {
        await this.git().reset('soft', 'HEAD~');
        (0, _reporterProxy.addEvent)('undo-last-commit', {
          "package": 'github'
        });
      } catch (e) {
        if (/unknown revision/.test(e.stdErr)) {
          // Initial commit
          await this.git().deleteRef('HEAD');
        } else {
          throw e;
        }
      }
    });
  } // Remote interactions


  fetch(branchName, options = {}) {
    return this.invalidate(() => [Keys.statusBundle, Keys.headDescription], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('FETCH', async branchName => {
      let finalRemoteName = options.remoteName;

      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);

        if (!remote.isPresent()) {
          return null;
        }

        finalRemoteName = remote.getName();
      }

      return this.git().fetch(finalRemoteName, branchName);
    }, branchName));
  }

  pull(branchName, options = {}) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription, Keys.branches], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PULL', async branchName => {
      let finalRemoteName = options.remoteName;

      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);

        if (!remote.isPresent()) {
          return null;
        }

        finalRemoteName = remote.getName();
      }

      return this.git().pull(finalRemoteName, branchName, options);
    }, branchName));
  }

  push(branchName, options = {}) {
    return this.invalidate(() => {
      const keys = [Keys.statusBundle, Keys.headDescription];

      if (options.setUpstream) {
        keys.push(Keys.branches);
        keys.push(...Keys.config.eachWithSetting(`branch.${branchName}.remote`));
      }

      return keys;
    }, // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PUSH', async (branchName, options) => {
      const remote = options.remote || (await this.getRemoteForBranch(branchName));
      return this.git().push(remote.getNameOr('origin'), branchName, options);
    }, branchName, options));
  } // Configuration


  setConfig(setting, value, options) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().setConfig(setting, value, options));
  }

  unsetConfig(setting) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().unsetConfig(setting));
  } // Direct blob interactions


  createBlob(options) {
    return this.git().createBlob(options);
  }

  expandBlobToFile(absFilePath, sha) {
    return this.git().expandBlobToFile(absFilePath, sha);
  } // Discard history


  createDiscardHistoryBlob() {
    return this.discardHistory.createHistoryBlob();
  }

  async updateDiscardHistory() {
    const history = await this.loadHistoryPayload();
    this.discardHistory.updateHistory(history);
  }

  async storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    const snapshots = await this.discardHistory.storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath);
    /* istanbul ignore else */

    if (snapshots) {
      await this.saveDiscardHistory();
    }

    return snapshots;
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return this.discardHistory.restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath);
  }

  async popDiscardHistory(partialDiscardFilePath = null) {
    const removed = await this.discardHistory.popHistory(partialDiscardFilePath);

    if (removed) {
      await this.saveDiscardHistory();
    }
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    this.discardHistory.clearHistory(partialDiscardFilePath);
    return this.saveDiscardHistory();
  }

  discardWorkDirChangesForPaths(paths) {
    return this.invalidate(() => [Keys.statusBundle, ...paths.map(filePath => Keys.filePatch.oneWith(filePath, {
      staged: false
    })), ...Keys.filePatch.eachNonHeadWithFiles(paths)], async () => {
      const untrackedFiles = await this.git().getUntrackedFiles();
      const [filesToRemove, filesToCheckout] = partition(paths, f => untrackedFiles.includes(f));
      await this.git().checkoutFiles(filesToCheckout);
      await Promise.all(filesToRemove.map(filePath => {
        const absPath = _path["default"].join(this.workdir(), filePath);

        return _fsExtra["default"].remove(absPath);
      }));
    });
  } // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Index queries


  getStatusBundle() {
    return this.cache.getOrSet(Keys.statusBundle, async () => {
      try {
        const bundle = await this.git().getStatusBundle();
        const results = await this.formatChangedFiles(bundle);
        results.branch = bundle.branch;
        return results;
      } catch (err) {
        if (err instanceof _gitShellOutStrategy.LargeRepoError) {
          this.transitionTo('TooLarge');
          return {
            branch: {},
            stagedFiles: {},
            unstagedFiles: {},
            mergeConflictFiles: {}
          };
        } else {
          throw err;
        }
      }
    });
  }

  async formatChangedFiles({
    changedEntries,
    untrackedEntries,
    renamedEntries,
    unmergedEntries
  }) {
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'modified',
      T: 'typechange'
    };
    const stagedFiles = {};
    const unstagedFiles = {};
    const mergeConflictFiles = {};
    changedEntries.forEach(entry => {
      if (entry.stagedStatus) {
        stagedFiles[entry.filePath] = statusMap[entry.stagedStatus];
      }

      if (entry.unstagedStatus) {
        unstagedFiles[entry.filePath] = statusMap[entry.unstagedStatus];
      }
    });
    untrackedEntries.forEach(entry => {
      unstagedFiles[entry.filePath] = statusMap.A;
    });
    renamedEntries.forEach(entry => {
      if (entry.stagedStatus === 'R') {
        stagedFiles[entry.filePath] = statusMap.A;
        stagedFiles[entry.origFilePath] = statusMap.D;
      }

      if (entry.unstagedStatus === 'R') {
        unstagedFiles[entry.filePath] = statusMap.A;
        unstagedFiles[entry.origFilePath] = statusMap.D;
      }

      if (entry.stagedStatus === 'C') {
        stagedFiles[entry.filePath] = statusMap.A;
      }

      if (entry.unstagedStatus === 'C') {
        unstagedFiles[entry.filePath] = statusMap.A;
      }
    });
    let statusToHead;

    for (let i = 0; i < unmergedEntries.length; i++) {
      const {
        stagedStatus,
        unstagedStatus,
        filePath
      } = unmergedEntries[i];

      if (stagedStatus === 'U' || unstagedStatus === 'U' || stagedStatus === 'A' && unstagedStatus === 'A') {
        // Skipping this check here because we only run a single `await`
        // and we only run it in the main, synchronous body of the for loop.
        // eslint-disable-next-line no-await-in-loop
        if (!statusToHead) {
          statusToHead = await this.git().diffFileStatus({
            target: 'HEAD'
          });
        }

        mergeConflictFiles[filePath] = {
          ours: statusMap[stagedStatus],
          theirs: statusMap[unstagedStatus],
          file: statusToHead[filePath] || 'equivalent'
        };
      }
    }

    return {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    };
  }

  async getStatusesForChangedFiles() {
    const {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    } = await this.getStatusBundle();
    return {
      stagedFiles,
      unstagedFiles,
      mergeConflictFiles
    };
  }

  getFilePatchForPath(filePath, options) {
    const opts = _objectSpread({
      staged: false,
      patchBuffer: null,
      builder: {},
      before: () => {},
      after: () => {}
    }, options);

    return this.cache.getOrSet(Keys.filePatch.oneWith(filePath, {
      staged: opts.staged
    }), async () => {
      const diffs = await this.git().getDiffsForFilePath(filePath, {
        staged: opts.staged
      });
      const payload = opts.before();
      const patch = (0, _patch.buildFilePatch)(diffs, opts.builder);

      if (opts.patchBuffer !== null) {
        patch.adoptBuffer(opts.patchBuffer);
      }

      opts.after(patch, payload);
      return patch;
    });
  }

  getDiffsForFilePath(filePath, baseCommit) {
    return this.cache.getOrSet(Keys.filePatch.oneWith(filePath, {
      baseCommit
    }), () => {
      return this.git().getDiffsForFilePath(filePath, {
        baseCommit
      });
    });
  }

  getStagedChangesPatch(options) {
    const opts = _objectSpread({
      builder: {},
      patchBuffer: null,
      before: () => {},
      after: () => {}
    }, options);

    return this.cache.getOrSet(Keys.stagedChanges, async () => {
      const diffs = await this.git().getStagedChangesPatch();
      const payload = opts.before();
      const patch = (0, _patch.buildMultiFilePatch)(diffs, opts.builder);

      if (opts.patchBuffer !== null) {
        patch.adoptBuffer(opts.patchBuffer);
      }

      opts.after(patch, payload);
      return patch;
    });
  }

  readFileFromIndex(filePath) {
    return this.cache.getOrSet(Keys.index.oneWith(filePath), () => {
      return this.git().readFileFromIndex(filePath);
    });
  } // Commit access


  getLastCommit() {
    return this.cache.getOrSet(Keys.lastCommit, async () => {
      const headCommit = await this.git().getHeadCommit();
      return headCommit.unbornRef ? _commit["default"].createUnborn() : new _commit["default"](headCommit);
    });
  }

  getCommit(sha) {
    return this.cache.getOrSet(Keys.blob.oneWith(sha), async () => {
      const [rawCommit] = await this.git().getCommits({
        max: 1,
        ref: sha,
        includePatch: true
      });
      const commit = new _commit["default"](rawCommit);
      return commit;
    });
  }

  getRecentCommits(options) {
    return this.cache.getOrSet(Keys.recentCommits, async () => {
      const commits = await this.git().getCommits(_objectSpread({
        ref: 'HEAD'
      }, options));
      return commits.map(commit => new _commit["default"](commit));
    });
  }

  async isCommitPushed(sha) {
    const remoteBranchesWithCommit = await this.git().getBranchesWithCommit(sha, {
      showLocal: false,
      showRemote: true
    });
    const currentRemote = (await this.repository.getCurrentBranch()).getUpstream();
    return remoteBranchesWithCommit.includes(currentRemote.getFullRef());
  } // Author information


  getAuthors(options) {
    // For now we'll do the naive thing and invalidate anytime HEAD moves. This ensures that we get new authors
    // introduced by newly created commits or pulled commits.
    // This means that we are constantly re-fetching data. If performance becomes a concern we can optimize
    return this.cache.getOrSet(Keys.authors, async () => {
      const authorMap = await this.git().getAuthors(options);
      return Object.keys(authorMap).map(email => new _author["default"](email, authorMap[email]));
    });
  } // Branches


  getBranches() {
    return this.cache.getOrSet(Keys.branches, async () => {
      const payloads = await this.git().getBranches();
      const branches = new _branchSet["default"]();

      for (const payload of payloads) {
        let upstream = _branch.nullBranch;

        if (payload.upstream) {
          upstream = payload.upstream.remoteName ? _branch["default"].createRemoteTracking(payload.upstream.trackingRef, payload.upstream.remoteName, payload.upstream.remoteRef) : new _branch["default"](payload.upstream.trackingRef);
        }

        let push = upstream;

        if (payload.push) {
          push = payload.push.remoteName ? _branch["default"].createRemoteTracking(payload.push.trackingRef, payload.push.remoteName, payload.push.remoteRef) : new _branch["default"](payload.push.trackingRef);
        }

        branches.add(new _branch["default"](payload.name, upstream, push, payload.head, {
          sha: payload.sha
        }));
      }

      return branches;
    });
  }

  getHeadDescription() {
    return this.cache.getOrSet(Keys.headDescription, () => {
      return this.git().describeHead();
    });
  } // Merging and rebasing status


  isMerging() {
    return this.git().isMerging(this.repository.getGitDirectoryPath());
  }

  isRebasing() {
    return this.git().isRebasing(this.repository.getGitDirectoryPath());
  } // Remotes


  getRemotes() {
    return this.cache.getOrSet(Keys.remotes, async () => {
      const remotesInfo = await this.git().getRemotes();
      return new _remoteSet["default"](remotesInfo.map(({
        name,
        url
      }) => new _remote["default"](name, url)));
    });
  }

  addRemote(name, url) {
    return this.invalidate(() => [...Keys.config.eachWithSetting(`remote.${name}.url`), ...Keys.config.eachWithSetting(`remote.${name}.fetch`), Keys.remotes], // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('ADDREMOTE', async (name, url) => {
      await this.git().addRemote(name, url);
      return new _remote["default"](name, url);
    }, name, url));
  }

  async getAheadCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.ahead;
  }

  async getBehindCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.behind;
  }

  getConfig(option, {
    local
  } = {
    local: false
  }) {
    return this.cache.getOrSet(Keys.config.oneWith(option, {
      local
    }), () => {
      return this.git().getConfig(option, {
        local
      });
    });
  }

  directGetConfig(key, options) {
    return this.getConfig(key, options);
  } // Direct blob access


  getBlobContents(sha) {
    return this.cache.getOrSet(Keys.blob.oneWith(sha), () => {
      return this.git().getBlobContents(sha);
    });
  }

  directGetBlobContents(sha) {
    return this.getBlobContents(sha);
  } // Discard history


  hasDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.hasHistory(partialDiscardFilePath);
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.getHistory(partialDiscardFilePath);
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return this.discardHistory.getLastSnapshots(partialDiscardFilePath);
  } // Cache

  /* istanbul ignore next */


  getCache() {
    return this.cache;
  }

  invalidate(spec, body) {
    return body().then(result => {
      this.acceptInvalidation(spec);
      return result;
    }, err => {
      this.acceptInvalidation(spec);
      return Promise.reject(err);
    });
  }

}

exports["default"] = Present;

_state["default"].register(Present);

function partition(array, predicate) {
  const matches = [];
  const nonmatches = [];
  array.forEach(item => {
    if (predicate(item)) {
      matches.push(item);
    } else {
      nonmatches.push(item);
    }
  });
  return [matches, nonmatches];
}

class Cache {
  constructor() {
    this.storage = new Map();
    this.byGroup = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  getOrSet(key, operation) {
    const primary = key.getPrimary();
    const existing = this.storage.get(primary);

    if (existing !== undefined) {
      existing.hits++;
      return existing.promise;
    }

    const created = operation();
    this.storage.set(primary, {
      createdAt: performance.now(),
      hits: 0,
      promise: created
    });
    const groups = key.getGroups();

    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];
      let groupSet = this.byGroup.get(group);

      if (groupSet === undefined) {
        groupSet = new Set();
        this.byGroup.set(group, groupSet);
      }

      groupSet.add(key);
    }

    this.didUpdate();
    return created;
  }

  invalidate(keys) {
    for (let i = 0; i < keys.length; i++) {
      keys[i].removeFromCache(this);
    }

    if (keys.length > 0) {
      this.didUpdate();
    }
  }

  keysInGroup(group) {
    return this.byGroup.get(group) || [];
  }

  removePrimary(primary) {
    this.storage["delete"](primary);
    this.didUpdate();
  }

  removeFromGroup(group, key) {
    const groupSet = this.byGroup.get(group);
    groupSet && groupSet["delete"](key);
    this.didUpdate();
  }
  /* istanbul ignore next */


  [Symbol.iterator]() {
    return this.storage[Symbol.iterator]();
  }

  clear() {
    this.storage.clear();
    this.byGroup.clear();
    this.didUpdate();
  }

  didUpdate() {
    this.emitter.emit('did-update');
  }
  /* istanbul ignore next */


  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  destroy() {
    this.emitter.dispose();
  }

}

class CacheKey {
  constructor(primary, groups = []) {
    this.primary = primary;
    this.groups = groups;
  }

  getPrimary() {
    return this.primary;
  }

  getGroups() {
    return this.groups;
  }

  removeFromCache(cache, withoutGroup = null) {
    cache.removePrimary(this.getPrimary());
    const groups = this.getGroups();

    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];

      if (group === withoutGroup) {
        continue;
      }

      cache.removeFromGroup(group, this);
    }
  }
  /* istanbul ignore next */


  toString() {
    return `CacheKey(${this.primary})`;
  }

}

class GroupKey {
  constructor(group) {
    this.group = group;
  }

  removeFromCache(cache) {
    for (const matchingKey of cache.keysInGroup(this.group)) {
      matchingKey.removeFromCache(cache, this.group);
    }
  }
  /* istanbul ignore next */


  toString() {
    return `GroupKey(${this.group})`;
  }

}

const Keys = {
  statusBundle: new CacheKey('status-bundle'),
  stagedChanges: new CacheKey('staged-changes'),
  filePatch: {
    _optKey: ({
      staged
    }) => staged ? 's' : 'u',
    oneWith: (fileName, options) => {
      // <-- Keys.filePatch
      const optKey = Keys.filePatch._optKey(options);

      const baseCommit = options.baseCommit || 'head';
      const extraGroups = [];

      if (options.baseCommit) {
        extraGroups.push(`file-patch:base-nonhead:path-${fileName}`);
        extraGroups.push('file-patch:base-nonhead');
      } else {
        extraGroups.push('file-patch:base-head');
      }

      return new CacheKey(`file-patch:${optKey}:${baseCommit}:${fileName}`, ['file-patch', `file-patch:opt-${optKey}`, `file-patch:opt-${optKey}:path-${fileName}`, ...extraGroups]);
    },
    eachWithFileOpts: (fileNames, opts) => {
      const keys = [];

      for (let i = 0; i < fileNames.length; i++) {
        for (let j = 0; j < opts.length; j++) {
          keys.push(new GroupKey(`file-patch:opt-${Keys.filePatch._optKey(opts[j])}:path-${fileNames[i]}`));
        }
      }

      return keys;
    },
    eachNonHeadWithFiles: fileNames => {
      return fileNames.map(fileName => new GroupKey(`file-patch:base-nonhead:path-${fileName}`));
    },
    allAgainstNonHead: new GroupKey('file-patch:base-nonhead'),
    eachWithOpts: (...opts) => opts.map(opt => new GroupKey(`file-patch:opt-${Keys.filePatch._optKey(opt)}`)),
    all: new GroupKey('file-patch')
  },
  index: {
    oneWith: fileName => new CacheKey(`index:${fileName}`, ['index']),
    all: new GroupKey('index')
  },
  lastCommit: new CacheKey('last-commit'),
  recentCommits: new CacheKey('recent-commits'),
  authors: new CacheKey('authors'),
  branches: new CacheKey('branches'),
  headDescription: new CacheKey('head-description'),
  remotes: new CacheKey('remotes'),
  config: {
    _optKey: options => options.local ? 'l' : '',
    oneWith: (setting, options) => {
      const optKey = Keys.config._optKey(options);

      return new CacheKey(`config:${optKey}:${setting}`, ['config', `config:${optKey}`]);
    },
    eachWithSetting: setting => [Keys.config.oneWith(setting, {
      local: true
    }), Keys.config.oneWith(setting, {
      local: false
    })],
    all: new GroupKey('config')
  },
  blob: {
    oneWith: sha => new CacheKey(`blob:${sha}`, ['blob'])
  },
  // Common collections of keys and patterns for use with invalidate().
  workdirOperationKeys: fileNames => [Keys.statusBundle, ...Keys.filePatch.eachWithFileOpts(fileNames, [{
    staged: false
  }])],
  cacheOperationKeys: fileNames => [...Keys.workdirOperationKeys(fileNames), ...Keys.filePatch.eachWithFileOpts(fileNames, [{
    staged: true
  }]), ...fileNames.map(Keys.index.oneWith), Keys.stagedChanges],
  headOperationKeys: () => [Keys.headDescription, Keys.branches, ...Keys.filePatch.eachWithOpts({
    staged: true
  }), Keys.filePatch.allAgainstNonHead, Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle]
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInByZXNlbnQuanMiXSwibmFtZXMiOlsiUHJlc2VudCIsIlN0YXRlIiwiY29uc3RydWN0b3IiLCJyZXBvc2l0b3J5IiwiaGlzdG9yeSIsImNhY2hlIiwiQ2FjaGUiLCJkaXNjYXJkSGlzdG9yeSIsIkRpc2NhcmRIaXN0b3J5IiwiY3JlYXRlQmxvYiIsImJpbmQiLCJleHBhbmRCbG9iVG9GaWxlIiwibWVyZ2VGaWxlIiwid29ya2RpciIsIm1heEhpc3RvcnlMZW5ndGgiLCJvcGVyYXRpb25TdGF0ZXMiLCJPcGVyYXRpb25TdGF0ZXMiLCJkaWRVcGRhdGUiLCJjb21taXRNZXNzYWdlIiwiY29tbWl0TWVzc2FnZVRlbXBsYXRlIiwiZmV0Y2hJbml0aWFsTWVzc2FnZSIsInVwZGF0ZUhpc3RvcnkiLCJzZXRDb21taXRNZXNzYWdlIiwibWVzc2FnZSIsInN1cHByZXNzVXBkYXRlIiwic2V0Q29tbWl0TWVzc2FnZVRlbXBsYXRlIiwidGVtcGxhdGUiLCJtZXJnZU1lc3NhZ2UiLCJnZXRNZXJnZU1lc3NhZ2UiLCJmZXRjaENvbW1pdE1lc3NhZ2VUZW1wbGF0ZSIsImdldENvbW1pdE1lc3NhZ2UiLCJnaXQiLCJnZXRPcGVyYXRpb25TdGF0ZXMiLCJpc1ByZXNlbnQiLCJkZXN0cm95Iiwic2hvd1N0YXR1c0JhclRpbGVzIiwiYWNjZXB0SW52YWxpZGF0aW9uIiwic3BlYyIsImludmFsaWRhdGUiLCJpbnZhbGlkYXRlQ2FjaGVBZnRlckZpbGVzeXN0ZW1DaGFuZ2UiLCJldmVudHMiLCJwYXRocyIsIm1hcCIsImUiLCJzcGVjaWFsIiwicGF0aCIsImtleXMiLCJTZXQiLCJpIiwibGVuZ3RoIiwiZnVsbFBhdGgiLCJGT0NVUyIsImFkZCIsIktleXMiLCJzdGF0dXNCdW5kbGUiLCJrIiwiZmlsZVBhdGNoIiwiZWFjaFdpdGhPcHRzIiwic3RhZ2VkIiwiaW5jbHVkZXMiLCJzZWdtZW50cyIsImpvaW4iLCJzdGFnZWRDaGFuZ2VzIiwiYWxsIiwiaW5kZXgiLCJicmFuY2hlcyIsImxhc3RDb21taXQiLCJyZWNlbnRDb21taXRzIiwiaGVhZERlc2NyaXB0aW9uIiwiYXV0aG9ycyIsInJlbW90ZXMiLCJjb25maWciLCJyZWxhdGl2ZVBhdGgiLCJyZWxhdGl2ZSIsImtleSIsImVhY2hXaXRoRmlsZU9wdHMiLCJzaXplIiwiQXJyYXkiLCJmcm9tIiwiaXNDb21taXRNZXNzYWdlQ2xlYW4iLCJ0cmltIiwidXBkYXRlQ29tbWl0TWVzc2FnZUFmdGVyRmlsZVN5c3RlbUNoYW5nZSIsImV2ZW50IiwiYWN0aW9uIiwib2JzZXJ2ZUZpbGVzeXN0ZW1DaGFuZ2UiLCJyZWZyZXNoIiwiY2xlYXIiLCJpbml0Iiwic3RkRXJyIiwiUHJvbWlzZSIsInJlamVjdCIsImNsb25lIiwic3RhZ2VGaWxlcyIsImNhY2hlT3BlcmF0aW9uS2V5cyIsInVuc3RhZ2VGaWxlcyIsInN0YWdlRmlsZXNGcm9tUGFyZW50Q29tbWl0Iiwic3RhZ2VGaWxlTW9kZUNoYW5nZSIsImZpbGVQYXRoIiwiZmlsZU1vZGUiLCJzdGFnZUZpbGVTeW1saW5rQ2hhbmdlIiwiYXBwbHlQYXRjaFRvSW5kZXgiLCJtdWx0aUZpbGVQYXRjaCIsImdldFBhdGhTZXQiLCJwYXRjaFN0ciIsInRvU3RyaW5nIiwiYXBwbHlQYXRjaCIsImFwcGx5UGF0Y2hUb1dvcmtkaXIiLCJ3b3JrZGlyT3BlcmF0aW9uS2V5cyIsImNvbW1pdCIsIm9wdGlvbnMiLCJoZWFkT3BlcmF0aW9uS2V5cyIsImV4ZWN1dGVQaXBlbGluZUFjdGlvbiIsImNvQXV0aG9ycyIsIm9wdHMiLCJhdXRob3IiLCJlbWFpbCIsImdldEVtYWlsIiwibmFtZSIsImdldEZ1bGxOYW1lIiwidW5zdGFnZWRGaWxlcyIsIm1lcmdlQ29uZmxpY3RGaWxlcyIsImdldFN0YXR1c2VzRm9yQ2hhbmdlZEZpbGVzIiwidW5zdGFnZWRDb3VudCIsIk9iamVjdCIsInBhcnRpYWwiLCJhbWVuZCIsImNvQXV0aG9yQ291bnQiLCJtZXJnZSIsImJyYW5jaE5hbWUiLCJhYm9ydE1lcmdlIiwiY2hlY2tvdXRTaWRlIiwic2lkZSIsIm91cnNQYXRoIiwiY29tbW9uQmFzZVBhdGgiLCJ0aGVpcnNQYXRoIiwicmVzdWx0UGF0aCIsIndyaXRlTWVyZ2VDb25mbGljdFRvSW5kZXgiLCJjb21tb25CYXNlU2hhIiwib3Vyc1NoYSIsInRoZWlyc1NoYSIsIm9uZVdpdGgiLCJjaGVja291dCIsInJldmlzaW9uIiwiYWxsQWdhaW5zdE5vbkhlYWQiLCJjaGVja291dFBhdGhzQXRSZXZpc2lvbiIsImZpbGVOYW1lIiwiZWFjaE5vbkhlYWRXaXRoRmlsZXMiLCJjaGVja291dEZpbGVzIiwidW5kb0xhc3RDb21taXQiLCJyZXNldCIsInRlc3QiLCJkZWxldGVSZWYiLCJmZXRjaCIsImZpbmFsUmVtb3RlTmFtZSIsInJlbW90ZU5hbWUiLCJyZW1vdGUiLCJnZXRSZW1vdGVGb3JCcmFuY2giLCJnZXROYW1lIiwicHVsbCIsInB1c2giLCJzZXRVcHN0cmVhbSIsImVhY2hXaXRoU2V0dGluZyIsImdldE5hbWVPciIsInNldENvbmZpZyIsInNldHRpbmciLCJ2YWx1ZSIsInVuc2V0Q29uZmlnIiwiYWJzRmlsZVBhdGgiLCJzaGEiLCJjcmVhdGVEaXNjYXJkSGlzdG9yeUJsb2IiLCJjcmVhdGVIaXN0b3J5QmxvYiIsInVwZGF0ZURpc2NhcmRIaXN0b3J5IiwibG9hZEhpc3RvcnlQYXlsb2FkIiwic3RvcmVCZWZvcmVBbmRBZnRlckJsb2JzIiwiZmlsZVBhdGhzIiwiaXNTYWZlIiwiZGVzdHJ1Y3RpdmVBY3Rpb24iLCJwYXJ0aWFsRGlzY2FyZEZpbGVQYXRoIiwic25hcHNob3RzIiwic2F2ZURpc2NhcmRIaXN0b3J5IiwicmVzdG9yZUxhc3REaXNjYXJkSW5UZW1wRmlsZXMiLCJwb3BEaXNjYXJkSGlzdG9yeSIsInJlbW92ZWQiLCJwb3BIaXN0b3J5IiwiY2xlYXJEaXNjYXJkSGlzdG9yeSIsImNsZWFySGlzdG9yeSIsImRpc2NhcmRXb3JrRGlyQ2hhbmdlc0ZvclBhdGhzIiwidW50cmFja2VkRmlsZXMiLCJnZXRVbnRyYWNrZWRGaWxlcyIsImZpbGVzVG9SZW1vdmUiLCJmaWxlc1RvQ2hlY2tvdXQiLCJwYXJ0aXRpb24iLCJmIiwiYWJzUGF0aCIsImZzIiwicmVtb3ZlIiwiZ2V0U3RhdHVzQnVuZGxlIiwiZ2V0T3JTZXQiLCJidW5kbGUiLCJyZXN1bHRzIiwiZm9ybWF0Q2hhbmdlZEZpbGVzIiwiYnJhbmNoIiwiZXJyIiwiTGFyZ2VSZXBvRXJyb3IiLCJ0cmFuc2l0aW9uVG8iLCJzdGFnZWRGaWxlcyIsImNoYW5nZWRFbnRyaWVzIiwidW50cmFja2VkRW50cmllcyIsInJlbmFtZWRFbnRyaWVzIiwidW5tZXJnZWRFbnRyaWVzIiwic3RhdHVzTWFwIiwiQSIsIk0iLCJEIiwiVSIsIlQiLCJmb3JFYWNoIiwiZW50cnkiLCJzdGFnZWRTdGF0dXMiLCJ1bnN0YWdlZFN0YXR1cyIsIm9yaWdGaWxlUGF0aCIsInN0YXR1c1RvSGVhZCIsImRpZmZGaWxlU3RhdHVzIiwidGFyZ2V0Iiwib3VycyIsInRoZWlycyIsImZpbGUiLCJnZXRGaWxlUGF0Y2hGb3JQYXRoIiwicGF0Y2hCdWZmZXIiLCJidWlsZGVyIiwiYmVmb3JlIiwiYWZ0ZXIiLCJkaWZmcyIsImdldERpZmZzRm9yRmlsZVBhdGgiLCJwYXlsb2FkIiwicGF0Y2giLCJhZG9wdEJ1ZmZlciIsImJhc2VDb21taXQiLCJnZXRTdGFnZWRDaGFuZ2VzUGF0Y2giLCJyZWFkRmlsZUZyb21JbmRleCIsImdldExhc3RDb21taXQiLCJoZWFkQ29tbWl0IiwiZ2V0SGVhZENvbW1pdCIsInVuYm9yblJlZiIsIkNvbW1pdCIsImNyZWF0ZVVuYm9ybiIsImdldENvbW1pdCIsImJsb2IiLCJyYXdDb21taXQiLCJnZXRDb21taXRzIiwibWF4IiwicmVmIiwiaW5jbHVkZVBhdGNoIiwiZ2V0UmVjZW50Q29tbWl0cyIsImNvbW1pdHMiLCJpc0NvbW1pdFB1c2hlZCIsInJlbW90ZUJyYW5jaGVzV2l0aENvbW1pdCIsImdldEJyYW5jaGVzV2l0aENvbW1pdCIsInNob3dMb2NhbCIsInNob3dSZW1vdGUiLCJjdXJyZW50UmVtb3RlIiwiZ2V0Q3VycmVudEJyYW5jaCIsImdldFVwc3RyZWFtIiwiZ2V0RnVsbFJlZiIsImdldEF1dGhvcnMiLCJhdXRob3JNYXAiLCJBdXRob3IiLCJnZXRCcmFuY2hlcyIsInBheWxvYWRzIiwiQnJhbmNoU2V0IiwidXBzdHJlYW0iLCJudWxsQnJhbmNoIiwiQnJhbmNoIiwiY3JlYXRlUmVtb3RlVHJhY2tpbmciLCJ0cmFja2luZ1JlZiIsInJlbW90ZVJlZiIsImhlYWQiLCJnZXRIZWFkRGVzY3JpcHRpb24iLCJkZXNjcmliZUhlYWQiLCJpc01lcmdpbmciLCJnZXRHaXREaXJlY3RvcnlQYXRoIiwiaXNSZWJhc2luZyIsImdldFJlbW90ZXMiLCJyZW1vdGVzSW5mbyIsIlJlbW90ZVNldCIsInVybCIsIlJlbW90ZSIsImFkZFJlbW90ZSIsImdldEFoZWFkQ291bnQiLCJhaGVhZEJlaGluZCIsImFoZWFkIiwiZ2V0QmVoaW5kQ291bnQiLCJiZWhpbmQiLCJnZXRDb25maWciLCJvcHRpb24iLCJsb2NhbCIsImRpcmVjdEdldENvbmZpZyIsImdldEJsb2JDb250ZW50cyIsImRpcmVjdEdldEJsb2JDb250ZW50cyIsImhhc0Rpc2NhcmRIaXN0b3J5IiwiaGFzSGlzdG9yeSIsImdldERpc2NhcmRIaXN0b3J5IiwiZ2V0SGlzdG9yeSIsImdldExhc3RIaXN0b3J5U25hcHNob3RzIiwiZ2V0TGFzdFNuYXBzaG90cyIsImdldENhY2hlIiwiYm9keSIsInRoZW4iLCJyZXN1bHQiLCJyZWdpc3RlciIsImFycmF5IiwicHJlZGljYXRlIiwibWF0Y2hlcyIsIm5vbm1hdGNoZXMiLCJpdGVtIiwic3RvcmFnZSIsIk1hcCIsImJ5R3JvdXAiLCJlbWl0dGVyIiwiRW1pdHRlciIsIm9wZXJhdGlvbiIsInByaW1hcnkiLCJnZXRQcmltYXJ5IiwiZXhpc3RpbmciLCJnZXQiLCJ1bmRlZmluZWQiLCJoaXRzIiwicHJvbWlzZSIsImNyZWF0ZWQiLCJzZXQiLCJjcmVhdGVkQXQiLCJwZXJmb3JtYW5jZSIsIm5vdyIsImdyb3VwcyIsImdldEdyb3VwcyIsImdyb3VwIiwiZ3JvdXBTZXQiLCJyZW1vdmVGcm9tQ2FjaGUiLCJrZXlzSW5Hcm91cCIsInJlbW92ZVByaW1hcnkiLCJyZW1vdmVGcm9tR3JvdXAiLCJTeW1ib2wiLCJpdGVyYXRvciIsImVtaXQiLCJvbkRpZFVwZGF0ZSIsImNhbGxiYWNrIiwib24iLCJkaXNwb3NlIiwiQ2FjaGVLZXkiLCJ3aXRob3V0R3JvdXAiLCJHcm91cEtleSIsIm1hdGNoaW5nS2V5IiwiX29wdEtleSIsIm9wdEtleSIsImV4dHJhR3JvdXBzIiwiZmlsZU5hbWVzIiwiaiIsIm9wdCJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOztBQUNBOztBQUNBOztBQUVBOztBQUVBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOzs7Ozs7Ozs7O0FBRUE7Ozs7O0FBS2UsTUFBTUEsT0FBTixTQUFzQkMsaUJBQXRCLENBQTRCO0FBQ3pDQyxFQUFBQSxXQUFXLENBQUNDLFVBQUQsRUFBYUMsT0FBYixFQUFzQjtBQUMvQixVQUFNRCxVQUFOO0FBRUEsU0FBS0UsS0FBTCxHQUFhLElBQUlDLEtBQUosRUFBYjtBQUVBLFNBQUtDLGNBQUwsR0FBc0IsSUFBSUMsMEJBQUosQ0FDcEIsS0FBS0MsVUFBTCxDQUFnQkMsSUFBaEIsQ0FBcUIsSUFBckIsQ0FEb0IsRUFFcEIsS0FBS0MsZ0JBQUwsQ0FBc0JELElBQXRCLENBQTJCLElBQTNCLENBRm9CLEVBR3BCLEtBQUtFLFNBQUwsQ0FBZUYsSUFBZixDQUFvQixJQUFwQixDQUhvQixFQUlwQixLQUFLRyxPQUFMLEVBSm9CLEVBS3BCO0FBQUNDLE1BQUFBLGdCQUFnQixFQUFFO0FBQW5CLEtBTG9CLENBQXRCO0FBUUEsU0FBS0MsZUFBTCxHQUF1QixJQUFJQywyQkFBSixDQUFvQjtBQUFDQyxNQUFBQSxTQUFTLEVBQUUsS0FBS0EsU0FBTCxDQUFlUCxJQUFmLENBQW9CLElBQXBCO0FBQVosS0FBcEIsQ0FBdkI7QUFFQSxTQUFLUSxhQUFMLEdBQXFCLEVBQXJCO0FBQ0EsU0FBS0MscUJBQUwsR0FBNkIsSUFBN0I7QUFDQSxTQUFLQyxtQkFBTDtBQUVBOztBQUNBLFFBQUloQixPQUFKLEVBQWE7QUFDWCxXQUFLRyxjQUFMLENBQW9CYyxhQUFwQixDQUFrQ2pCLE9BQWxDO0FBQ0Q7QUFDRjs7QUFFRGtCLEVBQUFBLGdCQUFnQixDQUFDQyxPQUFELEVBQVU7QUFBQ0MsSUFBQUE7QUFBRCxNQUFtQjtBQUFDQSxJQUFBQSxjQUFjLEVBQUU7QUFBakIsR0FBN0IsRUFBc0Q7QUFDcEUsU0FBS04sYUFBTCxHQUFxQkssT0FBckI7O0FBQ0EsUUFBSSxDQUFDQyxjQUFMLEVBQXFCO0FBQ25CLFdBQUtQLFNBQUw7QUFDRDtBQUNGOztBQUVEUSxFQUFBQSx3QkFBd0IsQ0FBQ0MsUUFBRCxFQUFXO0FBQ2pDLFNBQUtQLHFCQUFMLEdBQTZCTyxRQUE3QjtBQUNEOztBQUVELFFBQU1OLG1CQUFOLEdBQTRCO0FBQzFCLFVBQU1PLFlBQVksR0FBRyxNQUFNLEtBQUt4QixVQUFMLENBQWdCeUIsZUFBaEIsRUFBM0I7QUFDQSxVQUFNRixRQUFRLEdBQUcsTUFBTSxLQUFLRywwQkFBTCxFQUF2Qjs7QUFDQSxRQUFJSCxRQUFKLEVBQWM7QUFDWixXQUFLUCxxQkFBTCxHQUE2Qk8sUUFBN0I7QUFDRDs7QUFDRCxRQUFJQyxZQUFKLEVBQWtCO0FBQ2hCLFdBQUtMLGdCQUFMLENBQXNCSyxZQUF0QjtBQUNELEtBRkQsTUFFTyxJQUFJRCxRQUFKLEVBQWM7QUFDbkIsV0FBS0osZ0JBQUwsQ0FBc0JJLFFBQXRCO0FBQ0Q7QUFDRjs7QUFFREksRUFBQUEsZ0JBQWdCLEdBQUc7QUFDakIsV0FBTyxLQUFLWixhQUFaO0FBQ0Q7O0FBRURXLEVBQUFBLDBCQUEwQixHQUFHO0FBQzNCLFdBQU8sS0FBS0UsR0FBTCxHQUFXRiwwQkFBWCxFQUFQO0FBQ0Q7O0FBRURHLEVBQUFBLGtCQUFrQixHQUFHO0FBQ25CLFdBQU8sS0FBS2pCLGVBQVo7QUFDRDs7QUFFRGtCLEVBQUFBLFNBQVMsR0FBRztBQUNWLFdBQU8sSUFBUDtBQUNEOztBQUVEQyxFQUFBQSxPQUFPLEdBQUc7QUFDUixTQUFLN0IsS0FBTCxDQUFXNkIsT0FBWDtBQUNBLFVBQU1BLE9BQU47QUFDRDs7QUFFREMsRUFBQUEsa0JBQWtCLEdBQUc7QUFDbkIsV0FBTyxJQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLGtCQUFrQixDQUFDQyxJQUFELEVBQU87QUFDdkIsU0FBS2hDLEtBQUwsQ0FBV2lDLFVBQVgsQ0FBc0JELElBQUksRUFBMUI7QUFDQSxTQUFLcEIsU0FBTDtBQUNEOztBQUVEc0IsRUFBQUEsb0NBQW9DLENBQUNDLE1BQUQsRUFBUztBQUMzQyxVQUFNQyxLQUFLLEdBQUdELE1BQU0sQ0FBQ0UsR0FBUCxDQUFXQyxDQUFDLElBQUlBLENBQUMsQ0FBQ0MsT0FBRixJQUFhRCxDQUFDLENBQUNFLElBQS9CLENBQWQ7QUFDQSxVQUFNQyxJQUFJLEdBQUcsSUFBSUMsR0FBSixFQUFiOztBQUNBLFNBQUssSUFBSUMsQ0FBQyxHQUFHLENBQWIsRUFBZ0JBLENBQUMsR0FBR1AsS0FBSyxDQUFDUSxNQUExQixFQUFrQ0QsQ0FBQyxFQUFuQyxFQUF1QztBQUNyQyxZQUFNRSxRQUFRLEdBQUdULEtBQUssQ0FBQ08sQ0FBRCxDQUF0Qjs7QUFFQSxVQUFJRSxRQUFRLEtBQUtDLDhCQUFqQixFQUF3QjtBQUN0QkwsUUFBQUEsSUFBSSxDQUFDTSxHQUFMLENBQVNDLElBQUksQ0FBQ0MsWUFBZDs7QUFDQSxhQUFLLE1BQU1DLENBQVgsSUFBZ0JGLElBQUksQ0FBQ0csU0FBTCxDQUFlQyxZQUFmLENBQTRCO0FBQUNDLFVBQUFBLE1BQU0sRUFBRTtBQUFULFNBQTVCLENBQWhCLEVBQThEO0FBQzVEWixVQUFBQSxJQUFJLENBQUNNLEdBQUwsQ0FBU0csQ0FBVDtBQUNEOztBQUNEO0FBQ0Q7O0FBRUQsWUFBTUksUUFBUSxHQUFHLENBQUMsR0FBR0MsUUFBSixLQUFpQlYsUUFBUSxDQUFDUyxRQUFULENBQWtCZCxpQkFBS2dCLElBQUwsQ0FBVSxHQUFHRCxRQUFiLENBQWxCLENBQWxDOztBQUVBLFVBQUksK0JBQWlCVixRQUFqQixFQUEyQixNQUEzQixFQUFtQyxPQUFuQyxDQUFKLEVBQWlEO0FBQy9DSixRQUFBQSxJQUFJLENBQUNNLEdBQUwsQ0FBU0MsSUFBSSxDQUFDUyxhQUFkO0FBQ0FoQixRQUFBQSxJQUFJLENBQUNNLEdBQUwsQ0FBU0MsSUFBSSxDQUFDRyxTQUFMLENBQWVPLEdBQXhCO0FBQ0FqQixRQUFBQSxJQUFJLENBQUNNLEdBQUwsQ0FBU0MsSUFBSSxDQUFDVyxLQUFMLENBQVdELEdBQXBCO0FBQ0FqQixRQUFBQSxJQUFJLENBQUNNLEdBQUwsQ0FBU0MsSUFBSSxDQUFDQyxZQUFkO0FBQ0E7QUFDRDs7QUFFRCxVQUFJLCtCQUFpQkosUUFBakIsRUFBMkIsTUFBM0IsRUFBbUMsTUFBbkMsQ0FBSixFQUFnRDtBQUM5Q0osUUFBQUEsSUFBSSxDQUFDTSxHQUFMLENBQVNDLElBQUksQ0FBQ1ksUUFBZDtBQUNBbkIsUUFBQUEsSUFBSSxDQUFDTSxHQUFMLENBQVNDLElBQUksQ0FBQ2EsVUFBZDtBQUNBcEIsUUFBQUEsSUFBSSxDQUFDTSxHQUFMLENBQVNDLElBQUksQ0FBQ2MsYUFBZDtBQUNBckIsUUFBQUEsSUFBSSxDQUFDTSxHQUFMLENBQVNDLElBQUksQ0FBQ0MsWUFBZDtBQUNBUixRQUFBQSxJQUFJLENBQUNNLEdBQUwsQ0FBU0MsSUFBSSxDQUFDZSxlQUFkO0FBQ0F0QixRQUFBQSxJQUFJLENBQUNNLEdBQUwsQ0FBU0MsSUFBSSxDQUFDZ0IsT0FBZDtBQUNBO0FBQ0Q7O0FBRUQsVUFBSVYsUUFBUSxDQUFDLE1BQUQsRUFBUyxNQUFULEVBQWlCLE9BQWpCLENBQVosRUFBdUM7QUFDckNiLFFBQUFBLElBQUksQ0FBQ00sR0FBTCxDQUFTQyxJQUFJLENBQUNZLFFBQWQ7QUFDQW5CLFFBQUFBLElBQUksQ0FBQ00sR0FBTCxDQUFTQyxJQUFJLENBQUNhLFVBQWQ7QUFDQXBCLFFBQUFBLElBQUksQ0FBQ00sR0FBTCxDQUFTQyxJQUFJLENBQUNjLGFBQWQ7QUFDQXJCLFFBQUFBLElBQUksQ0FBQ00sR0FBTCxDQUFTQyxJQUFJLENBQUNlLGVBQWQ7QUFDQXRCLFFBQUFBLElBQUksQ0FBQ00sR0FBTCxDQUFTQyxJQUFJLENBQUNnQixPQUFkO0FBQ0E7QUFDRDs7QUFFRCxVQUFJVixRQUFRLENBQUMsTUFBRCxFQUFTLE1BQVQsRUFBaUIsU0FBakIsQ0FBWixFQUF5QztBQUN2Q2IsUUFBQUEsSUFBSSxDQUFDTSxHQUFMLENBQVNDLElBQUksQ0FBQ2lCLE9BQWQ7QUFDQXhCLFFBQUFBLElBQUksQ0FBQ00sR0FBTCxDQUFTQyxJQUFJLENBQUNDLFlBQWQ7QUFDQVIsUUFBQUEsSUFBSSxDQUFDTSxHQUFMLENBQVNDLElBQUksQ0FBQ2UsZUFBZDtBQUNBO0FBQ0Q7O0FBRUQsVUFBSSwrQkFBaUJsQixRQUFqQixFQUEyQixNQUEzQixFQUFtQyxRQUFuQyxDQUFKLEVBQWtEO0FBQ2hESixRQUFBQSxJQUFJLENBQUNNLEdBQUwsQ0FBU0MsSUFBSSxDQUFDaUIsT0FBZDtBQUNBeEIsUUFBQUEsSUFBSSxDQUFDTSxHQUFMLENBQVNDLElBQUksQ0FBQ2tCLE1BQUwsQ0FBWVIsR0FBckI7QUFDQWpCLFFBQUFBLElBQUksQ0FBQ00sR0FBTCxDQUFTQyxJQUFJLENBQUNDLFlBQWQ7QUFDQTtBQUNELE9BcERvQyxDQXNEckM7OztBQUNBLFlBQU1rQixZQUFZLEdBQUczQixpQkFBSzRCLFFBQUwsQ0FBYyxLQUFLNUQsT0FBTCxFQUFkLEVBQThCcUMsUUFBOUIsQ0FBckI7O0FBQ0EsV0FBSyxNQUFNd0IsR0FBWCxJQUFrQnJCLElBQUksQ0FBQ0csU0FBTCxDQUFlbUIsZ0JBQWYsQ0FBZ0MsQ0FBQ0gsWUFBRCxDQUFoQyxFQUFnRCxDQUFDO0FBQUNkLFFBQUFBLE1BQU0sRUFBRTtBQUFULE9BQUQsQ0FBaEQsQ0FBbEIsRUFBc0Y7QUFDcEZaLFFBQUFBLElBQUksQ0FBQ00sR0FBTCxDQUFTc0IsR0FBVDtBQUNEOztBQUNENUIsTUFBQUEsSUFBSSxDQUFDTSxHQUFMLENBQVNDLElBQUksQ0FBQ0MsWUFBZDtBQUNEO0FBRUQ7OztBQUNBLFFBQUlSLElBQUksQ0FBQzhCLElBQUwsR0FBWSxDQUFoQixFQUFtQjtBQUNqQixXQUFLdkUsS0FBTCxDQUFXaUMsVUFBWCxDQUFzQnVDLEtBQUssQ0FBQ0MsSUFBTixDQUFXaEMsSUFBWCxDQUF0QjtBQUNBLFdBQUs3QixTQUFMO0FBQ0Q7QUFDRjs7QUFFRDhELEVBQUFBLG9CQUFvQixHQUFHO0FBQ3JCLFFBQUksS0FBSzdELGFBQUwsQ0FBbUI4RCxJQUFuQixPQUE4QixFQUFsQyxFQUFzQztBQUNwQyxhQUFPLElBQVA7QUFDRCxLQUZELE1BRU8sSUFBSSxLQUFLN0QscUJBQVQsRUFBZ0M7QUFDckMsYUFBTyxLQUFLRCxhQUFMLEtBQXVCLEtBQUtDLHFCQUFuQztBQUNEOztBQUNELFdBQU8sS0FBUDtBQUNEOztBQUVELFFBQU04RCx3Q0FBTixDQUErQ3pDLE1BQS9DLEVBQXVEO0FBQ3JELFNBQUssSUFBSVEsQ0FBQyxHQUFHLENBQWIsRUFBZ0JBLENBQUMsR0FBR1IsTUFBTSxDQUFDUyxNQUEzQixFQUFtQ0QsQ0FBQyxFQUFwQyxFQUF3QztBQUN0QyxZQUFNa0MsS0FBSyxHQUFHMUMsTUFBTSxDQUFDUSxDQUFELENBQXBCOztBQUVBLFVBQUksQ0FBQ2tDLEtBQUssQ0FBQ3JDLElBQVgsRUFBaUI7QUFDZjtBQUNEOztBQUVELFVBQUksK0JBQWlCcUMsS0FBSyxDQUFDckMsSUFBdkIsRUFBNkIsTUFBN0IsRUFBcUMsWUFBckMsQ0FBSixFQUF3RDtBQUN0RCxZQUFJcUMsS0FBSyxDQUFDQyxNQUFOLEtBQWlCLFNBQXJCLEVBQWdDO0FBQzlCLGNBQUksS0FBS0osb0JBQUwsRUFBSixFQUFpQztBQUMvQixpQkFBS3pELGdCQUFMLEVBQXNCLE1BQU0sS0FBS25CLFVBQUwsQ0FBZ0J5QixlQUFoQixFQUE1QjtBQUNEO0FBQ0YsU0FKRCxNQUlPLElBQUlzRCxLQUFLLENBQUNDLE1BQU4sS0FBaUIsU0FBckIsRUFBZ0M7QUFDckMsZUFBSzdELGdCQUFMLENBQXNCLEtBQUtILHFCQUFMLElBQThCLEVBQXBEO0FBQ0Q7QUFDRjs7QUFFRCxVQUFJLCtCQUFpQitELEtBQUssQ0FBQ3JDLElBQXZCLEVBQTZCLE1BQTdCLEVBQXFDLFFBQXJDLENBQUosRUFBb0Q7QUFDbEQ7QUFDQSxjQUFNbkIsUUFBUSxHQUFHLE1BQU0sS0FBS0csMEJBQUwsRUFBdkI7O0FBQ0EsWUFBSUgsUUFBUSxLQUFLLElBQWpCLEVBQXVCO0FBQ3JCLGVBQUtKLGdCQUFMLENBQXNCLEVBQXRCO0FBQ0QsU0FGRCxNQUVPLElBQUksS0FBS0gscUJBQUwsS0FBK0JPLFFBQW5DLEVBQTZDO0FBQ2xELGVBQUtKLGdCQUFMLENBQXNCSSxRQUF0QjtBQUNEOztBQUNELGFBQUtELHdCQUFMLENBQThCQyxRQUE5QjtBQUNEO0FBQ0Y7QUFDRjs7QUFFRDBELEVBQUFBLHVCQUF1QixDQUFDNUMsTUFBRCxFQUFTO0FBQzlCLFNBQUtELG9DQUFMLENBQTBDQyxNQUExQztBQUNBLFNBQUt5Qyx3Q0FBTCxDQUE4Q3pDLE1BQTlDO0FBQ0Q7O0FBRUQ2QyxFQUFBQSxPQUFPLEdBQUc7QUFDUixTQUFLaEYsS0FBTCxDQUFXaUYsS0FBWDtBQUNBLFNBQUtyRSxTQUFMO0FBQ0Q7O0FBRURzRSxFQUFBQSxJQUFJLEdBQUc7QUFDTCxXQUFPLE1BQU1BLElBQU4sWUFBbUI1QyxDQUFDLElBQUk7QUFDN0JBLE1BQUFBLENBQUMsQ0FBQzZDLE1BQUYsR0FBVyxrREFBWDtBQUNBLGFBQU9DLE9BQU8sQ0FBQ0MsTUFBUixDQUFlL0MsQ0FBZixDQUFQO0FBQ0QsS0FITSxDQUFQO0FBSUQ7O0FBRURnRCxFQUFBQSxLQUFLLEdBQUc7QUFDTixXQUFPLE1BQU1BLEtBQU4sWUFBb0JoRCxDQUFDLElBQUk7QUFDOUJBLE1BQUFBLENBQUMsQ0FBQzZDLE1BQUYsR0FBVyxrREFBWDtBQUNBLGFBQU9DLE9BQU8sQ0FBQ0MsTUFBUixDQUFlL0MsQ0FBZixDQUFQO0FBQ0QsS0FITSxDQUFQO0FBSUQsR0F0TndDLENBd056QztBQUVBOzs7QUFFQWlELEVBQUFBLFVBQVUsQ0FBQ25ELEtBQUQsRUFBUTtBQUNoQixXQUFPLEtBQUtILFVBQUwsQ0FDTCxNQUFNZSxJQUFJLENBQUN3QyxrQkFBTCxDQUF3QnBELEtBQXhCLENBREQsRUFFTCxNQUFNLEtBQUtWLEdBQUwsR0FBVzZELFVBQVgsQ0FBc0JuRCxLQUF0QixDQUZELENBQVA7QUFJRDs7QUFFRHFELEVBQUFBLFlBQVksQ0FBQ3JELEtBQUQsRUFBUTtBQUNsQixXQUFPLEtBQUtILFVBQUwsQ0FDTCxNQUFNZSxJQUFJLENBQUN3QyxrQkFBTCxDQUF3QnBELEtBQXhCLENBREQsRUFFTCxNQUFNLEtBQUtWLEdBQUwsR0FBVytELFlBQVgsQ0FBd0JyRCxLQUF4QixDQUZELENBQVA7QUFJRDs7QUFFRHNELEVBQUFBLDBCQUEwQixDQUFDdEQsS0FBRCxFQUFRO0FBQ2hDLFdBQU8sS0FBS0gsVUFBTCxDQUNMLE1BQU1lLElBQUksQ0FBQ3dDLGtCQUFMLENBQXdCcEQsS0FBeEIsQ0FERCxFQUVMLE1BQU0sS0FBS1YsR0FBTCxHQUFXK0QsWUFBWCxDQUF3QnJELEtBQXhCLEVBQStCLE9BQS9CLENBRkQsQ0FBUDtBQUlEOztBQUVEdUQsRUFBQUEsbUJBQW1CLENBQUNDLFFBQUQsRUFBV0MsUUFBWCxFQUFxQjtBQUN0QyxXQUFPLEtBQUs1RCxVQUFMLENBQ0wsTUFBTWUsSUFBSSxDQUFDd0Msa0JBQUwsQ0FBd0IsQ0FBQ0ksUUFBRCxDQUF4QixDQURELEVBRUwsTUFBTSxLQUFLbEUsR0FBTCxHQUFXaUUsbUJBQVgsQ0FBK0JDLFFBQS9CLEVBQXlDQyxRQUF6QyxDQUZELENBQVA7QUFJRDs7QUFFREMsRUFBQUEsc0JBQXNCLENBQUNGLFFBQUQsRUFBVztBQUMvQixXQUFPLEtBQUszRCxVQUFMLENBQ0wsTUFBTWUsSUFBSSxDQUFDd0Msa0JBQUwsQ0FBd0IsQ0FBQ0ksUUFBRCxDQUF4QixDQURELEVBRUwsTUFBTSxLQUFLbEUsR0FBTCxHQUFXb0Usc0JBQVgsQ0FBa0NGLFFBQWxDLENBRkQsQ0FBUDtBQUlEOztBQUVERyxFQUFBQSxpQkFBaUIsQ0FBQ0MsY0FBRCxFQUFpQjtBQUNoQyxXQUFPLEtBQUsvRCxVQUFMLENBQ0wsTUFBTWUsSUFBSSxDQUFDd0Msa0JBQUwsQ0FBd0JoQixLQUFLLENBQUNDLElBQU4sQ0FBV3VCLGNBQWMsQ0FBQ0MsVUFBZixFQUFYLENBQXhCLENBREQsRUFFTCxNQUFNO0FBQ0osWUFBTUMsUUFBUSxHQUFHRixjQUFjLENBQUNHLFFBQWYsRUFBakI7QUFDQSxhQUFPLEtBQUt6RSxHQUFMLEdBQVcwRSxVQUFYLENBQXNCRixRQUF0QixFQUFnQztBQUFDdkMsUUFBQUEsS0FBSyxFQUFFO0FBQVIsT0FBaEMsQ0FBUDtBQUNELEtBTEksQ0FBUDtBQU9EOztBQUVEMEMsRUFBQUEsbUJBQW1CLENBQUNMLGNBQUQsRUFBaUI7QUFDbEMsV0FBTyxLQUFLL0QsVUFBTCxDQUNMLE1BQU1lLElBQUksQ0FBQ3NELG9CQUFMLENBQTBCOUIsS0FBSyxDQUFDQyxJQUFOLENBQVd1QixjQUFjLENBQUNDLFVBQWYsRUFBWCxDQUExQixDQURELEVBRUwsTUFBTTtBQUNKLFlBQU1DLFFBQVEsR0FBR0YsY0FBYyxDQUFDRyxRQUFmLEVBQWpCO0FBQ0EsYUFBTyxLQUFLekUsR0FBTCxHQUFXMEUsVUFBWCxDQUFzQkYsUUFBdEIsQ0FBUDtBQUNELEtBTEksQ0FBUDtBQU9ELEdBalJ3QyxDQW1SekM7OztBQUVBSyxFQUFBQSxNQUFNLENBQUNyRixPQUFELEVBQVVzRixPQUFWLEVBQW1CO0FBQ3ZCLFdBQU8sS0FBS3ZFLFVBQUwsQ0FDTGUsSUFBSSxDQUFDeUQsaUJBREEsRUFFTDtBQUNBLFVBQU0sS0FBS0MscUJBQUwsQ0FBMkIsUUFBM0IsRUFBcUMsT0FBT3hGLE9BQVAsRUFBZ0JzRixPQUFPLEdBQUcsRUFBMUIsS0FBaUM7QUFDMUUsWUFBTUcsU0FBUyxHQUFHSCxPQUFPLENBQUNHLFNBQTFCO0FBQ0EsWUFBTUMsSUFBSSxHQUFHLENBQUNELFNBQUQsR0FBYUgsT0FBYixxQkFDUkEsT0FEUTtBQUVYRyxRQUFBQSxTQUFTLEVBQUVBLFNBQVMsQ0FBQ3RFLEdBQVYsQ0FBY3dFLE1BQU0sSUFBSTtBQUNqQyxpQkFBTztBQUFDQyxZQUFBQSxLQUFLLEVBQUVELE1BQU0sQ0FBQ0UsUUFBUCxFQUFSO0FBQTJCQyxZQUFBQSxJQUFJLEVBQUVILE1BQU0sQ0FBQ0ksV0FBUDtBQUFqQyxXQUFQO0FBQ0QsU0FGVTtBQUZBLFFBQWI7QUFPQSxZQUFNLEtBQUt2RixHQUFMLEdBQVc2RSxNQUFYLENBQWtCckYsT0FBbEIsRUFBMkIwRixJQUEzQixDQUFOLENBVDBFLENBVzFFO0FBQ0E7QUFDQTs7QUFDQSxZQUFNO0FBQUNNLFFBQUFBLGFBQUQ7QUFBZ0JDLFFBQUFBO0FBQWhCLFVBQXNDLE1BQU0sS0FBS0MsMEJBQUwsRUFBbEQ7QUFDQSxZQUFNQyxhQUFhLEdBQUdDLE1BQU0sQ0FBQzdFLElBQVAsbUJBQWdCeUUsYUFBaEIsRUFBa0NDLGtCQUFsQyxHQUF1RHZFLE1BQTdFO0FBQ0EsbUNBQVMsUUFBVCxFQUFtQjtBQUNqQixtQkFBUyxRQURRO0FBRWpCMkUsUUFBQUEsT0FBTyxFQUFFRixhQUFhLEdBQUcsQ0FGUjtBQUdqQkcsUUFBQUEsS0FBSyxFQUFFLENBQUMsQ0FBQ2hCLE9BQU8sQ0FBQ2dCLEtBSEE7QUFJakJDLFFBQUFBLGFBQWEsRUFBRWQsU0FBUyxHQUFHQSxTQUFTLENBQUMvRCxNQUFiLEdBQXNCO0FBSjdCLE9BQW5CO0FBTUQsS0F0QkssRUFzQkgxQixPQXRCRyxFQXNCTXNGLE9BdEJOLENBSEQsQ0FBUDtBQTJCRCxHQWpUd0MsQ0FtVHpDOzs7QUFFQWtCLEVBQUFBLEtBQUssQ0FBQ0MsVUFBRCxFQUFhO0FBQ2hCLFdBQU8sS0FBSzFGLFVBQUwsQ0FDTCxNQUFNLENBQ0osR0FBR2UsSUFBSSxDQUFDeUQsaUJBQUwsRUFEQyxFQUVKekQsSUFBSSxDQUFDVyxLQUFMLENBQVdELEdBRlAsRUFHSlYsSUFBSSxDQUFDZSxlQUhELENBREQsRUFNTCxNQUFNLEtBQUtyQyxHQUFMLEdBQVdnRyxLQUFYLENBQWlCQyxVQUFqQixDQU5ELENBQVA7QUFRRDs7QUFFREMsRUFBQUEsVUFBVSxHQUFHO0FBQ1gsV0FBTyxLQUFLM0YsVUFBTCxDQUNMLE1BQU0sQ0FDSmUsSUFBSSxDQUFDQyxZQURELEVBRUpELElBQUksQ0FBQ1MsYUFGRCxFQUdKVCxJQUFJLENBQUNHLFNBQUwsQ0FBZU8sR0FIWCxFQUlKVixJQUFJLENBQUNXLEtBQUwsQ0FBV0QsR0FKUCxDQURELEVBT0wsWUFBWTtBQUNWLFlBQU0sS0FBS2hDLEdBQUwsR0FBV2tHLFVBQVgsRUFBTjtBQUNBLFdBQUszRyxnQkFBTCxDQUFzQixLQUFLSCxxQkFBTCxJQUE4QixFQUFwRDtBQUNELEtBVkksQ0FBUDtBQVlEOztBQUVEK0csRUFBQUEsWUFBWSxDQUFDQyxJQUFELEVBQU8xRixLQUFQLEVBQWM7QUFDeEIsV0FBTyxLQUFLVixHQUFMLEdBQVdtRyxZQUFYLENBQXdCQyxJQUF4QixFQUE4QjFGLEtBQTlCLENBQVA7QUFDRDs7QUFFRDdCLEVBQUFBLFNBQVMsQ0FBQ3dILFFBQUQsRUFBV0MsY0FBWCxFQUEyQkMsVUFBM0IsRUFBdUNDLFVBQXZDLEVBQW1EO0FBQzFELFdBQU8sS0FBS3hHLEdBQUwsR0FBV25CLFNBQVgsQ0FBcUJ3SCxRQUFyQixFQUErQkMsY0FBL0IsRUFBK0NDLFVBQS9DLEVBQTJEQyxVQUEzRCxDQUFQO0FBQ0Q7O0FBRURDLEVBQUFBLHlCQUF5QixDQUFDdkMsUUFBRCxFQUFXd0MsYUFBWCxFQUEwQkMsT0FBMUIsRUFBbUNDLFNBQW5DLEVBQThDO0FBQ3JFLFdBQU8sS0FBS3JHLFVBQUwsQ0FDTCxNQUFNLENBQ0plLElBQUksQ0FBQ0MsWUFERCxFQUVKRCxJQUFJLENBQUNTLGFBRkQsRUFHSixHQUFHVCxJQUFJLENBQUNHLFNBQUwsQ0FBZW1CLGdCQUFmLENBQWdDLENBQUNzQixRQUFELENBQWhDLEVBQTRDLENBQUM7QUFBQ3ZDLE1BQUFBLE1BQU0sRUFBRTtBQUFULEtBQUQsRUFBa0I7QUFBQ0EsTUFBQUEsTUFBTSxFQUFFO0FBQVQsS0FBbEIsQ0FBNUMsQ0FIQyxFQUlKTCxJQUFJLENBQUNXLEtBQUwsQ0FBVzRFLE9BQVgsQ0FBbUIzQyxRQUFuQixDQUpJLENBREQsRUFPTCxNQUFNLEtBQUtsRSxHQUFMLEdBQVd5Ryx5QkFBWCxDQUFxQ3ZDLFFBQXJDLEVBQStDd0MsYUFBL0MsRUFBOERDLE9BQTlELEVBQXVFQyxTQUF2RSxDQVBELENBQVA7QUFTRCxHQWpXd0MsQ0FtV3pDOzs7QUFFQUUsRUFBQUEsUUFBUSxDQUFDQyxRQUFELEVBQVdqQyxPQUFPLEdBQUcsRUFBckIsRUFBeUI7QUFDL0IsV0FBTyxLQUFLdkUsVUFBTCxDQUNMLE1BQU0sQ0FDSmUsSUFBSSxDQUFDUyxhQURELEVBRUpULElBQUksQ0FBQ2EsVUFGRCxFQUdKYixJQUFJLENBQUNjLGFBSEQsRUFJSmQsSUFBSSxDQUFDZ0IsT0FKRCxFQUtKaEIsSUFBSSxDQUFDQyxZQUxELEVBTUpELElBQUksQ0FBQ1csS0FBTCxDQUFXRCxHQU5QLEVBT0osR0FBR1YsSUFBSSxDQUFDRyxTQUFMLENBQWVDLFlBQWYsQ0FBNEI7QUFBQ0MsTUFBQUEsTUFBTSxFQUFFO0FBQVQsS0FBNUIsQ0FQQyxFQVFKTCxJQUFJLENBQUNHLFNBQUwsQ0FBZXVGLGlCQVJYLEVBU0oxRixJQUFJLENBQUNlLGVBVEQsRUFVSmYsSUFBSSxDQUFDWSxRQVZELENBREQsRUFhTDtBQUNBLFVBQU0sS0FBSzhDLHFCQUFMLENBQTJCLFVBQTNCLEVBQXVDLENBQUMrQixRQUFELEVBQVdqQyxPQUFYLEtBQXVCO0FBQ2xFLGFBQU8sS0FBSzlFLEdBQUwsR0FBVzhHLFFBQVgsQ0FBb0JDLFFBQXBCLEVBQThCakMsT0FBOUIsQ0FBUDtBQUNELEtBRkssRUFFSGlDLFFBRkcsRUFFT2pDLE9BRlAsQ0FkRCxDQUFQO0FBa0JEOztBQUVEbUMsRUFBQUEsdUJBQXVCLENBQUN2RyxLQUFELEVBQVFxRyxRQUFRLEdBQUcsTUFBbkIsRUFBMkI7QUFDaEQsV0FBTyxLQUFLeEcsVUFBTCxDQUNMLE1BQU0sQ0FDSmUsSUFBSSxDQUFDQyxZQURELEVBRUpELElBQUksQ0FBQ1MsYUFGRCxFQUdKLEdBQUdyQixLQUFLLENBQUNDLEdBQU4sQ0FBVXVHLFFBQVEsSUFBSTVGLElBQUksQ0FBQ1csS0FBTCxDQUFXNEUsT0FBWCxDQUFtQkssUUFBbkIsQ0FBdEIsQ0FIQyxFQUlKLEdBQUc1RixJQUFJLENBQUNHLFNBQUwsQ0FBZW1CLGdCQUFmLENBQWdDbEMsS0FBaEMsRUFBdUMsQ0FBQztBQUFDaUIsTUFBQUEsTUFBTSxFQUFFO0FBQVQsS0FBRCxDQUF2QyxDQUpDLEVBS0osR0FBR0wsSUFBSSxDQUFDRyxTQUFMLENBQWUwRixvQkFBZixDQUFvQ3pHLEtBQXBDLENBTEMsQ0FERCxFQVFMLE1BQU0sS0FBS1YsR0FBTCxHQUFXb0gsYUFBWCxDQUF5QjFHLEtBQXpCLEVBQWdDcUcsUUFBaEMsQ0FSRCxDQUFQO0FBVUQsR0FyWXdDLENBdVl6Qzs7O0FBRUFNLEVBQUFBLGNBQWMsR0FBRztBQUNmLFdBQU8sS0FBSzlHLFVBQUwsQ0FDTCxNQUFNLENBQ0plLElBQUksQ0FBQ1MsYUFERCxFQUVKVCxJQUFJLENBQUNhLFVBRkQsRUFHSmIsSUFBSSxDQUFDYyxhQUhELEVBSUpkLElBQUksQ0FBQ2dCLE9BSkQsRUFLSmhCLElBQUksQ0FBQ0MsWUFMRCxFQU1KRCxJQUFJLENBQUNXLEtBQUwsQ0FBV0QsR0FOUCxFQU9KLEdBQUdWLElBQUksQ0FBQ0csU0FBTCxDQUFlQyxZQUFmLENBQTRCO0FBQUNDLE1BQUFBLE1BQU0sRUFBRTtBQUFULEtBQTVCLENBUEMsRUFRSkwsSUFBSSxDQUFDZSxlQVJELENBREQsRUFXTCxZQUFZO0FBQ1YsVUFBSTtBQUNGLGNBQU0sS0FBS3JDLEdBQUwsR0FBV3NILEtBQVgsQ0FBaUIsTUFBakIsRUFBeUIsT0FBekIsQ0FBTjtBQUNBLHFDQUFTLGtCQUFULEVBQTZCO0FBQUMscUJBQVM7QUFBVixTQUE3QjtBQUNELE9BSEQsQ0FHRSxPQUFPMUcsQ0FBUCxFQUFVO0FBQ1YsWUFBSSxtQkFBbUIyRyxJQUFuQixDQUF3QjNHLENBQUMsQ0FBQzZDLE1BQTFCLENBQUosRUFBdUM7QUFDckM7QUFDQSxnQkFBTSxLQUFLekQsR0FBTCxHQUFXd0gsU0FBWCxDQUFxQixNQUFyQixDQUFOO0FBQ0QsU0FIRCxNQUdPO0FBQ0wsZ0JBQU01RyxDQUFOO0FBQ0Q7QUFDRjtBQUNGLEtBdkJJLENBQVA7QUF5QkQsR0FuYXdDLENBcWF6Qzs7O0FBRUE2RyxFQUFBQSxLQUFLLENBQUN4QixVQUFELEVBQWFuQixPQUFPLEdBQUcsRUFBdkIsRUFBMkI7QUFDOUIsV0FBTyxLQUFLdkUsVUFBTCxDQUNMLE1BQU0sQ0FDSmUsSUFBSSxDQUFDQyxZQURELEVBRUpELElBQUksQ0FBQ2UsZUFGRCxDQURELEVBS0w7QUFDQSxVQUFNLEtBQUsyQyxxQkFBTCxDQUEyQixPQUEzQixFQUFvQyxNQUFNaUIsVUFBTixJQUFvQjtBQUM1RCxVQUFJeUIsZUFBZSxHQUFHNUMsT0FBTyxDQUFDNkMsVUFBOUI7O0FBQ0EsVUFBSSxDQUFDRCxlQUFMLEVBQXNCO0FBQ3BCLGNBQU1FLE1BQU0sR0FBRyxNQUFNLEtBQUtDLGtCQUFMLENBQXdCNUIsVUFBeEIsQ0FBckI7O0FBQ0EsWUFBSSxDQUFDMkIsTUFBTSxDQUFDMUgsU0FBUCxFQUFMLEVBQXlCO0FBQ3ZCLGlCQUFPLElBQVA7QUFDRDs7QUFDRHdILFFBQUFBLGVBQWUsR0FBR0UsTUFBTSxDQUFDRSxPQUFQLEVBQWxCO0FBQ0Q7O0FBQ0QsYUFBTyxLQUFLOUgsR0FBTCxHQUFXeUgsS0FBWCxDQUFpQkMsZUFBakIsRUFBa0N6QixVQUFsQyxDQUFQO0FBQ0QsS0FWSyxFQVVIQSxVQVZHLENBTkQsQ0FBUDtBQWtCRDs7QUFFRDhCLEVBQUFBLElBQUksQ0FBQzlCLFVBQUQsRUFBYW5CLE9BQU8sR0FBRyxFQUF2QixFQUEyQjtBQUM3QixXQUFPLEtBQUt2RSxVQUFMLENBQ0wsTUFBTSxDQUNKLEdBQUdlLElBQUksQ0FBQ3lELGlCQUFMLEVBREMsRUFFSnpELElBQUksQ0FBQ1csS0FBTCxDQUFXRCxHQUZQLEVBR0pWLElBQUksQ0FBQ2UsZUFIRCxFQUlKZixJQUFJLENBQUNZLFFBSkQsQ0FERCxFQU9MO0FBQ0EsVUFBTSxLQUFLOEMscUJBQUwsQ0FBMkIsTUFBM0IsRUFBbUMsTUFBTWlCLFVBQU4sSUFBb0I7QUFDM0QsVUFBSXlCLGVBQWUsR0FBRzVDLE9BQU8sQ0FBQzZDLFVBQTlCOztBQUNBLFVBQUksQ0FBQ0QsZUFBTCxFQUFzQjtBQUNwQixjQUFNRSxNQUFNLEdBQUcsTUFBTSxLQUFLQyxrQkFBTCxDQUF3QjVCLFVBQXhCLENBQXJCOztBQUNBLFlBQUksQ0FBQzJCLE1BQU0sQ0FBQzFILFNBQVAsRUFBTCxFQUF5QjtBQUN2QixpQkFBTyxJQUFQO0FBQ0Q7O0FBQ0R3SCxRQUFBQSxlQUFlLEdBQUdFLE1BQU0sQ0FBQ0UsT0FBUCxFQUFsQjtBQUNEOztBQUNELGFBQU8sS0FBSzlILEdBQUwsR0FBVytILElBQVgsQ0FBZ0JMLGVBQWhCLEVBQWlDekIsVUFBakMsRUFBNkNuQixPQUE3QyxDQUFQO0FBQ0QsS0FWSyxFQVVIbUIsVUFWRyxDQVJELENBQVA7QUFvQkQ7O0FBRUQrQixFQUFBQSxJQUFJLENBQUMvQixVQUFELEVBQWFuQixPQUFPLEdBQUcsRUFBdkIsRUFBMkI7QUFDN0IsV0FBTyxLQUFLdkUsVUFBTCxDQUNMLE1BQU07QUFDSixZQUFNUSxJQUFJLEdBQUcsQ0FDWE8sSUFBSSxDQUFDQyxZQURNLEVBRVhELElBQUksQ0FBQ2UsZUFGTSxDQUFiOztBQUtBLFVBQUl5QyxPQUFPLENBQUNtRCxXQUFaLEVBQXlCO0FBQ3ZCbEgsUUFBQUEsSUFBSSxDQUFDaUgsSUFBTCxDQUFVMUcsSUFBSSxDQUFDWSxRQUFmO0FBQ0FuQixRQUFBQSxJQUFJLENBQUNpSCxJQUFMLENBQVUsR0FBRzFHLElBQUksQ0FBQ2tCLE1BQUwsQ0FBWTBGLGVBQVosQ0FBNkIsVUFBU2pDLFVBQVcsU0FBakQsQ0FBYjtBQUNEOztBQUVELGFBQU9sRixJQUFQO0FBQ0QsS0FiSSxFQWNMO0FBQ0EsVUFBTSxLQUFLaUUscUJBQUwsQ0FBMkIsTUFBM0IsRUFBbUMsT0FBT2lCLFVBQVAsRUFBbUJuQixPQUFuQixLQUErQjtBQUN0RSxZQUFNOEMsTUFBTSxHQUFHOUMsT0FBTyxDQUFDOEMsTUFBUixLQUFrQixNQUFNLEtBQUtDLGtCQUFMLENBQXdCNUIsVUFBeEIsQ0FBeEIsQ0FBZjtBQUNBLGFBQU8sS0FBS2pHLEdBQUwsR0FBV2dJLElBQVgsQ0FBZ0JKLE1BQU0sQ0FBQ08sU0FBUCxDQUFpQixRQUFqQixDQUFoQixFQUE0Q2xDLFVBQTVDLEVBQXdEbkIsT0FBeEQsQ0FBUDtBQUNELEtBSEssRUFHSG1CLFVBSEcsRUFHU25CLE9BSFQsQ0FmRCxDQUFQO0FBb0JELEdBeGV3QyxDQTBlekM7OztBQUVBc0QsRUFBQUEsU0FBUyxDQUFDQyxPQUFELEVBQVVDLEtBQVYsRUFBaUJ4RCxPQUFqQixFQUEwQjtBQUNqQyxXQUFPLEtBQUt2RSxVQUFMLENBQ0wsTUFBTWUsSUFBSSxDQUFDa0IsTUFBTCxDQUFZMEYsZUFBWixDQUE0QkcsT0FBNUIsQ0FERCxFQUVMLE1BQU0sS0FBS3JJLEdBQUwsR0FBV29JLFNBQVgsQ0FBcUJDLE9BQXJCLEVBQThCQyxLQUE5QixFQUFxQ3hELE9BQXJDLENBRkQsQ0FBUDtBQUlEOztBQUVEeUQsRUFBQUEsV0FBVyxDQUFDRixPQUFELEVBQVU7QUFDbkIsV0FBTyxLQUFLOUgsVUFBTCxDQUNMLE1BQU1lLElBQUksQ0FBQ2tCLE1BQUwsQ0FBWTBGLGVBQVosQ0FBNEJHLE9BQTVCLENBREQsRUFFTCxNQUFNLEtBQUtySSxHQUFMLEdBQVd1SSxXQUFYLENBQXVCRixPQUF2QixDQUZELENBQVA7QUFJRCxHQXhmd0MsQ0EwZnpDOzs7QUFFQTNKLEVBQUFBLFVBQVUsQ0FBQ29HLE9BQUQsRUFBVTtBQUNsQixXQUFPLEtBQUs5RSxHQUFMLEdBQVd0QixVQUFYLENBQXNCb0csT0FBdEIsQ0FBUDtBQUNEOztBQUVEbEcsRUFBQUEsZ0JBQWdCLENBQUM0SixXQUFELEVBQWNDLEdBQWQsRUFBbUI7QUFDakMsV0FBTyxLQUFLekksR0FBTCxHQUFXcEIsZ0JBQVgsQ0FBNEI0SixXQUE1QixFQUF5Q0MsR0FBekMsQ0FBUDtBQUNELEdBbGdCd0MsQ0FvZ0J6Qzs7O0FBRUFDLEVBQUFBLHdCQUF3QixHQUFHO0FBQ3pCLFdBQU8sS0FBS2xLLGNBQUwsQ0FBb0JtSyxpQkFBcEIsRUFBUDtBQUNEOztBQUVELFFBQU1DLG9CQUFOLEdBQTZCO0FBQzNCLFVBQU12SyxPQUFPLEdBQUcsTUFBTSxLQUFLd0ssa0JBQUwsRUFBdEI7QUFDQSxTQUFLckssY0FBTCxDQUFvQmMsYUFBcEIsQ0FBa0NqQixPQUFsQztBQUNEOztBQUVELFFBQU15Syx3QkFBTixDQUErQkMsU0FBL0IsRUFBMENDLE1BQTFDLEVBQWtEQyxpQkFBbEQsRUFBcUVDLHNCQUFzQixHQUFHLElBQTlGLEVBQW9HO0FBQ2xHLFVBQU1DLFNBQVMsR0FBRyxNQUFNLEtBQUszSyxjQUFMLENBQW9Cc0ssd0JBQXBCLENBQ3RCQyxTQURzQixFQUV0QkMsTUFGc0IsRUFHdEJDLGlCQUhzQixFQUl0QkMsc0JBSnNCLENBQXhCO0FBTUE7O0FBQ0EsUUFBSUMsU0FBSixFQUFlO0FBQ2IsWUFBTSxLQUFLQyxrQkFBTCxFQUFOO0FBQ0Q7O0FBQ0QsV0FBT0QsU0FBUDtBQUNEOztBQUVERSxFQUFBQSw2QkFBNkIsQ0FBQ0wsTUFBRCxFQUFTRSxzQkFBc0IsR0FBRyxJQUFsQyxFQUF3QztBQUNuRSxXQUFPLEtBQUsxSyxjQUFMLENBQW9CNkssNkJBQXBCLENBQWtETCxNQUFsRCxFQUEwREUsc0JBQTFELENBQVA7QUFDRDs7QUFFRCxRQUFNSSxpQkFBTixDQUF3Qkosc0JBQXNCLEdBQUcsSUFBakQsRUFBdUQ7QUFDckQsVUFBTUssT0FBTyxHQUFHLE1BQU0sS0FBSy9LLGNBQUwsQ0FBb0JnTCxVQUFwQixDQUErQk4sc0JBQS9CLENBQXRCOztBQUNBLFFBQUlLLE9BQUosRUFBYTtBQUNYLFlBQU0sS0FBS0gsa0JBQUwsRUFBTjtBQUNEO0FBQ0Y7O0FBRURLLEVBQUFBLG1CQUFtQixDQUFDUCxzQkFBc0IsR0FBRyxJQUExQixFQUFnQztBQUNqRCxTQUFLMUssY0FBTCxDQUFvQmtMLFlBQXBCLENBQWlDUixzQkFBakM7QUFDQSxXQUFPLEtBQUtFLGtCQUFMLEVBQVA7QUFDRDs7QUFFRE8sRUFBQUEsNkJBQTZCLENBQUNqSixLQUFELEVBQVE7QUFDbkMsV0FBTyxLQUFLSCxVQUFMLENBQ0wsTUFBTSxDQUNKZSxJQUFJLENBQUNDLFlBREQsRUFFSixHQUFHYixLQUFLLENBQUNDLEdBQU4sQ0FBVXVELFFBQVEsSUFBSTVDLElBQUksQ0FBQ0csU0FBTCxDQUFlb0YsT0FBZixDQUF1QjNDLFFBQXZCLEVBQWlDO0FBQUN2QyxNQUFBQSxNQUFNLEVBQUU7QUFBVCxLQUFqQyxDQUF0QixDQUZDLEVBR0osR0FBR0wsSUFBSSxDQUFDRyxTQUFMLENBQWUwRixvQkFBZixDQUFvQ3pHLEtBQXBDLENBSEMsQ0FERCxFQU1MLFlBQVk7QUFDVixZQUFNa0osY0FBYyxHQUFHLE1BQU0sS0FBSzVKLEdBQUwsR0FBVzZKLGlCQUFYLEVBQTdCO0FBQ0EsWUFBTSxDQUFDQyxhQUFELEVBQWdCQyxlQUFoQixJQUFtQ0MsU0FBUyxDQUFDdEosS0FBRCxFQUFRdUosQ0FBQyxJQUFJTCxjQUFjLENBQUNoSSxRQUFmLENBQXdCcUksQ0FBeEIsQ0FBYixDQUFsRDtBQUNBLFlBQU0sS0FBS2pLLEdBQUwsR0FBV29ILGFBQVgsQ0FBeUIyQyxlQUF6QixDQUFOO0FBQ0EsWUFBTXJHLE9BQU8sQ0FBQzFCLEdBQVIsQ0FBWThILGFBQWEsQ0FBQ25KLEdBQWQsQ0FBa0J1RCxRQUFRLElBQUk7QUFDOUMsY0FBTWdHLE9BQU8sR0FBR3BKLGlCQUFLZ0IsSUFBTCxDQUFVLEtBQUtoRCxPQUFMLEVBQVYsRUFBMEJvRixRQUExQixDQUFoQjs7QUFDQSxlQUFPaUcsb0JBQUdDLE1BQUgsQ0FBVUYsT0FBVixDQUFQO0FBQ0QsT0FIaUIsQ0FBWixDQUFOO0FBSUQsS0FkSSxDQUFQO0FBZ0JELEdBOWpCd0MsQ0Fna0J6QztBQUVBOzs7QUFFQUcsRUFBQUEsZUFBZSxHQUFHO0FBQ2hCLFdBQU8sS0FBSy9MLEtBQUwsQ0FBV2dNLFFBQVgsQ0FBb0JoSixJQUFJLENBQUNDLFlBQXpCLEVBQXVDLFlBQVk7QUFDeEQsVUFBSTtBQUNGLGNBQU1nSixNQUFNLEdBQUcsTUFBTSxLQUFLdkssR0FBTCxHQUFXcUssZUFBWCxFQUFyQjtBQUNBLGNBQU1HLE9BQU8sR0FBRyxNQUFNLEtBQUtDLGtCQUFMLENBQXdCRixNQUF4QixDQUF0QjtBQUNBQyxRQUFBQSxPQUFPLENBQUNFLE1BQVIsR0FBaUJILE1BQU0sQ0FBQ0csTUFBeEI7QUFDQSxlQUFPRixPQUFQO0FBQ0QsT0FMRCxDQUtFLE9BQU9HLEdBQVAsRUFBWTtBQUNaLFlBQUlBLEdBQUcsWUFBWUMsbUNBQW5CLEVBQW1DO0FBQ2pDLGVBQUtDLFlBQUwsQ0FBa0IsVUFBbEI7QUFDQSxpQkFBTztBQUNMSCxZQUFBQSxNQUFNLEVBQUUsRUFESDtBQUVMSSxZQUFBQSxXQUFXLEVBQUUsRUFGUjtBQUdMdEYsWUFBQUEsYUFBYSxFQUFFLEVBSFY7QUFJTEMsWUFBQUEsa0JBQWtCLEVBQUU7QUFKZixXQUFQO0FBTUQsU0FSRCxNQVFPO0FBQ0wsZ0JBQU1rRixHQUFOO0FBQ0Q7QUFDRjtBQUNGLEtBbkJNLENBQVA7QUFvQkQ7O0FBRUQsUUFBTUYsa0JBQU4sQ0FBeUI7QUFBQ00sSUFBQUEsY0FBRDtBQUFpQkMsSUFBQUEsZ0JBQWpCO0FBQW1DQyxJQUFBQSxjQUFuQztBQUFtREMsSUFBQUE7QUFBbkQsR0FBekIsRUFBOEY7QUFDNUYsVUFBTUMsU0FBUyxHQUFHO0FBQ2hCQyxNQUFBQSxDQUFDLEVBQUUsT0FEYTtBQUVoQkMsTUFBQUEsQ0FBQyxFQUFFLFVBRmE7QUFHaEJDLE1BQUFBLENBQUMsRUFBRSxTQUhhO0FBSWhCQyxNQUFBQSxDQUFDLEVBQUUsVUFKYTtBQUtoQkMsTUFBQUEsQ0FBQyxFQUFFO0FBTGEsS0FBbEI7QUFRQSxVQUFNVixXQUFXLEdBQUcsRUFBcEI7QUFDQSxVQUFNdEYsYUFBYSxHQUFHLEVBQXRCO0FBQ0EsVUFBTUMsa0JBQWtCLEdBQUcsRUFBM0I7QUFFQXNGLElBQUFBLGNBQWMsQ0FBQ1UsT0FBZixDQUF1QkMsS0FBSyxJQUFJO0FBQzlCLFVBQUlBLEtBQUssQ0FBQ0MsWUFBVixFQUF3QjtBQUN0QmIsUUFBQUEsV0FBVyxDQUFDWSxLQUFLLENBQUN4SCxRQUFQLENBQVgsR0FBOEJpSCxTQUFTLENBQUNPLEtBQUssQ0FBQ0MsWUFBUCxDQUF2QztBQUNEOztBQUNELFVBQUlELEtBQUssQ0FBQ0UsY0FBVixFQUEwQjtBQUN4QnBHLFFBQUFBLGFBQWEsQ0FBQ2tHLEtBQUssQ0FBQ3hILFFBQVAsQ0FBYixHQUFnQ2lILFNBQVMsQ0FBQ08sS0FBSyxDQUFDRSxjQUFQLENBQXpDO0FBQ0Q7QUFDRixLQVBEO0FBU0FaLElBQUFBLGdCQUFnQixDQUFDUyxPQUFqQixDQUF5QkMsS0FBSyxJQUFJO0FBQ2hDbEcsTUFBQUEsYUFBYSxDQUFDa0csS0FBSyxDQUFDeEgsUUFBUCxDQUFiLEdBQWdDaUgsU0FBUyxDQUFDQyxDQUExQztBQUNELEtBRkQ7QUFJQUgsSUFBQUEsY0FBYyxDQUFDUSxPQUFmLENBQXVCQyxLQUFLLElBQUk7QUFDOUIsVUFBSUEsS0FBSyxDQUFDQyxZQUFOLEtBQXVCLEdBQTNCLEVBQWdDO0FBQzlCYixRQUFBQSxXQUFXLENBQUNZLEtBQUssQ0FBQ3hILFFBQVAsQ0FBWCxHQUE4QmlILFNBQVMsQ0FBQ0MsQ0FBeEM7QUFDQU4sUUFBQUEsV0FBVyxDQUFDWSxLQUFLLENBQUNHLFlBQVAsQ0FBWCxHQUFrQ1YsU0FBUyxDQUFDRyxDQUE1QztBQUNEOztBQUNELFVBQUlJLEtBQUssQ0FBQ0UsY0FBTixLQUF5QixHQUE3QixFQUFrQztBQUNoQ3BHLFFBQUFBLGFBQWEsQ0FBQ2tHLEtBQUssQ0FBQ3hILFFBQVAsQ0FBYixHQUFnQ2lILFNBQVMsQ0FBQ0MsQ0FBMUM7QUFDQTVGLFFBQUFBLGFBQWEsQ0FBQ2tHLEtBQUssQ0FBQ0csWUFBUCxDQUFiLEdBQW9DVixTQUFTLENBQUNHLENBQTlDO0FBQ0Q7O0FBQ0QsVUFBSUksS0FBSyxDQUFDQyxZQUFOLEtBQXVCLEdBQTNCLEVBQWdDO0FBQzlCYixRQUFBQSxXQUFXLENBQUNZLEtBQUssQ0FBQ3hILFFBQVAsQ0FBWCxHQUE4QmlILFNBQVMsQ0FBQ0MsQ0FBeEM7QUFDRDs7QUFDRCxVQUFJTSxLQUFLLENBQUNFLGNBQU4sS0FBeUIsR0FBN0IsRUFBa0M7QUFDaENwRyxRQUFBQSxhQUFhLENBQUNrRyxLQUFLLENBQUN4SCxRQUFQLENBQWIsR0FBZ0NpSCxTQUFTLENBQUNDLENBQTFDO0FBQ0Q7QUFDRixLQWZEO0FBaUJBLFFBQUlVLFlBQUo7O0FBRUEsU0FBSyxJQUFJN0ssQ0FBQyxHQUFHLENBQWIsRUFBZ0JBLENBQUMsR0FBR2lLLGVBQWUsQ0FBQ2hLLE1BQXBDLEVBQTRDRCxDQUFDLEVBQTdDLEVBQWlEO0FBQy9DLFlBQU07QUFBQzBLLFFBQUFBLFlBQUQ7QUFBZUMsUUFBQUEsY0FBZjtBQUErQjFILFFBQUFBO0FBQS9CLFVBQTJDZ0gsZUFBZSxDQUFDakssQ0FBRCxDQUFoRTs7QUFDQSxVQUFJMEssWUFBWSxLQUFLLEdBQWpCLElBQXdCQyxjQUFjLEtBQUssR0FBM0MsSUFBbURELFlBQVksS0FBSyxHQUFqQixJQUF3QkMsY0FBYyxLQUFLLEdBQWxHLEVBQXdHO0FBQ3RHO0FBQ0E7QUFDQTtBQUNBLFlBQUksQ0FBQ0UsWUFBTCxFQUFtQjtBQUFFQSxVQUFBQSxZQUFZLEdBQUcsTUFBTSxLQUFLOUwsR0FBTCxHQUFXK0wsY0FBWCxDQUEwQjtBQUFDQyxZQUFBQSxNQUFNLEVBQUU7QUFBVCxXQUExQixDQUFyQjtBQUFtRTs7QUFDeEZ2RyxRQUFBQSxrQkFBa0IsQ0FBQ3ZCLFFBQUQsQ0FBbEIsR0FBK0I7QUFDN0IrSCxVQUFBQSxJQUFJLEVBQUVkLFNBQVMsQ0FBQ1EsWUFBRCxDQURjO0FBRTdCTyxVQUFBQSxNQUFNLEVBQUVmLFNBQVMsQ0FBQ1MsY0FBRCxDQUZZO0FBRzdCTyxVQUFBQSxJQUFJLEVBQUVMLFlBQVksQ0FBQzVILFFBQUQsQ0FBWixJQUEwQjtBQUhILFNBQS9CO0FBS0Q7QUFDRjs7QUFFRCxXQUFPO0FBQUM0RyxNQUFBQSxXQUFEO0FBQWN0RixNQUFBQSxhQUFkO0FBQTZCQyxNQUFBQTtBQUE3QixLQUFQO0FBQ0Q7O0FBRUQsUUFBTUMsMEJBQU4sR0FBbUM7QUFDakMsVUFBTTtBQUFDb0YsTUFBQUEsV0FBRDtBQUFjdEYsTUFBQUEsYUFBZDtBQUE2QkMsTUFBQUE7QUFBN0IsUUFBbUQsTUFBTSxLQUFLNEUsZUFBTCxFQUEvRDtBQUNBLFdBQU87QUFBQ1MsTUFBQUEsV0FBRDtBQUFjdEYsTUFBQUEsYUFBZDtBQUE2QkMsTUFBQUE7QUFBN0IsS0FBUDtBQUNEOztBQUVEMkcsRUFBQUEsbUJBQW1CLENBQUNsSSxRQUFELEVBQVdZLE9BQVgsRUFBb0I7QUFDckMsVUFBTUksSUFBSTtBQUNSdkQsTUFBQUEsTUFBTSxFQUFFLEtBREE7QUFFUjBLLE1BQUFBLFdBQVcsRUFBRSxJQUZMO0FBR1JDLE1BQUFBLE9BQU8sRUFBRSxFQUhEO0FBSVJDLE1BQUFBLE1BQU0sRUFBRSxNQUFNLENBQUUsQ0FKUjtBQUtSQyxNQUFBQSxLQUFLLEVBQUUsTUFBTSxDQUFFO0FBTFAsT0FNTDFILE9BTkssQ0FBVjs7QUFTQSxXQUFPLEtBQUt4RyxLQUFMLENBQVdnTSxRQUFYLENBQW9CaEosSUFBSSxDQUFDRyxTQUFMLENBQWVvRixPQUFmLENBQXVCM0MsUUFBdkIsRUFBaUM7QUFBQ3ZDLE1BQUFBLE1BQU0sRUFBRXVELElBQUksQ0FBQ3ZEO0FBQWQsS0FBakMsQ0FBcEIsRUFBNkUsWUFBWTtBQUM5RixZQUFNOEssS0FBSyxHQUFHLE1BQU0sS0FBS3pNLEdBQUwsR0FBVzBNLG1CQUFYLENBQStCeEksUUFBL0IsRUFBeUM7QUFBQ3ZDLFFBQUFBLE1BQU0sRUFBRXVELElBQUksQ0FBQ3ZEO0FBQWQsT0FBekMsQ0FBcEI7QUFDQSxZQUFNZ0wsT0FBTyxHQUFHekgsSUFBSSxDQUFDcUgsTUFBTCxFQUFoQjtBQUNBLFlBQU1LLEtBQUssR0FBRywyQkFBZUgsS0FBZixFQUFzQnZILElBQUksQ0FBQ29ILE9BQTNCLENBQWQ7O0FBQ0EsVUFBSXBILElBQUksQ0FBQ21ILFdBQUwsS0FBcUIsSUFBekIsRUFBK0I7QUFBRU8sUUFBQUEsS0FBSyxDQUFDQyxXQUFOLENBQWtCM0gsSUFBSSxDQUFDbUgsV0FBdkI7QUFBc0M7O0FBQ3ZFbkgsTUFBQUEsSUFBSSxDQUFDc0gsS0FBTCxDQUFXSSxLQUFYLEVBQWtCRCxPQUFsQjtBQUNBLGFBQU9DLEtBQVA7QUFDRCxLQVBNLENBQVA7QUFRRDs7QUFFREYsRUFBQUEsbUJBQW1CLENBQUN4SSxRQUFELEVBQVc0SSxVQUFYLEVBQXVCO0FBQ3hDLFdBQU8sS0FBS3hPLEtBQUwsQ0FBV2dNLFFBQVgsQ0FBb0JoSixJQUFJLENBQUNHLFNBQUwsQ0FBZW9GLE9BQWYsQ0FBdUIzQyxRQUF2QixFQUFpQztBQUFDNEksTUFBQUE7QUFBRCxLQUFqQyxDQUFwQixFQUFvRSxNQUFNO0FBQy9FLGFBQU8sS0FBSzlNLEdBQUwsR0FBVzBNLG1CQUFYLENBQStCeEksUUFBL0IsRUFBeUM7QUFBQzRJLFFBQUFBO0FBQUQsT0FBekMsQ0FBUDtBQUNELEtBRk0sQ0FBUDtBQUdEOztBQUVEQyxFQUFBQSxxQkFBcUIsQ0FBQ2pJLE9BQUQsRUFBVTtBQUM3QixVQUFNSSxJQUFJO0FBQ1JvSCxNQUFBQSxPQUFPLEVBQUUsRUFERDtBQUVSRCxNQUFBQSxXQUFXLEVBQUUsSUFGTDtBQUdSRSxNQUFBQSxNQUFNLEVBQUUsTUFBTSxDQUFFLENBSFI7QUFJUkMsTUFBQUEsS0FBSyxFQUFFLE1BQU0sQ0FBRTtBQUpQLE9BS0wxSCxPQUxLLENBQVY7O0FBUUEsV0FBTyxLQUFLeEcsS0FBTCxDQUFXZ00sUUFBWCxDQUFvQmhKLElBQUksQ0FBQ1MsYUFBekIsRUFBd0MsWUFBWTtBQUN6RCxZQUFNMEssS0FBSyxHQUFHLE1BQU0sS0FBS3pNLEdBQUwsR0FBVytNLHFCQUFYLEVBQXBCO0FBQ0EsWUFBTUosT0FBTyxHQUFHekgsSUFBSSxDQUFDcUgsTUFBTCxFQUFoQjtBQUNBLFlBQU1LLEtBQUssR0FBRyxnQ0FBb0JILEtBQXBCLEVBQTJCdkgsSUFBSSxDQUFDb0gsT0FBaEMsQ0FBZDs7QUFDQSxVQUFJcEgsSUFBSSxDQUFDbUgsV0FBTCxLQUFxQixJQUF6QixFQUErQjtBQUFFTyxRQUFBQSxLQUFLLENBQUNDLFdBQU4sQ0FBa0IzSCxJQUFJLENBQUNtSCxXQUF2QjtBQUFzQzs7QUFDdkVuSCxNQUFBQSxJQUFJLENBQUNzSCxLQUFMLENBQVdJLEtBQVgsRUFBa0JELE9BQWxCO0FBQ0EsYUFBT0MsS0FBUDtBQUNELEtBUE0sQ0FBUDtBQVFEOztBQUVESSxFQUFBQSxpQkFBaUIsQ0FBQzlJLFFBQUQsRUFBVztBQUMxQixXQUFPLEtBQUs1RixLQUFMLENBQVdnTSxRQUFYLENBQW9CaEosSUFBSSxDQUFDVyxLQUFMLENBQVc0RSxPQUFYLENBQW1CM0MsUUFBbkIsQ0FBcEIsRUFBa0QsTUFBTTtBQUM3RCxhQUFPLEtBQUtsRSxHQUFMLEdBQVdnTixpQkFBWCxDQUE2QjlJLFFBQTdCLENBQVA7QUFDRCxLQUZNLENBQVA7QUFHRCxHQWh0QndDLENBa3RCekM7OztBQUVBK0ksRUFBQUEsYUFBYSxHQUFHO0FBQ2QsV0FBTyxLQUFLM08sS0FBTCxDQUFXZ00sUUFBWCxDQUFvQmhKLElBQUksQ0FBQ2EsVUFBekIsRUFBcUMsWUFBWTtBQUN0RCxZQUFNK0ssVUFBVSxHQUFHLE1BQU0sS0FBS2xOLEdBQUwsR0FBV21OLGFBQVgsRUFBekI7QUFDQSxhQUFPRCxVQUFVLENBQUNFLFNBQVgsR0FBdUJDLG1CQUFPQyxZQUFQLEVBQXZCLEdBQStDLElBQUlELGtCQUFKLENBQVdILFVBQVgsQ0FBdEQ7QUFDRCxLQUhNLENBQVA7QUFJRDs7QUFFREssRUFBQUEsU0FBUyxDQUFDOUUsR0FBRCxFQUFNO0FBQ2IsV0FBTyxLQUFLbkssS0FBTCxDQUFXZ00sUUFBWCxDQUFvQmhKLElBQUksQ0FBQ2tNLElBQUwsQ0FBVTNHLE9BQVYsQ0FBa0I0QixHQUFsQixDQUFwQixFQUE0QyxZQUFZO0FBQzdELFlBQU0sQ0FBQ2dGLFNBQUQsSUFBYyxNQUFNLEtBQUt6TixHQUFMLEdBQVcwTixVQUFYLENBQXNCO0FBQUNDLFFBQUFBLEdBQUcsRUFBRSxDQUFOO0FBQVNDLFFBQUFBLEdBQUcsRUFBRW5GLEdBQWQ7QUFBbUJvRixRQUFBQSxZQUFZLEVBQUU7QUFBakMsT0FBdEIsQ0FBMUI7QUFDQSxZQUFNaEosTUFBTSxHQUFHLElBQUl3SSxrQkFBSixDQUFXSSxTQUFYLENBQWY7QUFDQSxhQUFPNUksTUFBUDtBQUNELEtBSk0sQ0FBUDtBQUtEOztBQUVEaUosRUFBQUEsZ0JBQWdCLENBQUNoSixPQUFELEVBQVU7QUFDeEIsV0FBTyxLQUFLeEcsS0FBTCxDQUFXZ00sUUFBWCxDQUFvQmhKLElBQUksQ0FBQ2MsYUFBekIsRUFBd0MsWUFBWTtBQUN6RCxZQUFNMkwsT0FBTyxHQUFHLE1BQU0sS0FBSy9OLEdBQUwsR0FBVzBOLFVBQVg7QUFBdUJFLFFBQUFBLEdBQUcsRUFBRTtBQUE1QixTQUF1QzlJLE9BQXZDLEVBQXRCO0FBQ0EsYUFBT2lKLE9BQU8sQ0FBQ3BOLEdBQVIsQ0FBWWtFLE1BQU0sSUFBSSxJQUFJd0ksa0JBQUosQ0FBV3hJLE1BQVgsQ0FBdEIsQ0FBUDtBQUNELEtBSE0sQ0FBUDtBQUlEOztBQUVELFFBQU1tSixjQUFOLENBQXFCdkYsR0FBckIsRUFBMEI7QUFDeEIsVUFBTXdGLHdCQUF3QixHQUFHLE1BQU0sS0FBS2pPLEdBQUwsR0FBV2tPLHFCQUFYLENBQWlDekYsR0FBakMsRUFBc0M7QUFBQzBGLE1BQUFBLFNBQVMsRUFBRSxLQUFaO0FBQW1CQyxNQUFBQSxVQUFVLEVBQUU7QUFBL0IsS0FBdEMsQ0FBdkM7QUFDQSxVQUFNQyxhQUFhLEdBQUcsQ0FBQyxNQUFNLEtBQUtqUSxVQUFMLENBQWdCa1EsZ0JBQWhCLEVBQVAsRUFBMkNDLFdBQTNDLEVBQXRCO0FBQ0EsV0FBT04sd0JBQXdCLENBQUNyTSxRQUF6QixDQUFrQ3lNLGFBQWEsQ0FBQ0csVUFBZCxFQUFsQyxDQUFQO0FBQ0QsR0E5dUJ3QyxDQWd2QnpDOzs7QUFFQUMsRUFBQUEsVUFBVSxDQUFDM0osT0FBRCxFQUFVO0FBQ2xCO0FBQ0E7QUFDQTtBQUNBLFdBQU8sS0FBS3hHLEtBQUwsQ0FBV2dNLFFBQVgsQ0FBb0JoSixJQUFJLENBQUNnQixPQUF6QixFQUFrQyxZQUFZO0FBQ25ELFlBQU1vTSxTQUFTLEdBQUcsTUFBTSxLQUFLMU8sR0FBTCxHQUFXeU8sVUFBWCxDQUFzQjNKLE9BQXRCLENBQXhCO0FBQ0EsYUFBT2MsTUFBTSxDQUFDN0UsSUFBUCxDQUFZMk4sU0FBWixFQUF1Qi9OLEdBQXZCLENBQTJCeUUsS0FBSyxJQUFJLElBQUl1SixrQkFBSixDQUFXdkosS0FBWCxFQUFrQnNKLFNBQVMsQ0FBQ3RKLEtBQUQsQ0FBM0IsQ0FBcEMsQ0FBUDtBQUNELEtBSE0sQ0FBUDtBQUlELEdBMXZCd0MsQ0E0dkJ6Qzs7O0FBRUF3SixFQUFBQSxXQUFXLEdBQUc7QUFDWixXQUFPLEtBQUt0USxLQUFMLENBQVdnTSxRQUFYLENBQW9CaEosSUFBSSxDQUFDWSxRQUF6QixFQUFtQyxZQUFZO0FBQ3BELFlBQU0yTSxRQUFRLEdBQUcsTUFBTSxLQUFLN08sR0FBTCxHQUFXNE8sV0FBWCxFQUF2QjtBQUNBLFlBQU0xTSxRQUFRLEdBQUcsSUFBSTRNLHFCQUFKLEVBQWpCOztBQUNBLFdBQUssTUFBTW5DLE9BQVgsSUFBc0JrQyxRQUF0QixFQUFnQztBQUM5QixZQUFJRSxRQUFRLEdBQUdDLGtCQUFmOztBQUNBLFlBQUlyQyxPQUFPLENBQUNvQyxRQUFaLEVBQXNCO0FBQ3BCQSxVQUFBQSxRQUFRLEdBQUdwQyxPQUFPLENBQUNvQyxRQUFSLENBQWlCcEgsVUFBakIsR0FDUHNILG1CQUFPQyxvQkFBUCxDQUNBdkMsT0FBTyxDQUFDb0MsUUFBUixDQUFpQkksV0FEakIsRUFFQXhDLE9BQU8sQ0FBQ29DLFFBQVIsQ0FBaUJwSCxVQUZqQixFQUdBZ0YsT0FBTyxDQUFDb0MsUUFBUixDQUFpQkssU0FIakIsQ0FETyxHQU1QLElBQUlILGtCQUFKLENBQVd0QyxPQUFPLENBQUNvQyxRQUFSLENBQWlCSSxXQUE1QixDQU5KO0FBT0Q7O0FBRUQsWUFBSW5ILElBQUksR0FBRytHLFFBQVg7O0FBQ0EsWUFBSXBDLE9BQU8sQ0FBQzNFLElBQVosRUFBa0I7QUFDaEJBLFVBQUFBLElBQUksR0FBRzJFLE9BQU8sQ0FBQzNFLElBQVIsQ0FBYUwsVUFBYixHQUNIc0gsbUJBQU9DLG9CQUFQLENBQ0F2QyxPQUFPLENBQUMzRSxJQUFSLENBQWFtSCxXQURiLEVBRUF4QyxPQUFPLENBQUMzRSxJQUFSLENBQWFMLFVBRmIsRUFHQWdGLE9BQU8sQ0FBQzNFLElBQVIsQ0FBYW9ILFNBSGIsQ0FERyxHQU1ILElBQUlILGtCQUFKLENBQVd0QyxPQUFPLENBQUMzRSxJQUFSLENBQWFtSCxXQUF4QixDQU5KO0FBT0Q7O0FBRURqTixRQUFBQSxRQUFRLENBQUNiLEdBQVQsQ0FBYSxJQUFJNE4sa0JBQUosQ0FBV3RDLE9BQU8sQ0FBQ3JILElBQW5CLEVBQXlCeUosUUFBekIsRUFBbUMvRyxJQUFuQyxFQUF5QzJFLE9BQU8sQ0FBQzBDLElBQWpELEVBQXVEO0FBQUM1RyxVQUFBQSxHQUFHLEVBQUVrRSxPQUFPLENBQUNsRTtBQUFkLFNBQXZELENBQWI7QUFDRDs7QUFDRCxhQUFPdkcsUUFBUDtBQUNELEtBN0JNLENBQVA7QUE4QkQ7O0FBRURvTixFQUFBQSxrQkFBa0IsR0FBRztBQUNuQixXQUFPLEtBQUtoUixLQUFMLENBQVdnTSxRQUFYLENBQW9CaEosSUFBSSxDQUFDZSxlQUF6QixFQUEwQyxNQUFNO0FBQ3JELGFBQU8sS0FBS3JDLEdBQUwsR0FBV3VQLFlBQVgsRUFBUDtBQUNELEtBRk0sQ0FBUDtBQUdELEdBbnlCd0MsQ0FxeUJ6Qzs7O0FBRUFDLEVBQUFBLFNBQVMsR0FBRztBQUNWLFdBQU8sS0FBS3hQLEdBQUwsR0FBV3dQLFNBQVgsQ0FBcUIsS0FBS3BSLFVBQUwsQ0FBZ0JxUixtQkFBaEIsRUFBckIsQ0FBUDtBQUNEOztBQUVEQyxFQUFBQSxVQUFVLEdBQUc7QUFDWCxXQUFPLEtBQUsxUCxHQUFMLEdBQVcwUCxVQUFYLENBQXNCLEtBQUt0UixVQUFMLENBQWdCcVIsbUJBQWhCLEVBQXRCLENBQVA7QUFDRCxHQTd5QndDLENBK3lCekM7OztBQUVBRSxFQUFBQSxVQUFVLEdBQUc7QUFDWCxXQUFPLEtBQUtyUixLQUFMLENBQVdnTSxRQUFYLENBQW9CaEosSUFBSSxDQUFDaUIsT0FBekIsRUFBa0MsWUFBWTtBQUNuRCxZQUFNcU4sV0FBVyxHQUFHLE1BQU0sS0FBSzVQLEdBQUwsR0FBVzJQLFVBQVgsRUFBMUI7QUFDQSxhQUFPLElBQUlFLHFCQUFKLENBQ0xELFdBQVcsQ0FBQ2pQLEdBQVosQ0FBZ0IsQ0FBQztBQUFDMkUsUUFBQUEsSUFBRDtBQUFPd0ssUUFBQUE7QUFBUCxPQUFELEtBQWlCLElBQUlDLGtCQUFKLENBQVd6SyxJQUFYLEVBQWlCd0ssR0FBakIsQ0FBakMsQ0FESyxDQUFQO0FBR0QsS0FMTSxDQUFQO0FBTUQ7O0FBRURFLEVBQUFBLFNBQVMsQ0FBQzFLLElBQUQsRUFBT3dLLEdBQVAsRUFBWTtBQUNuQixXQUFPLEtBQUt2UCxVQUFMLENBQ0wsTUFBTSxDQUNKLEdBQUdlLElBQUksQ0FBQ2tCLE1BQUwsQ0FBWTBGLGVBQVosQ0FBNkIsVUFBUzVDLElBQUssTUFBM0MsQ0FEQyxFQUVKLEdBQUdoRSxJQUFJLENBQUNrQixNQUFMLENBQVkwRixlQUFaLENBQTZCLFVBQVM1QyxJQUFLLFFBQTNDLENBRkMsRUFHSmhFLElBQUksQ0FBQ2lCLE9BSEQsQ0FERCxFQU1MO0FBQ0EsVUFBTSxLQUFLeUMscUJBQUwsQ0FBMkIsV0FBM0IsRUFBd0MsT0FBT00sSUFBUCxFQUFhd0ssR0FBYixLQUFxQjtBQUNqRSxZQUFNLEtBQUs5UCxHQUFMLEdBQVdnUSxTQUFYLENBQXFCMUssSUFBckIsRUFBMkJ3SyxHQUEzQixDQUFOO0FBQ0EsYUFBTyxJQUFJQyxrQkFBSixDQUFXekssSUFBWCxFQUFpQndLLEdBQWpCLENBQVA7QUFDRCxLQUhLLEVBR0h4SyxJQUhHLEVBR0d3SyxHQUhILENBUEQsQ0FBUDtBQVlEOztBQUVELFFBQU1HLGFBQU4sQ0FBb0JoSyxVQUFwQixFQUFnQztBQUM5QixVQUFNc0UsTUFBTSxHQUFHLE1BQU0sS0FBS0YsZUFBTCxFQUFyQjtBQUNBLFdBQU9FLE1BQU0sQ0FBQ0csTUFBUCxDQUFjd0YsV0FBZCxDQUEwQkMsS0FBakM7QUFDRDs7QUFFRCxRQUFNQyxjQUFOLENBQXFCbkssVUFBckIsRUFBaUM7QUFDL0IsVUFBTXNFLE1BQU0sR0FBRyxNQUFNLEtBQUtGLGVBQUwsRUFBckI7QUFDQSxXQUFPRSxNQUFNLENBQUNHLE1BQVAsQ0FBY3dGLFdBQWQsQ0FBMEJHLE1BQWpDO0FBQ0Q7O0FBRURDLEVBQUFBLFNBQVMsQ0FBQ0MsTUFBRCxFQUFTO0FBQUNDLElBQUFBO0FBQUQsTUFBVTtBQUFDQSxJQUFBQSxLQUFLLEVBQUU7QUFBUixHQUFuQixFQUFtQztBQUMxQyxXQUFPLEtBQUtsUyxLQUFMLENBQVdnTSxRQUFYLENBQW9CaEosSUFBSSxDQUFDa0IsTUFBTCxDQUFZcUUsT0FBWixDQUFvQjBKLE1BQXBCLEVBQTRCO0FBQUNDLE1BQUFBO0FBQUQsS0FBNUIsQ0FBcEIsRUFBMEQsTUFBTTtBQUNyRSxhQUFPLEtBQUt4USxHQUFMLEdBQVdzUSxTQUFYLENBQXFCQyxNQUFyQixFQUE2QjtBQUFDQyxRQUFBQTtBQUFELE9BQTdCLENBQVA7QUFDRCxLQUZNLENBQVA7QUFHRDs7QUFFREMsRUFBQUEsZUFBZSxDQUFDOU4sR0FBRCxFQUFNbUMsT0FBTixFQUFlO0FBQzVCLFdBQU8sS0FBS3dMLFNBQUwsQ0FBZTNOLEdBQWYsRUFBb0JtQyxPQUFwQixDQUFQO0FBQ0QsR0EzMUJ3QyxDQTYxQnpDOzs7QUFFQTRMLEVBQUFBLGVBQWUsQ0FBQ2pJLEdBQUQsRUFBTTtBQUNuQixXQUFPLEtBQUtuSyxLQUFMLENBQVdnTSxRQUFYLENBQW9CaEosSUFBSSxDQUFDa00sSUFBTCxDQUFVM0csT0FBVixDQUFrQjRCLEdBQWxCLENBQXBCLEVBQTRDLE1BQU07QUFDdkQsYUFBTyxLQUFLekksR0FBTCxHQUFXMFEsZUFBWCxDQUEyQmpJLEdBQTNCLENBQVA7QUFDRCxLQUZNLENBQVA7QUFHRDs7QUFFRGtJLEVBQUFBLHFCQUFxQixDQUFDbEksR0FBRCxFQUFNO0FBQ3pCLFdBQU8sS0FBS2lJLGVBQUwsQ0FBcUJqSSxHQUFyQixDQUFQO0FBQ0QsR0F2MkJ3QyxDQXkyQnpDOzs7QUFFQW1JLEVBQUFBLGlCQUFpQixDQUFDMUgsc0JBQXNCLEdBQUcsSUFBMUIsRUFBZ0M7QUFDL0MsV0FBTyxLQUFLMUssY0FBTCxDQUFvQnFTLFVBQXBCLENBQStCM0gsc0JBQS9CLENBQVA7QUFDRDs7QUFFRDRILEVBQUFBLGlCQUFpQixDQUFDNUgsc0JBQXNCLEdBQUcsSUFBMUIsRUFBZ0M7QUFDL0MsV0FBTyxLQUFLMUssY0FBTCxDQUFvQnVTLFVBQXBCLENBQStCN0gsc0JBQS9CLENBQVA7QUFDRDs7QUFFRDhILEVBQUFBLHVCQUF1QixDQUFDOUgsc0JBQXNCLEdBQUcsSUFBMUIsRUFBZ0M7QUFDckQsV0FBTyxLQUFLMUssY0FBTCxDQUFvQnlTLGdCQUFwQixDQUFxQy9ILHNCQUFyQyxDQUFQO0FBQ0QsR0FyM0J3QyxDQXUzQnpDOztBQUVBOzs7QUFDQWdJLEVBQUFBLFFBQVEsR0FBRztBQUNULFdBQU8sS0FBSzVTLEtBQVo7QUFDRDs7QUFFRGlDLEVBQUFBLFVBQVUsQ0FBQ0QsSUFBRCxFQUFPNlEsSUFBUCxFQUFhO0FBQ3JCLFdBQU9BLElBQUksR0FBR0MsSUFBUCxDQUNMQyxNQUFNLElBQUk7QUFDUixXQUFLaFIsa0JBQUwsQ0FBd0JDLElBQXhCO0FBQ0EsYUFBTytRLE1BQVA7QUFDRCxLQUpJLEVBS0wxRyxHQUFHLElBQUk7QUFDTCxXQUFLdEssa0JBQUwsQ0FBd0JDLElBQXhCO0FBQ0EsYUFBT29ELE9BQU8sQ0FBQ0MsTUFBUixDQUFlZ0gsR0FBZixDQUFQO0FBQ0QsS0FSSSxDQUFQO0FBVUQ7O0FBejRCd0M7Ozs7QUE0NEIzQ3pNLGtCQUFNb1QsUUFBTixDQUFlclQsT0FBZjs7QUFFQSxTQUFTK0wsU0FBVCxDQUFtQnVILEtBQW5CLEVBQTBCQyxTQUExQixFQUFxQztBQUNuQyxRQUFNQyxPQUFPLEdBQUcsRUFBaEI7QUFDQSxRQUFNQyxVQUFVLEdBQUcsRUFBbkI7QUFDQUgsRUFBQUEsS0FBSyxDQUFDOUYsT0FBTixDQUFja0csSUFBSSxJQUFJO0FBQ3BCLFFBQUlILFNBQVMsQ0FBQ0csSUFBRCxDQUFiLEVBQXFCO0FBQ25CRixNQUFBQSxPQUFPLENBQUN6SixJQUFSLENBQWEySixJQUFiO0FBQ0QsS0FGRCxNQUVPO0FBQ0xELE1BQUFBLFVBQVUsQ0FBQzFKLElBQVgsQ0FBZ0IySixJQUFoQjtBQUNEO0FBQ0YsR0FORDtBQU9BLFNBQU8sQ0FBQ0YsT0FBRCxFQUFVQyxVQUFWLENBQVA7QUFDRDs7QUFFRCxNQUFNblQsS0FBTixDQUFZO0FBQ1ZKLEVBQUFBLFdBQVcsR0FBRztBQUNaLFNBQUt5VCxPQUFMLEdBQWUsSUFBSUMsR0FBSixFQUFmO0FBQ0EsU0FBS0MsT0FBTCxHQUFlLElBQUlELEdBQUosRUFBZjtBQUVBLFNBQUtFLE9BQUwsR0FBZSxJQUFJQyxpQkFBSixFQUFmO0FBQ0Q7O0FBRUQxSCxFQUFBQSxRQUFRLENBQUMzSCxHQUFELEVBQU1zUCxTQUFOLEVBQWlCO0FBQ3ZCLFVBQU1DLE9BQU8sR0FBR3ZQLEdBQUcsQ0FBQ3dQLFVBQUosRUFBaEI7QUFDQSxVQUFNQyxRQUFRLEdBQUcsS0FBS1IsT0FBTCxDQUFhUyxHQUFiLENBQWlCSCxPQUFqQixDQUFqQjs7QUFDQSxRQUFJRSxRQUFRLEtBQUtFLFNBQWpCLEVBQTRCO0FBQzFCRixNQUFBQSxRQUFRLENBQUNHLElBQVQ7QUFDQSxhQUFPSCxRQUFRLENBQUNJLE9BQWhCO0FBQ0Q7O0FBRUQsVUFBTUMsT0FBTyxHQUFHUixTQUFTLEVBQXpCO0FBRUEsU0FBS0wsT0FBTCxDQUFhYyxHQUFiLENBQWlCUixPQUFqQixFQUEwQjtBQUN4QlMsTUFBQUEsU0FBUyxFQUFFQyxXQUFXLENBQUNDLEdBQVosRUFEYTtBQUV4Qk4sTUFBQUEsSUFBSSxFQUFFLENBRmtCO0FBR3hCQyxNQUFBQSxPQUFPLEVBQUVDO0FBSGUsS0FBMUI7QUFNQSxVQUFNSyxNQUFNLEdBQUduUSxHQUFHLENBQUNvUSxTQUFKLEVBQWY7O0FBQ0EsU0FBSyxJQUFJOVIsQ0FBQyxHQUFHLENBQWIsRUFBZ0JBLENBQUMsR0FBRzZSLE1BQU0sQ0FBQzVSLE1BQTNCLEVBQW1DRCxDQUFDLEVBQXBDLEVBQXdDO0FBQ3RDLFlBQU0rUixLQUFLLEdBQUdGLE1BQU0sQ0FBQzdSLENBQUQsQ0FBcEI7QUFDQSxVQUFJZ1MsUUFBUSxHQUFHLEtBQUtuQixPQUFMLENBQWFPLEdBQWIsQ0FBaUJXLEtBQWpCLENBQWY7O0FBQ0EsVUFBSUMsUUFBUSxLQUFLWCxTQUFqQixFQUE0QjtBQUMxQlcsUUFBQUEsUUFBUSxHQUFHLElBQUlqUyxHQUFKLEVBQVg7QUFDQSxhQUFLOFEsT0FBTCxDQUFhWSxHQUFiLENBQWlCTSxLQUFqQixFQUF3QkMsUUFBeEI7QUFDRDs7QUFDREEsTUFBQUEsUUFBUSxDQUFDNVIsR0FBVCxDQUFhc0IsR0FBYjtBQUNEOztBQUVELFNBQUt6RCxTQUFMO0FBRUEsV0FBT3VULE9BQVA7QUFDRDs7QUFFRGxTLEVBQUFBLFVBQVUsQ0FBQ1EsSUFBRCxFQUFPO0FBQ2YsU0FBSyxJQUFJRSxDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHRixJQUFJLENBQUNHLE1BQXpCLEVBQWlDRCxDQUFDLEVBQWxDLEVBQXNDO0FBQ3BDRixNQUFBQSxJQUFJLENBQUNFLENBQUQsQ0FBSixDQUFRaVMsZUFBUixDQUF3QixJQUF4QjtBQUNEOztBQUVELFFBQUluUyxJQUFJLENBQUNHLE1BQUwsR0FBYyxDQUFsQixFQUFxQjtBQUNuQixXQUFLaEMsU0FBTDtBQUNEO0FBQ0Y7O0FBRURpVSxFQUFBQSxXQUFXLENBQUNILEtBQUQsRUFBUTtBQUNqQixXQUFPLEtBQUtsQixPQUFMLENBQWFPLEdBQWIsQ0FBaUJXLEtBQWpCLEtBQTJCLEVBQWxDO0FBQ0Q7O0FBRURJLEVBQUFBLGFBQWEsQ0FBQ2xCLE9BQUQsRUFBVTtBQUNyQixTQUFLTixPQUFMLFdBQW9CTSxPQUFwQjtBQUNBLFNBQUtoVCxTQUFMO0FBQ0Q7O0FBRURtVSxFQUFBQSxlQUFlLENBQUNMLEtBQUQsRUFBUXJRLEdBQVIsRUFBYTtBQUMxQixVQUFNc1EsUUFBUSxHQUFHLEtBQUtuQixPQUFMLENBQWFPLEdBQWIsQ0FBaUJXLEtBQWpCLENBQWpCO0FBQ0FDLElBQUFBLFFBQVEsSUFBSUEsUUFBUSxVQUFSLENBQWdCdFEsR0FBaEIsQ0FBWjtBQUNBLFNBQUt6RCxTQUFMO0FBQ0Q7QUFFRDs7O0FBQ0EsR0FBQ29VLE1BQU0sQ0FBQ0MsUUFBUixJQUFvQjtBQUNsQixXQUFPLEtBQUszQixPQUFMLENBQWEwQixNQUFNLENBQUNDLFFBQXBCLEdBQVA7QUFDRDs7QUFFRGhRLEVBQUFBLEtBQUssR0FBRztBQUNOLFNBQUtxTyxPQUFMLENBQWFyTyxLQUFiO0FBQ0EsU0FBS3VPLE9BQUwsQ0FBYXZPLEtBQWI7QUFDQSxTQUFLckUsU0FBTDtBQUNEOztBQUVEQSxFQUFBQSxTQUFTLEdBQUc7QUFDVixTQUFLNlMsT0FBTCxDQUFheUIsSUFBYixDQUFrQixZQUFsQjtBQUNEO0FBRUQ7OztBQUNBQyxFQUFBQSxXQUFXLENBQUNDLFFBQUQsRUFBVztBQUNwQixXQUFPLEtBQUszQixPQUFMLENBQWE0QixFQUFiLENBQWdCLFlBQWhCLEVBQThCRCxRQUE5QixDQUFQO0FBQ0Q7O0FBRUR2VCxFQUFBQSxPQUFPLEdBQUc7QUFDUixTQUFLNFIsT0FBTCxDQUFhNkIsT0FBYjtBQUNEOztBQXZGUzs7QUEwRlosTUFBTUMsUUFBTixDQUFlO0FBQ2IxVixFQUFBQSxXQUFXLENBQUMrVCxPQUFELEVBQVVZLE1BQU0sR0FBRyxFQUFuQixFQUF1QjtBQUNoQyxTQUFLWixPQUFMLEdBQWVBLE9BQWY7QUFDQSxTQUFLWSxNQUFMLEdBQWNBLE1BQWQ7QUFDRDs7QUFFRFgsRUFBQUEsVUFBVSxHQUFHO0FBQ1gsV0FBTyxLQUFLRCxPQUFaO0FBQ0Q7O0FBRURhLEVBQUFBLFNBQVMsR0FBRztBQUNWLFdBQU8sS0FBS0QsTUFBWjtBQUNEOztBQUVESSxFQUFBQSxlQUFlLENBQUM1VSxLQUFELEVBQVF3VixZQUFZLEdBQUcsSUFBdkIsRUFBNkI7QUFDMUN4VixJQUFBQSxLQUFLLENBQUM4VSxhQUFOLENBQW9CLEtBQUtqQixVQUFMLEVBQXBCO0FBRUEsVUFBTVcsTUFBTSxHQUFHLEtBQUtDLFNBQUwsRUFBZjs7QUFDQSxTQUFLLElBQUk5UixDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHNlIsTUFBTSxDQUFDNVIsTUFBM0IsRUFBbUNELENBQUMsRUFBcEMsRUFBd0M7QUFDdEMsWUFBTStSLEtBQUssR0FBR0YsTUFBTSxDQUFDN1IsQ0FBRCxDQUFwQjs7QUFDQSxVQUFJK1IsS0FBSyxLQUFLYyxZQUFkLEVBQTRCO0FBQzFCO0FBQ0Q7O0FBRUR4VixNQUFBQSxLQUFLLENBQUMrVSxlQUFOLENBQXNCTCxLQUF0QixFQUE2QixJQUE3QjtBQUNEO0FBQ0Y7QUFFRDs7O0FBQ0F2TyxFQUFBQSxRQUFRLEdBQUc7QUFDVCxXQUFRLFlBQVcsS0FBS3lOLE9BQVEsR0FBaEM7QUFDRDs7QUEvQlk7O0FBa0NmLE1BQU02QixRQUFOLENBQWU7QUFDYjVWLEVBQUFBLFdBQVcsQ0FBQzZVLEtBQUQsRUFBUTtBQUNqQixTQUFLQSxLQUFMLEdBQWFBLEtBQWI7QUFDRDs7QUFFREUsRUFBQUEsZUFBZSxDQUFDNVUsS0FBRCxFQUFRO0FBQ3JCLFNBQUssTUFBTTBWLFdBQVgsSUFBMEIxVixLQUFLLENBQUM2VSxXQUFOLENBQWtCLEtBQUtILEtBQXZCLENBQTFCLEVBQXlEO0FBQ3ZEZ0IsTUFBQUEsV0FBVyxDQUFDZCxlQUFaLENBQTRCNVUsS0FBNUIsRUFBbUMsS0FBSzBVLEtBQXhDO0FBQ0Q7QUFDRjtBQUVEOzs7QUFDQXZPLEVBQUFBLFFBQVEsR0FBRztBQUNULFdBQVEsWUFBVyxLQUFLdU8sS0FBTSxHQUE5QjtBQUNEOztBQWRZOztBQWlCZixNQUFNMVIsSUFBSSxHQUFHO0FBQ1hDLEVBQUFBLFlBQVksRUFBRSxJQUFJc1MsUUFBSixDQUFhLGVBQWIsQ0FESDtBQUdYOVIsRUFBQUEsYUFBYSxFQUFFLElBQUk4UixRQUFKLENBQWEsZ0JBQWIsQ0FISjtBQUtYcFMsRUFBQUEsU0FBUyxFQUFFO0FBQ1R3UyxJQUFBQSxPQUFPLEVBQUUsQ0FBQztBQUFDdFMsTUFBQUE7QUFBRCxLQUFELEtBQWVBLE1BQU0sR0FBRyxHQUFILEdBQVMsR0FEOUI7QUFHVGtGLElBQUFBLE9BQU8sRUFBRSxDQUFDSyxRQUFELEVBQVdwQyxPQUFYLEtBQXVCO0FBQUU7QUFDaEMsWUFBTW9QLE1BQU0sR0FBRzVTLElBQUksQ0FBQ0csU0FBTCxDQUFld1MsT0FBZixDQUF1Qm5QLE9BQXZCLENBQWY7O0FBQ0EsWUFBTWdJLFVBQVUsR0FBR2hJLE9BQU8sQ0FBQ2dJLFVBQVIsSUFBc0IsTUFBekM7QUFFQSxZQUFNcUgsV0FBVyxHQUFHLEVBQXBCOztBQUNBLFVBQUlyUCxPQUFPLENBQUNnSSxVQUFaLEVBQXdCO0FBQ3RCcUgsUUFBQUEsV0FBVyxDQUFDbk0sSUFBWixDQUFrQixnQ0FBK0JkLFFBQVMsRUFBMUQ7QUFDQWlOLFFBQUFBLFdBQVcsQ0FBQ25NLElBQVosQ0FBaUIseUJBQWpCO0FBQ0QsT0FIRCxNQUdPO0FBQ0xtTSxRQUFBQSxXQUFXLENBQUNuTSxJQUFaLENBQWlCLHNCQUFqQjtBQUNEOztBQUVELGFBQU8sSUFBSTZMLFFBQUosQ0FBYyxjQUFhSyxNQUFPLElBQUdwSCxVQUFXLElBQUc1RixRQUFTLEVBQTVELEVBQStELENBQ3BFLFlBRG9FLEVBRW5FLGtCQUFpQmdOLE1BQU8sRUFGMkMsRUFHbkUsa0JBQWlCQSxNQUFPLFNBQVFoTixRQUFTLEVBSDBCLEVBSXBFLEdBQUdpTixXQUppRSxDQUEvRCxDQUFQO0FBTUQsS0FyQlE7QUF1QlR2UixJQUFBQSxnQkFBZ0IsRUFBRSxDQUFDd1IsU0FBRCxFQUFZbFAsSUFBWixLQUFxQjtBQUNyQyxZQUFNbkUsSUFBSSxHQUFHLEVBQWI7O0FBQ0EsV0FBSyxJQUFJRSxDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHbVQsU0FBUyxDQUFDbFQsTUFBOUIsRUFBc0NELENBQUMsRUFBdkMsRUFBMkM7QUFDekMsYUFBSyxJQUFJb1QsQ0FBQyxHQUFHLENBQWIsRUFBZ0JBLENBQUMsR0FBR25QLElBQUksQ0FBQ2hFLE1BQXpCLEVBQWlDbVQsQ0FBQyxFQUFsQyxFQUFzQztBQUNwQ3RULFVBQUFBLElBQUksQ0FBQ2lILElBQUwsQ0FBVSxJQUFJK0wsUUFBSixDQUFjLGtCQUFpQnpTLElBQUksQ0FBQ0csU0FBTCxDQUFld1MsT0FBZixDQUF1Qi9PLElBQUksQ0FBQ21QLENBQUQsQ0FBM0IsQ0FBZ0MsU0FBUUQsU0FBUyxDQUFDblQsQ0FBRCxDQUFJLEVBQXBGLENBQVY7QUFDRDtBQUNGOztBQUNELGFBQU9GLElBQVA7QUFDRCxLQS9CUTtBQWlDVG9HLElBQUFBLG9CQUFvQixFQUFFaU4sU0FBUyxJQUFJO0FBQ2pDLGFBQU9BLFNBQVMsQ0FBQ3pULEdBQVYsQ0FBY3VHLFFBQVEsSUFBSSxJQUFJNk0sUUFBSixDQUFjLGdDQUErQjdNLFFBQVMsRUFBdEQsQ0FBMUIsQ0FBUDtBQUNELEtBbkNRO0FBcUNURixJQUFBQSxpQkFBaUIsRUFBRSxJQUFJK00sUUFBSixDQUFhLHlCQUFiLENBckNWO0FBdUNUclMsSUFBQUEsWUFBWSxFQUFFLENBQUMsR0FBR3dELElBQUosS0FBYUEsSUFBSSxDQUFDdkUsR0FBTCxDQUFTMlQsR0FBRyxJQUFJLElBQUlQLFFBQUosQ0FBYyxrQkFBaUJ6UyxJQUFJLENBQUNHLFNBQUwsQ0FBZXdTLE9BQWYsQ0FBdUJLLEdBQXZCLENBQTRCLEVBQTNELENBQWhCLENBdkNsQjtBQXlDVHRTLElBQUFBLEdBQUcsRUFBRSxJQUFJK1IsUUFBSixDQUFhLFlBQWI7QUF6Q0ksR0FMQTtBQWlEWDlSLEVBQUFBLEtBQUssRUFBRTtBQUNMNEUsSUFBQUEsT0FBTyxFQUFFSyxRQUFRLElBQUksSUFBSTJNLFFBQUosQ0FBYyxTQUFRM00sUUFBUyxFQUEvQixFQUFrQyxDQUFDLE9BQUQsQ0FBbEMsQ0FEaEI7QUFHTGxGLElBQUFBLEdBQUcsRUFBRSxJQUFJK1IsUUFBSixDQUFhLE9BQWI7QUFIQSxHQWpESTtBQXVEWDVSLEVBQUFBLFVBQVUsRUFBRSxJQUFJMFIsUUFBSixDQUFhLGFBQWIsQ0F2REQ7QUF5RFh6UixFQUFBQSxhQUFhLEVBQUUsSUFBSXlSLFFBQUosQ0FBYSxnQkFBYixDQXpESjtBQTJEWHZSLEVBQUFBLE9BQU8sRUFBRSxJQUFJdVIsUUFBSixDQUFhLFNBQWIsQ0EzREU7QUE2RFgzUixFQUFBQSxRQUFRLEVBQUUsSUFBSTJSLFFBQUosQ0FBYSxVQUFiLENBN0RDO0FBK0RYeFIsRUFBQUEsZUFBZSxFQUFFLElBQUl3UixRQUFKLENBQWEsa0JBQWIsQ0EvRE47QUFpRVh0UixFQUFBQSxPQUFPLEVBQUUsSUFBSXNSLFFBQUosQ0FBYSxTQUFiLENBakVFO0FBbUVYclIsRUFBQUEsTUFBTSxFQUFFO0FBQ055UixJQUFBQSxPQUFPLEVBQUVuUCxPQUFPLElBQUtBLE9BQU8sQ0FBQzBMLEtBQVIsR0FBZ0IsR0FBaEIsR0FBc0IsRUFEckM7QUFHTjNKLElBQUFBLE9BQU8sRUFBRSxDQUFDd0IsT0FBRCxFQUFVdkQsT0FBVixLQUFzQjtBQUM3QixZQUFNb1AsTUFBTSxHQUFHNVMsSUFBSSxDQUFDa0IsTUFBTCxDQUFZeVIsT0FBWixDQUFvQm5QLE9BQXBCLENBQWY7O0FBQ0EsYUFBTyxJQUFJK08sUUFBSixDQUFjLFVBQVNLLE1BQU8sSUFBRzdMLE9BQVEsRUFBekMsRUFBNEMsQ0FBQyxRQUFELEVBQVksVUFBUzZMLE1BQU8sRUFBNUIsQ0FBNUMsQ0FBUDtBQUNELEtBTks7QUFRTmhNLElBQUFBLGVBQWUsRUFBRUcsT0FBTyxJQUFJLENBQzFCL0csSUFBSSxDQUFDa0IsTUFBTCxDQUFZcUUsT0FBWixDQUFvQndCLE9BQXBCLEVBQTZCO0FBQUNtSSxNQUFBQSxLQUFLLEVBQUU7QUFBUixLQUE3QixDQUQwQixFQUUxQmxQLElBQUksQ0FBQ2tCLE1BQUwsQ0FBWXFFLE9BQVosQ0FBb0J3QixPQUFwQixFQUE2QjtBQUFDbUksTUFBQUEsS0FBSyxFQUFFO0FBQVIsS0FBN0IsQ0FGMEIsQ0FSdEI7QUFhTnhPLElBQUFBLEdBQUcsRUFBRSxJQUFJK1IsUUFBSixDQUFhLFFBQWI7QUFiQyxHQW5FRztBQW1GWHZHLEVBQUFBLElBQUksRUFBRTtBQUNKM0csSUFBQUEsT0FBTyxFQUFFNEIsR0FBRyxJQUFJLElBQUlvTCxRQUFKLENBQWMsUUFBT3BMLEdBQUksRUFBekIsRUFBNEIsQ0FBQyxNQUFELENBQTVCO0FBRFosR0FuRks7QUF1Rlg7QUFFQTdELEVBQUFBLG9CQUFvQixFQUFFd1AsU0FBUyxJQUFJLENBQ2pDOVMsSUFBSSxDQUFDQyxZQUQ0QixFQUVqQyxHQUFHRCxJQUFJLENBQUNHLFNBQUwsQ0FBZW1CLGdCQUFmLENBQWdDd1IsU0FBaEMsRUFBMkMsQ0FBQztBQUFDelMsSUFBQUEsTUFBTSxFQUFFO0FBQVQsR0FBRCxDQUEzQyxDQUY4QixDQXpGeEI7QUE4RlhtQyxFQUFBQSxrQkFBa0IsRUFBRXNRLFNBQVMsSUFBSSxDQUMvQixHQUFHOVMsSUFBSSxDQUFDc0Qsb0JBQUwsQ0FBMEJ3UCxTQUExQixDQUQ0QixFQUUvQixHQUFHOVMsSUFBSSxDQUFDRyxTQUFMLENBQWVtQixnQkFBZixDQUFnQ3dSLFNBQWhDLEVBQTJDLENBQUM7QUFBQ3pTLElBQUFBLE1BQU0sRUFBRTtBQUFULEdBQUQsQ0FBM0MsQ0FGNEIsRUFHL0IsR0FBR3lTLFNBQVMsQ0FBQ3pULEdBQVYsQ0FBY1csSUFBSSxDQUFDVyxLQUFMLENBQVc0RSxPQUF6QixDQUg0QixFQUkvQnZGLElBQUksQ0FBQ1MsYUFKMEIsQ0E5RnRCO0FBcUdYZ0QsRUFBQUEsaUJBQWlCLEVBQUUsTUFBTSxDQUN2QnpELElBQUksQ0FBQ2UsZUFEa0IsRUFFdkJmLElBQUksQ0FBQ1ksUUFGa0IsRUFHdkIsR0FBR1osSUFBSSxDQUFDRyxTQUFMLENBQWVDLFlBQWYsQ0FBNEI7QUFBQ0MsSUFBQUEsTUFBTSxFQUFFO0FBQVQsR0FBNUIsQ0FIb0IsRUFJdkJMLElBQUksQ0FBQ0csU0FBTCxDQUFldUYsaUJBSlEsRUFLdkIxRixJQUFJLENBQUNTLGFBTGtCLEVBTXZCVCxJQUFJLENBQUNhLFVBTmtCLEVBT3ZCYixJQUFJLENBQUNjLGFBUGtCLEVBUXZCZCxJQUFJLENBQUNnQixPQVJrQixFQVN2QmhCLElBQUksQ0FBQ0MsWUFUa0I7QUFyR2QsQ0FBYiIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS0xLjQxLjAvb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQge0VtaXR0ZXJ9IGZyb20gJ2V2ZW50LWtpdCc7XG5pbXBvcnQgZnMgZnJvbSAnZnMtZXh0cmEnO1xuXG5pbXBvcnQgU3RhdGUgZnJvbSAnLi9zdGF0ZSc7XG5cbmltcG9ydCB7TGFyZ2VSZXBvRXJyb3J9IGZyb20gJy4uLy4uL2dpdC1zaGVsbC1vdXQtc3RyYXRlZ3knO1xuaW1wb3J0IHtGT0NVU30gZnJvbSAnLi4vd29ya3NwYWNlLWNoYW5nZS1vYnNlcnZlcic7XG5pbXBvcnQge2J1aWxkRmlsZVBhdGNoLCBidWlsZE11bHRpRmlsZVBhdGNofSBmcm9tICcuLi9wYXRjaCc7XG5pbXBvcnQgRGlzY2FyZEhpc3RvcnkgZnJvbSAnLi4vZGlzY2FyZC1oaXN0b3J5JztcbmltcG9ydCBCcmFuY2gsIHtudWxsQnJhbmNofSBmcm9tICcuLi9icmFuY2gnO1xuaW1wb3J0IEF1dGhvciBmcm9tICcuLi9hdXRob3InO1xuaW1wb3J0IEJyYW5jaFNldCBmcm9tICcuLi9icmFuY2gtc2V0JztcbmltcG9ydCBSZW1vdGUgZnJvbSAnLi4vcmVtb3RlJztcbmltcG9ydCBSZW1vdGVTZXQgZnJvbSAnLi4vcmVtb3RlLXNldCc7XG5pbXBvcnQgQ29tbWl0IGZyb20gJy4uL2NvbW1pdCc7XG5pbXBvcnQgT3BlcmF0aW9uU3RhdGVzIGZyb20gJy4uL29wZXJhdGlvbi1zdGF0ZXMnO1xuaW1wb3J0IHthZGRFdmVudH0gZnJvbSAnLi4vLi4vcmVwb3J0ZXItcHJveHknO1xuaW1wb3J0IHtmaWxlUGF0aEVuZHNXaXRofSBmcm9tICcuLi8uLi9oZWxwZXJzJztcblxuLyoqXG4gKiBTdGF0ZSB1c2VkIHdoZW4gdGhlIHdvcmtpbmcgZGlyZWN0b3J5IGNvbnRhaW5zIGEgdmFsaWQgZ2l0IHJlcG9zaXRvcnkgYW5kIGNhbiBiZSBpbnRlcmFjdGVkIHdpdGguIFBlcmZvcm1zXG4gKiBhY3R1YWwgZ2l0IG9wZXJhdGlvbnMsIGNhY2hpbmcgdGhlIHJlc3VsdHMsIGFuZCBicm9hZGNhc3RzIGBvbkRpZFVwZGF0ZWAgZXZlbnRzIHdoZW4gd3JpdGUgYWN0aW9ucyBhcmVcbiAqIHBlcmZvcm1lZC5cbiAqL1xuZXhwb3J0IGRlZmF1bHQgY2xhc3MgUHJlc2VudCBleHRlbmRzIFN0YXRlIHtcbiAgY29uc3RydWN0b3IocmVwb3NpdG9yeSwgaGlzdG9yeSkge1xuICAgIHN1cGVyKHJlcG9zaXRvcnkpO1xuXG4gICAgdGhpcy5jYWNoZSA9IG5ldyBDYWNoZSgpO1xuXG4gICAgdGhpcy5kaXNjYXJkSGlzdG9yeSA9IG5ldyBEaXNjYXJkSGlzdG9yeShcbiAgICAgIHRoaXMuY3JlYXRlQmxvYi5iaW5kKHRoaXMpLFxuICAgICAgdGhpcy5leHBhbmRCbG9iVG9GaWxlLmJpbmQodGhpcyksXG4gICAgICB0aGlzLm1lcmdlRmlsZS5iaW5kKHRoaXMpLFxuICAgICAgdGhpcy53b3JrZGlyKCksXG4gICAgICB7bWF4SGlzdG9yeUxlbmd0aDogNjB9LFxuICAgICk7XG5cbiAgICB0aGlzLm9wZXJhdGlvblN0YXRlcyA9IG5ldyBPcGVyYXRpb25TdGF0ZXMoe2RpZFVwZGF0ZTogdGhpcy5kaWRVcGRhdGUuYmluZCh0aGlzKX0pO1xuXG4gICAgdGhpcy5jb21taXRNZXNzYWdlID0gJyc7XG4gICAgdGhpcy5jb21taXRNZXNzYWdlVGVtcGxhdGUgPSBudWxsO1xuICAgIHRoaXMuZmV0Y2hJbml0aWFsTWVzc2FnZSgpO1xuXG4gICAgLyogaXN0YW5idWwgaWdub3JlIGVsc2UgKi9cbiAgICBpZiAoaGlzdG9yeSkge1xuICAgICAgdGhpcy5kaXNjYXJkSGlzdG9yeS51cGRhdGVIaXN0b3J5KGhpc3RvcnkpO1xuICAgIH1cbiAgfVxuXG4gIHNldENvbW1pdE1lc3NhZ2UobWVzc2FnZSwge3N1cHByZXNzVXBkYXRlfSA9IHtzdXBwcmVzc1VwZGF0ZTogZmFsc2V9KSB7XG4gICAgdGhpcy5jb21taXRNZXNzYWdlID0gbWVzc2FnZTtcbiAgICBpZiAoIXN1cHByZXNzVXBkYXRlKSB7XG4gICAgICB0aGlzLmRpZFVwZGF0ZSgpO1xuICAgIH1cbiAgfVxuXG4gIHNldENvbW1pdE1lc3NhZ2VUZW1wbGF0ZSh0ZW1wbGF0ZSkge1xuICAgIHRoaXMuY29tbWl0TWVzc2FnZVRlbXBsYXRlID0gdGVtcGxhdGU7XG4gIH1cblxuICBhc3luYyBmZXRjaEluaXRpYWxNZXNzYWdlKCkge1xuICAgIGNvbnN0IG1lcmdlTWVzc2FnZSA9IGF3YWl0IHRoaXMucmVwb3NpdG9yeS5nZXRNZXJnZU1lc3NhZ2UoKTtcbiAgICBjb25zdCB0ZW1wbGF0ZSA9IGF3YWl0IHRoaXMuZmV0Y2hDb21taXRNZXNzYWdlVGVtcGxhdGUoKTtcbiAgICBpZiAodGVtcGxhdGUpIHtcbiAgICAgIHRoaXMuY29tbWl0TWVzc2FnZVRlbXBsYXRlID0gdGVtcGxhdGU7XG4gICAgfVxuICAgIGlmIChtZXJnZU1lc3NhZ2UpIHtcbiAgICAgIHRoaXMuc2V0Q29tbWl0TWVzc2FnZShtZXJnZU1lc3NhZ2UpO1xuICAgIH0gZWxzZSBpZiAodGVtcGxhdGUpIHtcbiAgICAgIHRoaXMuc2V0Q29tbWl0TWVzc2FnZSh0ZW1wbGF0ZSk7XG4gICAgfVxuICB9XG5cbiAgZ2V0Q29tbWl0TWVzc2FnZSgpIHtcbiAgICByZXR1cm4gdGhpcy5jb21taXRNZXNzYWdlO1xuICB9XG5cbiAgZmV0Y2hDb21taXRNZXNzYWdlVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2l0KCkuZmV0Y2hDb21taXRNZXNzYWdlVGVtcGxhdGUoKTtcbiAgfVxuXG4gIGdldE9wZXJhdGlvblN0YXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy5vcGVyYXRpb25TdGF0ZXM7XG4gIH1cblxuICBpc1ByZXNlbnQoKSB7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICBkZXN0cm95KCkge1xuICAgIHRoaXMuY2FjaGUuZGVzdHJveSgpO1xuICAgIHN1cGVyLmRlc3Ryb3koKTtcbiAgfVxuXG4gIHNob3dTdGF0dXNCYXJUaWxlcygpIHtcbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuXG4gIGFjY2VwdEludmFsaWRhdGlvbihzcGVjKSB7XG4gICAgdGhpcy5jYWNoZS5pbnZhbGlkYXRlKHNwZWMoKSk7XG4gICAgdGhpcy5kaWRVcGRhdGUoKTtcbiAgfVxuXG4gIGludmFsaWRhdGVDYWNoZUFmdGVyRmlsZXN5c3RlbUNoYW5nZShldmVudHMpIHtcbiAgICBjb25zdCBwYXRocyA9IGV2ZW50cy5tYXAoZSA9PiBlLnNwZWNpYWwgfHwgZS5wYXRoKTtcbiAgICBjb25zdCBrZXlzID0gbmV3IFNldCgpO1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgcGF0aHMubGVuZ3RoOyBpKyspIHtcbiAgICAgIGNvbnN0IGZ1bGxQYXRoID0gcGF0aHNbaV07XG5cbiAgICAgIGlmIChmdWxsUGF0aCA9PT0gRk9DVVMpIHtcbiAgICAgICAga2V5cy5hZGQoS2V5cy5zdGF0dXNCdW5kbGUpO1xuICAgICAgICBmb3IgKGNvbnN0IGsgb2YgS2V5cy5maWxlUGF0Y2guZWFjaFdpdGhPcHRzKHtzdGFnZWQ6IGZhbHNlfSkpIHtcbiAgICAgICAgICBrZXlzLmFkZChrKTtcbiAgICAgICAgfVxuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgaW5jbHVkZXMgPSAoLi4uc2VnbWVudHMpID0+IGZ1bGxQYXRoLmluY2x1ZGVzKHBhdGguam9pbiguLi5zZWdtZW50cykpO1xuXG4gICAgICBpZiAoZmlsZVBhdGhFbmRzV2l0aChmdWxsUGF0aCwgJy5naXQnLCAnaW5kZXgnKSkge1xuICAgICAgICBrZXlzLmFkZChLZXlzLnN0YWdlZENoYW5nZXMpO1xuICAgICAgICBrZXlzLmFkZChLZXlzLmZpbGVQYXRjaC5hbGwpO1xuICAgICAgICBrZXlzLmFkZChLZXlzLmluZGV4LmFsbCk7XG4gICAgICAgIGtleXMuYWRkKEtleXMuc3RhdHVzQnVuZGxlKTtcbiAgICAgICAgY29udGludWU7XG4gICAgICB9XG5cbiAgICAgIGlmIChmaWxlUGF0aEVuZHNXaXRoKGZ1bGxQYXRoLCAnLmdpdCcsICdIRUFEJykpIHtcbiAgICAgICAga2V5cy5hZGQoS2V5cy5icmFuY2hlcyk7XG4gICAgICAgIGtleXMuYWRkKEtleXMubGFzdENvbW1pdCk7XG4gICAgICAgIGtleXMuYWRkKEtleXMucmVjZW50Q29tbWl0cyk7XG4gICAgICAgIGtleXMuYWRkKEtleXMuc3RhdHVzQnVuZGxlKTtcbiAgICAgICAga2V5cy5hZGQoS2V5cy5oZWFkRGVzY3JpcHRpb24pO1xuICAgICAgICBrZXlzLmFkZChLZXlzLmF1dGhvcnMpO1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cblxuICAgICAgaWYgKGluY2x1ZGVzKCcuZ2l0JywgJ3JlZnMnLCAnaGVhZHMnKSkge1xuICAgICAgICBrZXlzLmFkZChLZXlzLmJyYW5jaGVzKTtcbiAgICAgICAga2V5cy5hZGQoS2V5cy5sYXN0Q29tbWl0KTtcbiAgICAgICAga2V5cy5hZGQoS2V5cy5yZWNlbnRDb21taXRzKTtcbiAgICAgICAga2V5cy5hZGQoS2V5cy5oZWFkRGVzY3JpcHRpb24pO1xuICAgICAgICBrZXlzLmFkZChLZXlzLmF1dGhvcnMpO1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cblxuICAgICAgaWYgKGluY2x1ZGVzKCcuZ2l0JywgJ3JlZnMnLCAncmVtb3RlcycpKSB7XG4gICAgICAgIGtleXMuYWRkKEtleXMucmVtb3Rlcyk7XG4gICAgICAgIGtleXMuYWRkKEtleXMuc3RhdHVzQnVuZGxlKTtcbiAgICAgICAga2V5cy5hZGQoS2V5cy5oZWFkRGVzY3JpcHRpb24pO1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cblxuICAgICAgaWYgKGZpbGVQYXRoRW5kc1dpdGgoZnVsbFBhdGgsICcuZ2l0JywgJ2NvbmZpZycpKSB7XG4gICAgICAgIGtleXMuYWRkKEtleXMucmVtb3Rlcyk7XG4gICAgICAgIGtleXMuYWRkKEtleXMuY29uZmlnLmFsbCk7XG4gICAgICAgIGtleXMuYWRkKEtleXMuc3RhdHVzQnVuZGxlKTtcbiAgICAgICAgY29udGludWU7XG4gICAgICB9XG5cbiAgICAgIC8vIEZpbGUgY2hhbmdlIHdpdGhpbiB0aGUgd29ya2luZyBkaXJlY3RvcnlcbiAgICAgIGNvbnN0IHJlbGF0aXZlUGF0aCA9IHBhdGgucmVsYXRpdmUodGhpcy53b3JrZGlyKCksIGZ1bGxQYXRoKTtcbiAgICAgIGZvciAoY29uc3Qga2V5IG9mIEtleXMuZmlsZVBhdGNoLmVhY2hXaXRoRmlsZU9wdHMoW3JlbGF0aXZlUGF0aF0sIFt7c3RhZ2VkOiBmYWxzZX1dKSkge1xuICAgICAgICBrZXlzLmFkZChrZXkpO1xuICAgICAgfVxuICAgICAga2V5cy5hZGQoS2V5cy5zdGF0dXNCdW5kbGUpO1xuICAgIH1cblxuICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBlbHNlICovXG4gICAgaWYgKGtleXMuc2l6ZSA+IDApIHtcbiAgICAgIHRoaXMuY2FjaGUuaW52YWxpZGF0ZShBcnJheS5mcm9tKGtleXMpKTtcbiAgICAgIHRoaXMuZGlkVXBkYXRlKCk7XG4gICAgfVxuICB9XG5cbiAgaXNDb21taXRNZXNzYWdlQ2xlYW4oKSB7XG4gICAgaWYgKHRoaXMuY29tbWl0TWVzc2FnZS50cmltKCkgPT09ICcnKSB7XG4gICAgICByZXR1cm4gdHJ1ZTtcbiAgICB9IGVsc2UgaWYgKHRoaXMuY29tbWl0TWVzc2FnZVRlbXBsYXRlKSB7XG4gICAgICByZXR1cm4gdGhpcy5jb21taXRNZXNzYWdlID09PSB0aGlzLmNvbW1pdE1lc3NhZ2VUZW1wbGF0ZTtcbiAgICB9XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgYXN5bmMgdXBkYXRlQ29tbWl0TWVzc2FnZUFmdGVyRmlsZVN5c3RlbUNoYW5nZShldmVudHMpIHtcbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IGV2ZW50cy5sZW5ndGg7IGkrKykge1xuICAgICAgY29uc3QgZXZlbnQgPSBldmVudHNbaV07XG5cbiAgICAgIGlmICghZXZlbnQucGF0aCkge1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cblxuICAgICAgaWYgKGZpbGVQYXRoRW5kc1dpdGgoZXZlbnQucGF0aCwgJy5naXQnLCAnTUVSR0VfSEVBRCcpKSB7XG4gICAgICAgIGlmIChldmVudC5hY3Rpb24gPT09ICdjcmVhdGVkJykge1xuICAgICAgICAgIGlmICh0aGlzLmlzQ29tbWl0TWVzc2FnZUNsZWFuKCkpIHtcbiAgICAgICAgICAgIHRoaXMuc2V0Q29tbWl0TWVzc2FnZShhd2FpdCB0aGlzLnJlcG9zaXRvcnkuZ2V0TWVyZ2VNZXNzYWdlKCkpO1xuICAgICAgICAgIH1cbiAgICAgICAgfSBlbHNlIGlmIChldmVudC5hY3Rpb24gPT09ICdkZWxldGVkJykge1xuICAgICAgICAgIHRoaXMuc2V0Q29tbWl0TWVzc2FnZSh0aGlzLmNvbW1pdE1lc3NhZ2VUZW1wbGF0ZSB8fCAnJyk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgaWYgKGZpbGVQYXRoRW5kc1dpdGgoZXZlbnQucGF0aCwgJy5naXQnLCAnY29uZmlnJykpIHtcbiAgICAgICAgLy8gdGhpcyB3b24ndCBjYXRjaCBjaGFuZ2VzIG1hZGUgdG8gdGhlIHRlbXBsYXRlIGZpbGUgaXRzZWxmLi4uXG4gICAgICAgIGNvbnN0IHRlbXBsYXRlID0gYXdhaXQgdGhpcy5mZXRjaENvbW1pdE1lc3NhZ2VUZW1wbGF0ZSgpO1xuICAgICAgICBpZiAodGVtcGxhdGUgPT09IG51bGwpIHtcbiAgICAgICAgICB0aGlzLnNldENvbW1pdE1lc3NhZ2UoJycpO1xuICAgICAgICB9IGVsc2UgaWYgKHRoaXMuY29tbWl0TWVzc2FnZVRlbXBsYXRlICE9PSB0ZW1wbGF0ZSkge1xuICAgICAgICAgIHRoaXMuc2V0Q29tbWl0TWVzc2FnZSh0ZW1wbGF0ZSk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5zZXRDb21taXRNZXNzYWdlVGVtcGxhdGUodGVtcGxhdGUpO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIG9ic2VydmVGaWxlc3lzdGVtQ2hhbmdlKGV2ZW50cykge1xuICAgIHRoaXMuaW52YWxpZGF0ZUNhY2hlQWZ0ZXJGaWxlc3lzdGVtQ2hhbmdlKGV2ZW50cyk7XG4gICAgdGhpcy51cGRhdGVDb21taXRNZXNzYWdlQWZ0ZXJGaWxlU3lzdGVtQ2hhbmdlKGV2ZW50cyk7XG4gIH1cblxuICByZWZyZXNoKCkge1xuICAgIHRoaXMuY2FjaGUuY2xlYXIoKTtcbiAgICB0aGlzLmRpZFVwZGF0ZSgpO1xuICB9XG5cbiAgaW5pdCgpIHtcbiAgICByZXR1cm4gc3VwZXIuaW5pdCgpLmNhdGNoKGUgPT4ge1xuICAgICAgZS5zdGRFcnIgPSAnVGhpcyBkaXJlY3RvcnkgYWxyZWFkeSBjb250YWlucyBhIGdpdCByZXBvc2l0b3J5JztcbiAgICAgIHJldHVybiBQcm9taXNlLnJlamVjdChlKTtcbiAgICB9KTtcbiAgfVxuXG4gIGNsb25lKCkge1xuICAgIHJldHVybiBzdXBlci5jbG9uZSgpLmNhdGNoKGUgPT4ge1xuICAgICAgZS5zdGRFcnIgPSAnVGhpcyBkaXJlY3RvcnkgYWxyZWFkeSBjb250YWlucyBhIGdpdCByZXBvc2l0b3J5JztcbiAgICAgIHJldHVybiBQcm9taXNlLnJlamVjdChlKTtcbiAgICB9KTtcbiAgfVxuXG4gIC8vIEdpdCBvcGVyYXRpb25zIC8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy9cblxuICAvLyBTdGFnaW5nIGFuZCB1bnN0YWdpbmdcblxuICBzdGFnZUZpbGVzKHBhdGhzKSB7XG4gICAgcmV0dXJuIHRoaXMuaW52YWxpZGF0ZShcbiAgICAgICgpID0+IEtleXMuY2FjaGVPcGVyYXRpb25LZXlzKHBhdGhzKSxcbiAgICAgICgpID0+IHRoaXMuZ2l0KCkuc3RhZ2VGaWxlcyhwYXRocyksXG4gICAgKTtcbiAgfVxuXG4gIHVuc3RhZ2VGaWxlcyhwYXRocykge1xuICAgIHJldHVybiB0aGlzLmludmFsaWRhdGUoXG4gICAgICAoKSA9PiBLZXlzLmNhY2hlT3BlcmF0aW9uS2V5cyhwYXRocyksXG4gICAgICAoKSA9PiB0aGlzLmdpdCgpLnVuc3RhZ2VGaWxlcyhwYXRocyksXG4gICAgKTtcbiAgfVxuXG4gIHN0YWdlRmlsZXNGcm9tUGFyZW50Q29tbWl0KHBhdGhzKSB7XG4gICAgcmV0dXJuIHRoaXMuaW52YWxpZGF0ZShcbiAgICAgICgpID0+IEtleXMuY2FjaGVPcGVyYXRpb25LZXlzKHBhdGhzKSxcbiAgICAgICgpID0+IHRoaXMuZ2l0KCkudW5zdGFnZUZpbGVzKHBhdGhzLCAnSEVBRH4nKSxcbiAgICApO1xuICB9XG5cbiAgc3RhZ2VGaWxlTW9kZUNoYW5nZShmaWxlUGF0aCwgZmlsZU1vZGUpIHtcbiAgICByZXR1cm4gdGhpcy5pbnZhbGlkYXRlKFxuICAgICAgKCkgPT4gS2V5cy5jYWNoZU9wZXJhdGlvbktleXMoW2ZpbGVQYXRoXSksXG4gICAgICAoKSA9PiB0aGlzLmdpdCgpLnN0YWdlRmlsZU1vZGVDaGFuZ2UoZmlsZVBhdGgsIGZpbGVNb2RlKSxcbiAgICApO1xuICB9XG5cbiAgc3RhZ2VGaWxlU3ltbGlua0NoYW5nZShmaWxlUGF0aCkge1xuICAgIHJldHVybiB0aGlzLmludmFsaWRhdGUoXG4gICAgICAoKSA9PiBLZXlzLmNhY2hlT3BlcmF0aW9uS2V5cyhbZmlsZVBhdGhdKSxcbiAgICAgICgpID0+IHRoaXMuZ2l0KCkuc3RhZ2VGaWxlU3ltbGlua0NoYW5nZShmaWxlUGF0aCksXG4gICAgKTtcbiAgfVxuXG4gIGFwcGx5UGF0Y2hUb0luZGV4KG11bHRpRmlsZVBhdGNoKSB7XG4gICAgcmV0dXJuIHRoaXMuaW52YWxpZGF0ZShcbiAgICAgICgpID0+IEtleXMuY2FjaGVPcGVyYXRpb25LZXlzKEFycmF5LmZyb20obXVsdGlGaWxlUGF0Y2guZ2V0UGF0aFNldCgpKSksXG4gICAgICAoKSA9PiB7XG4gICAgICAgIGNvbnN0IHBhdGNoU3RyID0gbXVsdGlGaWxlUGF0Y2gudG9TdHJpbmcoKTtcbiAgICAgICAgcmV0dXJuIHRoaXMuZ2l0KCkuYXBwbHlQYXRjaChwYXRjaFN0ciwge2luZGV4OiB0cnVlfSk7XG4gICAgICB9LFxuICAgICk7XG4gIH1cblxuICBhcHBseVBhdGNoVG9Xb3JrZGlyKG11bHRpRmlsZVBhdGNoKSB7XG4gICAgcmV0dXJuIHRoaXMuaW52YWxpZGF0ZShcbiAgICAgICgpID0+IEtleXMud29ya2Rpck9wZXJhdGlvbktleXMoQXJyYXkuZnJvbShtdWx0aUZpbGVQYXRjaC5nZXRQYXRoU2V0KCkpKSxcbiAgICAgICgpID0+IHtcbiAgICAgICAgY29uc3QgcGF0Y2hTdHIgPSBtdWx0aUZpbGVQYXRjaC50b1N0cmluZygpO1xuICAgICAgICByZXR1cm4gdGhpcy5naXQoKS5hcHBseVBhdGNoKHBhdGNoU3RyKTtcbiAgICAgIH0sXG4gICAgKTtcbiAgfVxuXG4gIC8vIENvbW1pdHRpbmdcblxuICBjb21taXQobWVzc2FnZSwgb3B0aW9ucykge1xuICAgIHJldHVybiB0aGlzLmludmFsaWRhdGUoXG4gICAgICBLZXlzLmhlYWRPcGVyYXRpb25LZXlzLFxuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLXNoYWRvd1xuICAgICAgKCkgPT4gdGhpcy5leGVjdXRlUGlwZWxpbmVBY3Rpb24oJ0NPTU1JVCcsIGFzeW5jIChtZXNzYWdlLCBvcHRpb25zID0ge30pID0+IHtcbiAgICAgICAgY29uc3QgY29BdXRob3JzID0gb3B0aW9ucy5jb0F1dGhvcnM7XG4gICAgICAgIGNvbnN0IG9wdHMgPSAhY29BdXRob3JzID8gb3B0aW9ucyA6IHtcbiAgICAgICAgICAuLi5vcHRpb25zLFxuICAgICAgICAgIGNvQXV0aG9yczogY29BdXRob3JzLm1hcChhdXRob3IgPT4ge1xuICAgICAgICAgICAgcmV0dXJuIHtlbWFpbDogYXV0aG9yLmdldEVtYWlsKCksIG5hbWU6IGF1dGhvci5nZXRGdWxsTmFtZSgpfTtcbiAgICAgICAgICB9KSxcbiAgICAgICAgfTtcblxuICAgICAgICBhd2FpdCB0aGlzLmdpdCgpLmNvbW1pdChtZXNzYWdlLCBvcHRzKTtcblxuICAgICAgICAvLyBDb2xsZWN0IGNvbW1pdCBtZXRhZGF0YSBtZXRyaWNzXG4gICAgICAgIC8vIG5vdGU6IGluIEdpdFNoZWxsT3V0U3RyYXRlZ3kgd2UgaGF2ZSBjb3VudGVycyBmb3IgYWxsIGdpdCBjb21tYW5kcywgaW5jbHVkaW5nIGBjb21taXRgLCBidXQgaGVyZSB3ZSBoYXZlXG4gICAgICAgIC8vICAgICAgIGFjY2VzcyB0byBhZGRpdGlvbmFsIG1ldGFkYXRhICh1bnN0YWdlZCBmaWxlIGNvdW50KSBzbyBpdCBtYWtlcyBzZW5zZSB0byBjb2xsZWN0IGNvbW1pdCBldmVudHMgaGVyZVxuICAgICAgICBjb25zdCB7dW5zdGFnZWRGaWxlcywgbWVyZ2VDb25mbGljdEZpbGVzfSA9IGF3YWl0IHRoaXMuZ2V0U3RhdHVzZXNGb3JDaGFuZ2VkRmlsZXMoKTtcbiAgICAgICAgY29uc3QgdW5zdGFnZWRDb3VudCA9IE9iamVjdC5rZXlzKHsuLi51bnN0YWdlZEZpbGVzLCAuLi5tZXJnZUNvbmZsaWN0RmlsZXN9KS5sZW5ndGg7XG4gICAgICAgIGFkZEV2ZW50KCdjb21taXQnLCB7XG4gICAgICAgICAgcGFja2FnZTogJ2dpdGh1YicsXG4gICAgICAgICAgcGFydGlhbDogdW5zdGFnZWRDb3VudCA+IDAsXG4gICAgICAgICAgYW1lbmQ6ICEhb3B0aW9ucy5hbWVuZCxcbiAgICAgICAgICBjb0F1dGhvckNvdW50OiBjb0F1dGhvcnMgPyBjb0F1dGhvcnMubGVuZ3RoIDogMCxcbiAgICAgICAgfSk7XG4gICAgICB9LCBtZXNzYWdlLCBvcHRpb25zKSxcbiAgICApO1xuICB9XG5cbiAgLy8gTWVyZ2luZ1xuXG4gIG1lcmdlKGJyYW5jaE5hbWUpIHtcbiAgICByZXR1cm4gdGhpcy5pbnZhbGlkYXRlKFxuICAgICAgKCkgPT4gW1xuICAgICAgICAuLi5LZXlzLmhlYWRPcGVyYXRpb25LZXlzKCksXG4gICAgICAgIEtleXMuaW5kZXguYWxsLFxuICAgICAgICBLZXlzLmhlYWREZXNjcmlwdGlvbixcbiAgICAgIF0sXG4gICAgICAoKSA9PiB0aGlzLmdpdCgpLm1lcmdlKGJyYW5jaE5hbWUpLFxuICAgICk7XG4gIH1cblxuICBhYm9ydE1lcmdlKCkge1xuICAgIHJldHVybiB0aGlzLmludmFsaWRhdGUoXG4gICAgICAoKSA9PiBbXG4gICAgICAgIEtleXMuc3RhdHVzQnVuZGxlLFxuICAgICAgICBLZXlzLnN0YWdlZENoYW5nZXMsXG4gICAgICAgIEtleXMuZmlsZVBhdGNoLmFsbCxcbiAgICAgICAgS2V5cy5pbmRleC5hbGwsXG4gICAgICBdLFxuICAgICAgYXN5bmMgKCkgPT4ge1xuICAgICAgICBhd2FpdCB0aGlzLmdpdCgpLmFib3J0TWVyZ2UoKTtcbiAgICAgICAgdGhpcy5zZXRDb21taXRNZXNzYWdlKHRoaXMuY29tbWl0TWVzc2FnZVRlbXBsYXRlIHx8ICcnKTtcbiAgICAgIH0sXG4gICAgKTtcbiAgfVxuXG4gIGNoZWNrb3V0U2lkZShzaWRlLCBwYXRocykge1xuICAgIHJldHVybiB0aGlzLmdpdCgpLmNoZWNrb3V0U2lkZShzaWRlLCBwYXRocyk7XG4gIH1cblxuICBtZXJnZUZpbGUob3Vyc1BhdGgsIGNvbW1vbkJhc2VQYXRoLCB0aGVpcnNQYXRoLCByZXN1bHRQYXRoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2l0KCkubWVyZ2VGaWxlKG91cnNQYXRoLCBjb21tb25CYXNlUGF0aCwgdGhlaXJzUGF0aCwgcmVzdWx0UGF0aCk7XG4gIH1cblxuICB3cml0ZU1lcmdlQ29uZmxpY3RUb0luZGV4KGZpbGVQYXRoLCBjb21tb25CYXNlU2hhLCBvdXJzU2hhLCB0aGVpcnNTaGEpIHtcbiAgICByZXR1cm4gdGhpcy5pbnZhbGlkYXRlKFxuICAgICAgKCkgPT4gW1xuICAgICAgICBLZXlzLnN0YXR1c0J1bmRsZSxcbiAgICAgICAgS2V5cy5zdGFnZWRDaGFuZ2VzLFxuICAgICAgICAuLi5LZXlzLmZpbGVQYXRjaC5lYWNoV2l0aEZpbGVPcHRzKFtmaWxlUGF0aF0sIFt7c3RhZ2VkOiBmYWxzZX0sIHtzdGFnZWQ6IHRydWV9XSksXG4gICAgICAgIEtleXMuaW5kZXgub25lV2l0aChmaWxlUGF0aCksXG4gICAgICBdLFxuICAgICAgKCkgPT4gdGhpcy5naXQoKS53cml0ZU1lcmdlQ29uZmxpY3RUb0luZGV4KGZpbGVQYXRoLCBjb21tb25CYXNlU2hhLCBvdXJzU2hhLCB0aGVpcnNTaGEpLFxuICAgICk7XG4gIH1cblxuICAvLyBDaGVja291dFxuXG4gIGNoZWNrb3V0KHJldmlzaW9uLCBvcHRpb25zID0ge30pIHtcbiAgICByZXR1cm4gdGhpcy5pbnZhbGlkYXRlKFxuICAgICAgKCkgPT4gW1xuICAgICAgICBLZXlzLnN0YWdlZENoYW5nZXMsXG4gICAgICAgIEtleXMubGFzdENvbW1pdCxcbiAgICAgICAgS2V5cy5yZWNlbnRDb21taXRzLFxuICAgICAgICBLZXlzLmF1dGhvcnMsXG4gICAgICAgIEtleXMuc3RhdHVzQnVuZGxlLFxuICAgICAgICBLZXlzLmluZGV4LmFsbCxcbiAgICAgICAgLi4uS2V5cy5maWxlUGF0Y2guZWFjaFdpdGhPcHRzKHtzdGFnZWQ6IHRydWV9KSxcbiAgICAgICAgS2V5cy5maWxlUGF0Y2guYWxsQWdhaW5zdE5vbkhlYWQsXG4gICAgICAgIEtleXMuaGVhZERlc2NyaXB0aW9uLFxuICAgICAgICBLZXlzLmJyYW5jaGVzLFxuICAgICAgXSxcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1zaGFkb3dcbiAgICAgICgpID0+IHRoaXMuZXhlY3V0ZVBpcGVsaW5lQWN0aW9uKCdDSEVDS09VVCcsIChyZXZpc2lvbiwgb3B0aW9ucykgPT4ge1xuICAgICAgICByZXR1cm4gdGhpcy5naXQoKS5jaGVja291dChyZXZpc2lvbiwgb3B0aW9ucyk7XG4gICAgICB9LCByZXZpc2lvbiwgb3B0aW9ucyksXG4gICAgKTtcbiAgfVxuXG4gIGNoZWNrb3V0UGF0aHNBdFJldmlzaW9uKHBhdGhzLCByZXZpc2lvbiA9ICdIRUFEJykge1xuICAgIHJldHVybiB0aGlzLmludmFsaWRhdGUoXG4gICAgICAoKSA9PiBbXG4gICAgICAgIEtleXMuc3RhdHVzQnVuZGxlLFxuICAgICAgICBLZXlzLnN0YWdlZENoYW5nZXMsXG4gICAgICAgIC4uLnBhdGhzLm1hcChmaWxlTmFtZSA9PiBLZXlzLmluZGV4Lm9uZVdpdGgoZmlsZU5hbWUpKSxcbiAgICAgICAgLi4uS2V5cy5maWxlUGF0Y2guZWFjaFdpdGhGaWxlT3B0cyhwYXRocywgW3tzdGFnZWQ6IHRydWV9XSksXG4gICAgICAgIC4uLktleXMuZmlsZVBhdGNoLmVhY2hOb25IZWFkV2l0aEZpbGVzKHBhdGhzKSxcbiAgICAgIF0sXG4gICAgICAoKSA9PiB0aGlzLmdpdCgpLmNoZWNrb3V0RmlsZXMocGF0aHMsIHJldmlzaW9uKSxcbiAgICApO1xuICB9XG5cbiAgLy8gUmVzZXRcblxuICB1bmRvTGFzdENvbW1pdCgpIHtcbiAgICByZXR1cm4gdGhpcy5pbnZhbGlkYXRlKFxuICAgICAgKCkgPT4gW1xuICAgICAgICBLZXlzLnN0YWdlZENoYW5nZXMsXG4gICAgICAgIEtleXMubGFzdENvbW1pdCxcbiAgICAgICAgS2V5cy5yZWNlbnRDb21taXRzLFxuICAgICAgICBLZXlzLmF1dGhvcnMsXG4gICAgICAgIEtleXMuc3RhdHVzQnVuZGxlLFxuICAgICAgICBLZXlzLmluZGV4LmFsbCxcbiAgICAgICAgLi4uS2V5cy5maWxlUGF0Y2guZWFjaFdpdGhPcHRzKHtzdGFnZWQ6IHRydWV9KSxcbiAgICAgICAgS2V5cy5oZWFkRGVzY3JpcHRpb24sXG4gICAgICBdLFxuICAgICAgYXN5bmMgKCkgPT4ge1xuICAgICAgICB0cnkge1xuICAgICAgICAgIGF3YWl0IHRoaXMuZ2l0KCkucmVzZXQoJ3NvZnQnLCAnSEVBRH4nKTtcbiAgICAgICAgICBhZGRFdmVudCgndW5kby1sYXN0LWNvbW1pdCcsIHtwYWNrYWdlOiAnZ2l0aHViJ30pO1xuICAgICAgICB9IGNhdGNoIChlKSB7XG4gICAgICAgICAgaWYgKC91bmtub3duIHJldmlzaW9uLy50ZXN0KGUuc3RkRXJyKSkge1xuICAgICAgICAgICAgLy8gSW5pdGlhbCBjb21taXRcbiAgICAgICAgICAgIGF3YWl0IHRoaXMuZ2l0KCkuZGVsZXRlUmVmKCdIRUFEJyk7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHRocm93IGU7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICk7XG4gIH1cblxuICAvLyBSZW1vdGUgaW50ZXJhY3Rpb25zXG5cbiAgZmV0Y2goYnJhbmNoTmFtZSwgb3B0aW9ucyA9IHt9KSB7XG4gICAgcmV0dXJuIHRoaXMuaW52YWxpZGF0ZShcbiAgICAgICgpID0+IFtcbiAgICAgICAgS2V5cy5zdGF0dXNCdW5kbGUsXG4gICAgICAgIEtleXMuaGVhZERlc2NyaXB0aW9uLFxuICAgICAgXSxcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1zaGFkb3dcbiAgICAgICgpID0+IHRoaXMuZXhlY3V0ZVBpcGVsaW5lQWN0aW9uKCdGRVRDSCcsIGFzeW5jIGJyYW5jaE5hbWUgPT4ge1xuICAgICAgICBsZXQgZmluYWxSZW1vdGVOYW1lID0gb3B0aW9ucy5yZW1vdGVOYW1lO1xuICAgICAgICBpZiAoIWZpbmFsUmVtb3RlTmFtZSkge1xuICAgICAgICAgIGNvbnN0IHJlbW90ZSA9IGF3YWl0IHRoaXMuZ2V0UmVtb3RlRm9yQnJhbmNoKGJyYW5jaE5hbWUpO1xuICAgICAgICAgIGlmICghcmVtb3RlLmlzUHJlc2VudCgpKSB7XG4gICAgICAgICAgICByZXR1cm4gbnVsbDtcbiAgICAgICAgICB9XG4gICAgICAgICAgZmluYWxSZW1vdGVOYW1lID0gcmVtb3RlLmdldE5hbWUoKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdGhpcy5naXQoKS5mZXRjaChmaW5hbFJlbW90ZU5hbWUsIGJyYW5jaE5hbWUpO1xuICAgICAgfSwgYnJhbmNoTmFtZSksXG4gICAgKTtcbiAgfVxuXG4gIHB1bGwoYnJhbmNoTmFtZSwgb3B0aW9ucyA9IHt9KSB7XG4gICAgcmV0dXJuIHRoaXMuaW52YWxpZGF0ZShcbiAgICAgICgpID0+IFtcbiAgICAgICAgLi4uS2V5cy5oZWFkT3BlcmF0aW9uS2V5cygpLFxuICAgICAgICBLZXlzLmluZGV4LmFsbCxcbiAgICAgICAgS2V5cy5oZWFkRGVzY3JpcHRpb24sXG4gICAgICAgIEtleXMuYnJhbmNoZXMsXG4gICAgICBdLFxuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLXNoYWRvd1xuICAgICAgKCkgPT4gdGhpcy5leGVjdXRlUGlwZWxpbmVBY3Rpb24oJ1BVTEwnLCBhc3luYyBicmFuY2hOYW1lID0+IHtcbiAgICAgICAgbGV0IGZpbmFsUmVtb3RlTmFtZSA9IG9wdGlvbnMucmVtb3RlTmFtZTtcbiAgICAgICAgaWYgKCFmaW5hbFJlbW90ZU5hbWUpIHtcbiAgICAgICAgICBjb25zdCByZW1vdGUgPSBhd2FpdCB0aGlzLmdldFJlbW90ZUZvckJyYW5jaChicmFuY2hOYW1lKTtcbiAgICAgICAgICBpZiAoIXJlbW90ZS5pc1ByZXNlbnQoKSkge1xuICAgICAgICAgICAgcmV0dXJuIG51bGw7XG4gICAgICAgICAgfVxuICAgICAgICAgIGZpbmFsUmVtb3RlTmFtZSA9IHJlbW90ZS5nZXROYW1lKCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuZ2l0KCkucHVsbChmaW5hbFJlbW90ZU5hbWUsIGJyYW5jaE5hbWUsIG9wdGlvbnMpO1xuICAgICAgfSwgYnJhbmNoTmFtZSksXG4gICAgKTtcbiAgfVxuXG4gIHB1c2goYnJhbmNoTmFtZSwgb3B0aW9ucyA9IHt9KSB7XG4gICAgcmV0dXJuIHRoaXMuaW52YWxpZGF0ZShcbiAgICAgICgpID0+IHtcbiAgICAgICAgY29uc3Qga2V5cyA9IFtcbiAgICAgICAgICBLZXlzLnN0YXR1c0J1bmRsZSxcbiAgICAgICAgICBLZXlzLmhlYWREZXNjcmlwdGlvbixcbiAgICAgICAgXTtcblxuICAgICAgICBpZiAob3B0aW9ucy5zZXRVcHN0cmVhbSkge1xuICAgICAgICAgIGtleXMucHVzaChLZXlzLmJyYW5jaGVzKTtcbiAgICAgICAgICBrZXlzLnB1c2goLi4uS2V5cy5jb25maWcuZWFjaFdpdGhTZXR0aW5nKGBicmFuY2guJHticmFuY2hOYW1lfS5yZW1vdGVgKSk7XG4gICAgICAgIH1cblxuICAgICAgICByZXR1cm4ga2V5cztcbiAgICAgIH0sXG4gICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tc2hhZG93XG4gICAgICAoKSA9PiB0aGlzLmV4ZWN1dGVQaXBlbGluZUFjdGlvbignUFVTSCcsIGFzeW5jIChicmFuY2hOYW1lLCBvcHRpb25zKSA9PiB7XG4gICAgICAgIGNvbnN0IHJlbW90ZSA9IG9wdGlvbnMucmVtb3RlIHx8IGF3YWl0IHRoaXMuZ2V0UmVtb3RlRm9yQnJhbmNoKGJyYW5jaE5hbWUpO1xuICAgICAgICByZXR1cm4gdGhpcy5naXQoKS5wdXNoKHJlbW90ZS5nZXROYW1lT3IoJ29yaWdpbicpLCBicmFuY2hOYW1lLCBvcHRpb25zKTtcbiAgICAgIH0sIGJyYW5jaE5hbWUsIG9wdGlvbnMpLFxuICAgICk7XG4gIH1cblxuICAvLyBDb25maWd1cmF0aW9uXG5cbiAgc2V0Q29uZmlnKHNldHRpbmcsIHZhbHVlLCBvcHRpb25zKSB7XG4gICAgcmV0dXJuIHRoaXMuaW52YWxpZGF0ZShcbiAgICAgICgpID0+IEtleXMuY29uZmlnLmVhY2hXaXRoU2V0dGluZyhzZXR0aW5nKSxcbiAgICAgICgpID0+IHRoaXMuZ2l0KCkuc2V0Q29uZmlnKHNldHRpbmcsIHZhbHVlLCBvcHRpb25zKSxcbiAgICApO1xuICB9XG5cbiAgdW5zZXRDb25maWcoc2V0dGluZykge1xuICAgIHJldHVybiB0aGlzLmludmFsaWRhdGUoXG4gICAgICAoKSA9PiBLZXlzLmNvbmZpZy5lYWNoV2l0aFNldHRpbmcoc2V0dGluZyksXG4gICAgICAoKSA9PiB0aGlzLmdpdCgpLnVuc2V0Q29uZmlnKHNldHRpbmcpLFxuICAgICk7XG4gIH1cblxuICAvLyBEaXJlY3QgYmxvYiBpbnRlcmFjdGlvbnNcblxuICBjcmVhdGVCbG9iKG9wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5naXQoKS5jcmVhdGVCbG9iKG9wdGlvbnMpO1xuICB9XG5cbiAgZXhwYW5kQmxvYlRvRmlsZShhYnNGaWxlUGF0aCwgc2hhKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2l0KCkuZXhwYW5kQmxvYlRvRmlsZShhYnNGaWxlUGF0aCwgc2hhKTtcbiAgfVxuXG4gIC8vIERpc2NhcmQgaGlzdG9yeVxuXG4gIGNyZWF0ZURpc2NhcmRIaXN0b3J5QmxvYigpIHtcbiAgICByZXR1cm4gdGhpcy5kaXNjYXJkSGlzdG9yeS5jcmVhdGVIaXN0b3J5QmxvYigpO1xuICB9XG5cbiAgYXN5bmMgdXBkYXRlRGlzY2FyZEhpc3RvcnkoKSB7XG4gICAgY29uc3QgaGlzdG9yeSA9IGF3YWl0IHRoaXMubG9hZEhpc3RvcnlQYXlsb2FkKCk7XG4gICAgdGhpcy5kaXNjYXJkSGlzdG9yeS51cGRhdGVIaXN0b3J5KGhpc3RvcnkpO1xuICB9XG5cbiAgYXN5bmMgc3RvcmVCZWZvcmVBbmRBZnRlckJsb2JzKGZpbGVQYXRocywgaXNTYWZlLCBkZXN0cnVjdGl2ZUFjdGlvbiwgcGFydGlhbERpc2NhcmRGaWxlUGF0aCA9IG51bGwpIHtcbiAgICBjb25zdCBzbmFwc2hvdHMgPSBhd2FpdCB0aGlzLmRpc2NhcmRIaXN0b3J5LnN0b3JlQmVmb3JlQW5kQWZ0ZXJCbG9icyhcbiAgICAgIGZpbGVQYXRocyxcbiAgICAgIGlzU2FmZSxcbiAgICAgIGRlc3RydWN0aXZlQWN0aW9uLFxuICAgICAgcGFydGlhbERpc2NhcmRGaWxlUGF0aCxcbiAgICApO1xuICAgIC8qIGlzdGFuYnVsIGlnbm9yZSBlbHNlICovXG4gICAgaWYgKHNuYXBzaG90cykge1xuICAgICAgYXdhaXQgdGhpcy5zYXZlRGlzY2FyZEhpc3RvcnkoKTtcbiAgICB9XG4gICAgcmV0dXJuIHNuYXBzaG90cztcbiAgfVxuXG4gIHJlc3RvcmVMYXN0RGlzY2FyZEluVGVtcEZpbGVzKGlzU2FmZSwgcGFydGlhbERpc2NhcmRGaWxlUGF0aCA9IG51bGwpIHtcbiAgICByZXR1cm4gdGhpcy5kaXNjYXJkSGlzdG9yeS5yZXN0b3JlTGFzdERpc2NhcmRJblRlbXBGaWxlcyhpc1NhZmUsIHBhcnRpYWxEaXNjYXJkRmlsZVBhdGgpO1xuICB9XG5cbiAgYXN5bmMgcG9wRGlzY2FyZEhpc3RvcnkocGFydGlhbERpc2NhcmRGaWxlUGF0aCA9IG51bGwpIHtcbiAgICBjb25zdCByZW1vdmVkID0gYXdhaXQgdGhpcy5kaXNjYXJkSGlzdG9yeS5wb3BIaXN0b3J5KHBhcnRpYWxEaXNjYXJkRmlsZVBhdGgpO1xuICAgIGlmIChyZW1vdmVkKSB7XG4gICAgICBhd2FpdCB0aGlzLnNhdmVEaXNjYXJkSGlzdG9yeSgpO1xuICAgIH1cbiAgfVxuXG4gIGNsZWFyRGlzY2FyZEhpc3RvcnkocGFydGlhbERpc2NhcmRGaWxlUGF0aCA9IG51bGwpIHtcbiAgICB0aGlzLmRpc2NhcmRIaXN0b3J5LmNsZWFySGlzdG9yeShwYXJ0aWFsRGlzY2FyZEZpbGVQYXRoKTtcbiAgICByZXR1cm4gdGhpcy5zYXZlRGlzY2FyZEhpc3RvcnkoKTtcbiAgfVxuXG4gIGRpc2NhcmRXb3JrRGlyQ2hhbmdlc0ZvclBhdGhzKHBhdGhzKSB7XG4gICAgcmV0dXJuIHRoaXMuaW52YWxpZGF0ZShcbiAgICAgICgpID0+IFtcbiAgICAgICAgS2V5cy5zdGF0dXNCdW5kbGUsXG4gICAgICAgIC4uLnBhdGhzLm1hcChmaWxlUGF0aCA9PiBLZXlzLmZpbGVQYXRjaC5vbmVXaXRoKGZpbGVQYXRoLCB7c3RhZ2VkOiBmYWxzZX0pKSxcbiAgICAgICAgLi4uS2V5cy5maWxlUGF0Y2guZWFjaE5vbkhlYWRXaXRoRmlsZXMocGF0aHMpLFxuICAgICAgXSxcbiAgICAgIGFzeW5jICgpID0+IHtcbiAgICAgICAgY29uc3QgdW50cmFja2VkRmlsZXMgPSBhd2FpdCB0aGlzLmdpdCgpLmdldFVudHJhY2tlZEZpbGVzKCk7XG4gICAgICAgIGNvbnN0IFtmaWxlc1RvUmVtb3ZlLCBmaWxlc1RvQ2hlY2tvdXRdID0gcGFydGl0aW9uKHBhdGhzLCBmID0+IHVudHJhY2tlZEZpbGVzLmluY2x1ZGVzKGYpKTtcbiAgICAgICAgYXdhaXQgdGhpcy5naXQoKS5jaGVja291dEZpbGVzKGZpbGVzVG9DaGVja291dCk7XG4gICAgICAgIGF3YWl0IFByb21pc2UuYWxsKGZpbGVzVG9SZW1vdmUubWFwKGZpbGVQYXRoID0+IHtcbiAgICAgICAgICBjb25zdCBhYnNQYXRoID0gcGF0aC5qb2luKHRoaXMud29ya2RpcigpLCBmaWxlUGF0aCk7XG4gICAgICAgICAgcmV0dXJuIGZzLnJlbW92ZShhYnNQYXRoKTtcbiAgICAgICAgfSkpO1xuICAgICAgfSxcbiAgICApO1xuICB9XG5cbiAgLy8gQWNjZXNzb3JzIC8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vLy8vL1xuXG4gIC8vIEluZGV4IHF1ZXJpZXNcblxuICBnZXRTdGF0dXNCdW5kbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMuY2FjaGUuZ2V0T3JTZXQoS2V5cy5zdGF0dXNCdW5kbGUsIGFzeW5jICgpID0+IHtcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IGJ1bmRsZSA9IGF3YWl0IHRoaXMuZ2l0KCkuZ2V0U3RhdHVzQnVuZGxlKCk7XG4gICAgICAgIGNvbnN0IHJlc3VsdHMgPSBhd2FpdCB0aGlzLmZvcm1hdENoYW5nZWRGaWxlcyhidW5kbGUpO1xuICAgICAgICByZXN1bHRzLmJyYW5jaCA9IGJ1bmRsZS5icmFuY2g7XG4gICAgICAgIHJldHVybiByZXN1bHRzO1xuICAgICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICAgIGlmIChlcnIgaW5zdGFuY2VvZiBMYXJnZVJlcG9FcnJvcikge1xuICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKCdUb29MYXJnZScpO1xuICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBicmFuY2g6IHt9LFxuICAgICAgICAgICAgc3RhZ2VkRmlsZXM6IHt9LFxuICAgICAgICAgICAgdW5zdGFnZWRGaWxlczoge30sXG4gICAgICAgICAgICBtZXJnZUNvbmZsaWN0RmlsZXM6IHt9LFxuICAgICAgICAgIH07XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhyb3cgZXJyO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfSk7XG4gIH1cblxuICBhc3luYyBmb3JtYXRDaGFuZ2VkRmlsZXMoe2NoYW5nZWRFbnRyaWVzLCB1bnRyYWNrZWRFbnRyaWVzLCByZW5hbWVkRW50cmllcywgdW5tZXJnZWRFbnRyaWVzfSkge1xuICAgIGNvbnN0IHN0YXR1c01hcCA9IHtcbiAgICAgIEE6ICdhZGRlZCcsXG4gICAgICBNOiAnbW9kaWZpZWQnLFxuICAgICAgRDogJ2RlbGV0ZWQnLFxuICAgICAgVTogJ21vZGlmaWVkJyxcbiAgICAgIFQ6ICd0eXBlY2hhbmdlJyxcbiAgICB9O1xuXG4gICAgY29uc3Qgc3RhZ2VkRmlsZXMgPSB7fTtcbiAgICBjb25zdCB1bnN0YWdlZEZpbGVzID0ge307XG4gICAgY29uc3QgbWVyZ2VDb25mbGljdEZpbGVzID0ge307XG5cbiAgICBjaGFuZ2VkRW50cmllcy5mb3JFYWNoKGVudHJ5ID0+IHtcbiAgICAgIGlmIChlbnRyeS5zdGFnZWRTdGF0dXMpIHtcbiAgICAgICAgc3RhZ2VkRmlsZXNbZW50cnkuZmlsZVBhdGhdID0gc3RhdHVzTWFwW2VudHJ5LnN0YWdlZFN0YXR1c107XG4gICAgICB9XG4gICAgICBpZiAoZW50cnkudW5zdGFnZWRTdGF0dXMpIHtcbiAgICAgICAgdW5zdGFnZWRGaWxlc1tlbnRyeS5maWxlUGF0aF0gPSBzdGF0dXNNYXBbZW50cnkudW5zdGFnZWRTdGF0dXNdO1xuICAgICAgfVxuICAgIH0pO1xuXG4gICAgdW50cmFja2VkRW50cmllcy5mb3JFYWNoKGVudHJ5ID0+IHtcbiAgICAgIHVuc3RhZ2VkRmlsZXNbZW50cnkuZmlsZVBhdGhdID0gc3RhdHVzTWFwLkE7XG4gICAgfSk7XG5cbiAgICByZW5hbWVkRW50cmllcy5mb3JFYWNoKGVudHJ5ID0+IHtcbiAgICAgIGlmIChlbnRyeS5zdGFnZWRTdGF0dXMgPT09ICdSJykge1xuICAgICAgICBzdGFnZWRGaWxlc1tlbnRyeS5maWxlUGF0aF0gPSBzdGF0dXNNYXAuQTtcbiAgICAgICAgc3RhZ2VkRmlsZXNbZW50cnkub3JpZ0ZpbGVQYXRoXSA9IHN0YXR1c01hcC5EO1xuICAgICAgfVxuICAgICAgaWYgKGVudHJ5LnVuc3RhZ2VkU3RhdHVzID09PSAnUicpIHtcbiAgICAgICAgdW5zdGFnZWRGaWxlc1tlbnRyeS5maWxlUGF0aF0gPSBzdGF0dXNNYXAuQTtcbiAgICAgICAgdW5zdGFnZWRGaWxlc1tlbnRyeS5vcmlnRmlsZVBhdGhdID0gc3RhdHVzTWFwLkQ7XG4gICAgICB9XG4gICAgICBpZiAoZW50cnkuc3RhZ2VkU3RhdHVzID09PSAnQycpIHtcbiAgICAgICAgc3RhZ2VkRmlsZXNbZW50cnkuZmlsZVBhdGhdID0gc3RhdHVzTWFwLkE7XG4gICAgICB9XG4gICAgICBpZiAoZW50cnkudW5zdGFnZWRTdGF0dXMgPT09ICdDJykge1xuICAgICAgICB1bnN0YWdlZEZpbGVzW2VudHJ5LmZpbGVQYXRoXSA9IHN0YXR1c01hcC5BO1xuICAgICAgfVxuICAgIH0pO1xuXG4gICAgbGV0IHN0YXR1c1RvSGVhZDtcblxuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgdW5tZXJnZWRFbnRyaWVzLmxlbmd0aDsgaSsrKSB7XG4gICAgICBjb25zdCB7c3RhZ2VkU3RhdHVzLCB1bnN0YWdlZFN0YXR1cywgZmlsZVBhdGh9ID0gdW5tZXJnZWRFbnRyaWVzW2ldO1xuICAgICAgaWYgKHN0YWdlZFN0YXR1cyA9PT0gJ1UnIHx8IHVuc3RhZ2VkU3RhdHVzID09PSAnVScgfHwgKHN0YWdlZFN0YXR1cyA9PT0gJ0EnICYmIHVuc3RhZ2VkU3RhdHVzID09PSAnQScpKSB7XG4gICAgICAgIC8vIFNraXBwaW5nIHRoaXMgY2hlY2sgaGVyZSBiZWNhdXNlIHdlIG9ubHkgcnVuIGEgc2luZ2xlIGBhd2FpdGBcbiAgICAgICAgLy8gYW5kIHdlIG9ubHkgcnVuIGl0IGluIHRoZSBtYWluLCBzeW5jaHJvbm91cyBib2R5IG9mIHRoZSBmb3IgbG9vcC5cbiAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWF3YWl0LWluLWxvb3BcbiAgICAgICAgaWYgKCFzdGF0dXNUb0hlYWQpIHsgc3RhdHVzVG9IZWFkID0gYXdhaXQgdGhpcy5naXQoKS5kaWZmRmlsZVN0YXR1cyh7dGFyZ2V0OiAnSEVBRCd9KTsgfVxuICAgICAgICBtZXJnZUNvbmZsaWN0RmlsZXNbZmlsZVBhdGhdID0ge1xuICAgICAgICAgIG91cnM6IHN0YXR1c01hcFtzdGFnZWRTdGF0dXNdLFxuICAgICAgICAgIHRoZWlyczogc3RhdHVzTWFwW3Vuc3RhZ2VkU3RhdHVzXSxcbiAgICAgICAgICBmaWxlOiBzdGF0dXNUb0hlYWRbZmlsZVBhdGhdIHx8ICdlcXVpdmFsZW50JyxcbiAgICAgICAgfTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4ge3N0YWdlZEZpbGVzLCB1bnN0YWdlZEZpbGVzLCBtZXJnZUNvbmZsaWN0RmlsZXN9O1xuICB9XG5cbiAgYXN5bmMgZ2V0U3RhdHVzZXNGb3JDaGFuZ2VkRmlsZXMoKSB7XG4gICAgY29uc3Qge3N0YWdlZEZpbGVzLCB1bnN0YWdlZEZpbGVzLCBtZXJnZUNvbmZsaWN0RmlsZXN9ID0gYXdhaXQgdGhpcy5nZXRTdGF0dXNCdW5kbGUoKTtcbiAgICByZXR1cm4ge3N0YWdlZEZpbGVzLCB1bnN0YWdlZEZpbGVzLCBtZXJnZUNvbmZsaWN0RmlsZXN9O1xuICB9XG5cbiAgZ2V0RmlsZVBhdGNoRm9yUGF0aChmaWxlUGF0aCwgb3B0aW9ucykge1xuICAgIGNvbnN0IG9wdHMgPSB7XG4gICAgICBzdGFnZWQ6IGZhbHNlLFxuICAgICAgcGF0Y2hCdWZmZXI6IG51bGwsXG4gICAgICBidWlsZGVyOiB7fSxcbiAgICAgIGJlZm9yZTogKCkgPT4ge30sXG4gICAgICBhZnRlcjogKCkgPT4ge30sXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH07XG5cbiAgICByZXR1cm4gdGhpcy5jYWNoZS5nZXRPclNldChLZXlzLmZpbGVQYXRjaC5vbmVXaXRoKGZpbGVQYXRoLCB7c3RhZ2VkOiBvcHRzLnN0YWdlZH0pLCBhc3luYyAoKSA9PiB7XG4gICAgICBjb25zdCBkaWZmcyA9IGF3YWl0IHRoaXMuZ2l0KCkuZ2V0RGlmZnNGb3JGaWxlUGF0aChmaWxlUGF0aCwge3N0YWdlZDogb3B0cy5zdGFnZWR9KTtcbiAgICAgIGNvbnN0IHBheWxvYWQgPSBvcHRzLmJlZm9yZSgpO1xuICAgICAgY29uc3QgcGF0Y2ggPSBidWlsZEZpbGVQYXRjaChkaWZmcywgb3B0cy5idWlsZGVyKTtcbiAgICAgIGlmIChvcHRzLnBhdGNoQnVmZmVyICE9PSBudWxsKSB7IHBhdGNoLmFkb3B0QnVmZmVyKG9wdHMucGF0Y2hCdWZmZXIpOyB9XG4gICAgICBvcHRzLmFmdGVyKHBhdGNoLCBwYXlsb2FkKTtcbiAgICAgIHJldHVybiBwYXRjaDtcbiAgICB9KTtcbiAgfVxuXG4gIGdldERpZmZzRm9yRmlsZVBhdGgoZmlsZVBhdGgsIGJhc2VDb21taXQpIHtcbiAgICByZXR1cm4gdGhpcy5jYWNoZS5nZXRPclNldChLZXlzLmZpbGVQYXRjaC5vbmVXaXRoKGZpbGVQYXRoLCB7YmFzZUNvbW1pdH0pLCAoKSA9PiB7XG4gICAgICByZXR1cm4gdGhpcy5naXQoKS5nZXREaWZmc0ZvckZpbGVQYXRoKGZpbGVQYXRoLCB7YmFzZUNvbW1pdH0pO1xuICAgIH0pO1xuICB9XG5cbiAgZ2V0U3RhZ2VkQ2hhbmdlc1BhdGNoKG9wdGlvbnMpIHtcbiAgICBjb25zdCBvcHRzID0ge1xuICAgICAgYnVpbGRlcjoge30sXG4gICAgICBwYXRjaEJ1ZmZlcjogbnVsbCxcbiAgICAgIGJlZm9yZTogKCkgPT4ge30sXG4gICAgICBhZnRlcjogKCkgPT4ge30sXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH07XG5cbiAgICByZXR1cm4gdGhpcy5jYWNoZS5nZXRPclNldChLZXlzLnN0YWdlZENoYW5nZXMsIGFzeW5jICgpID0+IHtcbiAgICAgIGNvbnN0IGRpZmZzID0gYXdhaXQgdGhpcy5naXQoKS5nZXRTdGFnZWRDaGFuZ2VzUGF0Y2goKTtcbiAgICAgIGNvbnN0IHBheWxvYWQgPSBvcHRzLmJlZm9yZSgpO1xuICAgICAgY29uc3QgcGF0Y2ggPSBidWlsZE11bHRpRmlsZVBhdGNoKGRpZmZzLCBvcHRzLmJ1aWxkZXIpO1xuICAgICAgaWYgKG9wdHMucGF0Y2hCdWZmZXIgIT09IG51bGwpIHsgcGF0Y2guYWRvcHRCdWZmZXIob3B0cy5wYXRjaEJ1ZmZlcik7IH1cbiAgICAgIG9wdHMuYWZ0ZXIocGF0Y2gsIHBheWxvYWQpO1xuICAgICAgcmV0dXJuIHBhdGNoO1xuICAgIH0pO1xuICB9XG5cbiAgcmVhZEZpbGVGcm9tSW5kZXgoZmlsZVBhdGgpIHtcbiAgICByZXR1cm4gdGhpcy5jYWNoZS5nZXRPclNldChLZXlzLmluZGV4Lm9uZVdpdGgoZmlsZVBhdGgpLCAoKSA9PiB7XG4gICAgICByZXR1cm4gdGhpcy5naXQoKS5yZWFkRmlsZUZyb21JbmRleChmaWxlUGF0aCk7XG4gICAgfSk7XG4gIH1cblxuICAvLyBDb21taXQgYWNjZXNzXG5cbiAgZ2V0TGFzdENvbW1pdCgpIHtcbiAgICByZXR1cm4gdGhpcy5jYWNoZS5nZXRPclNldChLZXlzLmxhc3RDb21taXQsIGFzeW5jICgpID0+IHtcbiAgICAgIGNvbnN0IGhlYWRDb21taXQgPSBhd2FpdCB0aGlzLmdpdCgpLmdldEhlYWRDb21taXQoKTtcbiAgICAgIHJldHVybiBoZWFkQ29tbWl0LnVuYm9yblJlZiA/IENvbW1pdC5jcmVhdGVVbmJvcm4oKSA6IG5ldyBDb21taXQoaGVhZENvbW1pdCk7XG4gICAgfSk7XG4gIH1cblxuICBnZXRDb21taXQoc2hhKSB7XG4gICAgcmV0dXJuIHRoaXMuY2FjaGUuZ2V0T3JTZXQoS2V5cy5ibG9iLm9uZVdpdGgoc2hhKSwgYXN5bmMgKCkgPT4ge1xuICAgICAgY29uc3QgW3Jhd0NvbW1pdF0gPSBhd2FpdCB0aGlzLmdpdCgpLmdldENvbW1pdHMoe21heDogMSwgcmVmOiBzaGEsIGluY2x1ZGVQYXRjaDogdHJ1ZX0pO1xuICAgICAgY29uc3QgY29tbWl0ID0gbmV3IENvbW1pdChyYXdDb21taXQpO1xuICAgICAgcmV0dXJuIGNvbW1pdDtcbiAgICB9KTtcbiAgfVxuXG4gIGdldFJlY2VudENvbW1pdHMob3B0aW9ucykge1xuICAgIHJldHVybiB0aGlzLmNhY2hlLmdldE9yU2V0KEtleXMucmVjZW50Q29tbWl0cywgYXN5bmMgKCkgPT4ge1xuICAgICAgY29uc3QgY29tbWl0cyA9IGF3YWl0IHRoaXMuZ2l0KCkuZ2V0Q29tbWl0cyh7cmVmOiAnSEVBRCcsIC4uLm9wdGlvbnN9KTtcbiAgICAgIHJldHVybiBjb21taXRzLm1hcChjb21taXQgPT4gbmV3IENvbW1pdChjb21taXQpKTtcbiAgICB9KTtcbiAgfVxuXG4gIGFzeW5jIGlzQ29tbWl0UHVzaGVkKHNoYSkge1xuICAgIGNvbnN0IHJlbW90ZUJyYW5jaGVzV2l0aENvbW1pdCA9IGF3YWl0IHRoaXMuZ2l0KCkuZ2V0QnJhbmNoZXNXaXRoQ29tbWl0KHNoYSwge3Nob3dMb2NhbDogZmFsc2UsIHNob3dSZW1vdGU6IHRydWV9KTtcbiAgICBjb25zdCBjdXJyZW50UmVtb3RlID0gKGF3YWl0IHRoaXMucmVwb3NpdG9yeS5nZXRDdXJyZW50QnJhbmNoKCkpLmdldFVwc3RyZWFtKCk7XG4gICAgcmV0dXJuIHJlbW90ZUJyYW5jaGVzV2l0aENvbW1pdC5pbmNsdWRlcyhjdXJyZW50UmVtb3RlLmdldEZ1bGxSZWYoKSk7XG4gIH1cblxuICAvLyBBdXRob3IgaW5mb3JtYXRpb25cblxuICBnZXRBdXRob3JzKG9wdGlvbnMpIHtcbiAgICAvLyBGb3Igbm93IHdlJ2xsIGRvIHRoZSBuYWl2ZSB0aGluZyBhbmQgaW52YWxpZGF0ZSBhbnl0aW1lIEhFQUQgbW92ZXMuIFRoaXMgZW5zdXJlcyB0aGF0IHdlIGdldCBuZXcgYXV0aG9yc1xuICAgIC8vIGludHJvZHVjZWQgYnkgbmV3bHkgY3JlYXRlZCBjb21taXRzIG9yIHB1bGxlZCBjb21taXRzLlxuICAgIC8vIFRoaXMgbWVhbnMgdGhhdCB3ZSBhcmUgY29uc3RhbnRseSByZS1mZXRjaGluZyBkYXRhLiBJZiBwZXJmb3JtYW5jZSBiZWNvbWVzIGEgY29uY2VybiB3ZSBjYW4gb3B0aW1pemVcbiAgICByZXR1cm4gdGhpcy5jYWNoZS5nZXRPclNldChLZXlzLmF1dGhvcnMsIGFzeW5jICgpID0+IHtcbiAgICAgIGNvbnN0IGF1dGhvck1hcCA9IGF3YWl0IHRoaXMuZ2l0KCkuZ2V0QXV0aG9ycyhvcHRpb25zKTtcbiAgICAgIHJldHVybiBPYmplY3Qua2V5cyhhdXRob3JNYXApLm1hcChlbWFpbCA9PiBuZXcgQXV0aG9yKGVtYWlsLCBhdXRob3JNYXBbZW1haWxdKSk7XG4gICAgfSk7XG4gIH1cblxuICAvLyBCcmFuY2hlc1xuXG4gIGdldEJyYW5jaGVzKCkge1xuICAgIHJldHVybiB0aGlzLmNhY2hlLmdldE9yU2V0KEtleXMuYnJhbmNoZXMsIGFzeW5jICgpID0+IHtcbiAgICAgIGNvbnN0IHBheWxvYWRzID0gYXdhaXQgdGhpcy5naXQoKS5nZXRCcmFuY2hlcygpO1xuICAgICAgY29uc3QgYnJhbmNoZXMgPSBuZXcgQnJhbmNoU2V0KCk7XG4gICAgICBmb3IgKGNvbnN0IHBheWxvYWQgb2YgcGF5bG9hZHMpIHtcbiAgICAgICAgbGV0IHVwc3RyZWFtID0gbnVsbEJyYW5jaDtcbiAgICAgICAgaWYgKHBheWxvYWQudXBzdHJlYW0pIHtcbiAgICAgICAgICB1cHN0cmVhbSA9IHBheWxvYWQudXBzdHJlYW0ucmVtb3RlTmFtZVxuICAgICAgICAgICAgPyBCcmFuY2guY3JlYXRlUmVtb3RlVHJhY2tpbmcoXG4gICAgICAgICAgICAgIHBheWxvYWQudXBzdHJlYW0udHJhY2tpbmdSZWYsXG4gICAgICAgICAgICAgIHBheWxvYWQudXBzdHJlYW0ucmVtb3RlTmFtZSxcbiAgICAgICAgICAgICAgcGF5bG9hZC51cHN0cmVhbS5yZW1vdGVSZWYsXG4gICAgICAgICAgICApXG4gICAgICAgICAgICA6IG5ldyBCcmFuY2gocGF5bG9hZC51cHN0cmVhbS50cmFja2luZ1JlZik7XG4gICAgICAgIH1cblxuICAgICAgICBsZXQgcHVzaCA9IHVwc3RyZWFtO1xuICAgICAgICBpZiAocGF5bG9hZC5wdXNoKSB7XG4gICAgICAgICAgcHVzaCA9IHBheWxvYWQucHVzaC5yZW1vdGVOYW1lXG4gICAgICAgICAgICA/IEJyYW5jaC5jcmVhdGVSZW1vdGVUcmFja2luZyhcbiAgICAgICAgICAgICAgcGF5bG9hZC5wdXNoLnRyYWNraW5nUmVmLFxuICAgICAgICAgICAgICBwYXlsb2FkLnB1c2gucmVtb3RlTmFtZSxcbiAgICAgICAgICAgICAgcGF5bG9hZC5wdXNoLnJlbW90ZVJlZixcbiAgICAgICAgICAgIClcbiAgICAgICAgICAgIDogbmV3IEJyYW5jaChwYXlsb2FkLnB1c2gudHJhY2tpbmdSZWYpO1xuICAgICAgICB9XG5cbiAgICAgICAgYnJhbmNoZXMuYWRkKG5ldyBCcmFuY2gocGF5bG9hZC5uYW1lLCB1cHN0cmVhbSwgcHVzaCwgcGF5bG9hZC5oZWFkLCB7c2hhOiBwYXlsb2FkLnNoYX0pKTtcbiAgICAgIH1cbiAgICAgIHJldHVybiBicmFuY2hlcztcbiAgICB9KTtcbiAgfVxuXG4gIGdldEhlYWREZXNjcmlwdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy5jYWNoZS5nZXRPclNldChLZXlzLmhlYWREZXNjcmlwdGlvbiwgKCkgPT4ge1xuICAgICAgcmV0dXJuIHRoaXMuZ2l0KCkuZGVzY3JpYmVIZWFkKCk7XG4gICAgfSk7XG4gIH1cblxuICAvLyBNZXJnaW5nIGFuZCByZWJhc2luZyBzdGF0dXNcblxuICBpc01lcmdpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2l0KCkuaXNNZXJnaW5nKHRoaXMucmVwb3NpdG9yeS5nZXRHaXREaXJlY3RvcnlQYXRoKCkpO1xuICB9XG5cbiAgaXNSZWJhc2luZygpIHtcbiAgICByZXR1cm4gdGhpcy5naXQoKS5pc1JlYmFzaW5nKHRoaXMucmVwb3NpdG9yeS5nZXRHaXREaXJlY3RvcnlQYXRoKCkpO1xuICB9XG5cbiAgLy8gUmVtb3Rlc1xuXG4gIGdldFJlbW90ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMuY2FjaGUuZ2V0T3JTZXQoS2V5cy5yZW1vdGVzLCBhc3luYyAoKSA9PiB7XG4gICAgICBjb25zdCByZW1vdGVzSW5mbyA9IGF3YWl0IHRoaXMuZ2l0KCkuZ2V0UmVtb3RlcygpO1xuICAgICAgcmV0dXJuIG5ldyBSZW1vdGVTZXQoXG4gICAgICAgIHJlbW90ZXNJbmZvLm1hcCgoe25hbWUsIHVybH0pID0+IG5ldyBSZW1vdGUobmFtZSwgdXJsKSksXG4gICAgICApO1xuICAgIH0pO1xuICB9XG5cbiAgYWRkUmVtb3RlKG5hbWUsIHVybCkge1xuICAgIHJldHVybiB0aGlzLmludmFsaWRhdGUoXG4gICAgICAoKSA9PiBbXG4gICAgICAgIC4uLktleXMuY29uZmlnLmVhY2hXaXRoU2V0dGluZyhgcmVtb3RlLiR7bmFtZX0udXJsYCksXG4gICAgICAgIC4uLktleXMuY29uZmlnLmVhY2hXaXRoU2V0dGluZyhgcmVtb3RlLiR7bmFtZX0uZmV0Y2hgKSxcbiAgICAgICAgS2V5cy5yZW1vdGVzLFxuICAgICAgXSxcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1zaGFkb3dcbiAgICAgICgpID0+IHRoaXMuZXhlY3V0ZVBpcGVsaW5lQWN0aW9uKCdBRERSRU1PVEUnLCBhc3luYyAobmFtZSwgdXJsKSA9PiB7XG4gICAgICAgIGF3YWl0IHRoaXMuZ2l0KCkuYWRkUmVtb3RlKG5hbWUsIHVybCk7XG4gICAgICAgIHJldHVybiBuZXcgUmVtb3RlKG5hbWUsIHVybCk7XG4gICAgICB9LCBuYW1lLCB1cmwpLFxuICAgICk7XG4gIH1cblxuICBhc3luYyBnZXRBaGVhZENvdW50KGJyYW5jaE5hbWUpIHtcbiAgICBjb25zdCBidW5kbGUgPSBhd2FpdCB0aGlzLmdldFN0YXR1c0J1bmRsZSgpO1xuICAgIHJldHVybiBidW5kbGUuYnJhbmNoLmFoZWFkQmVoaW5kLmFoZWFkO1xuICB9XG5cbiAgYXN5bmMgZ2V0QmVoaW5kQ291bnQoYnJhbmNoTmFtZSkge1xuICAgIGNvbnN0IGJ1bmRsZSA9IGF3YWl0IHRoaXMuZ2V0U3RhdHVzQnVuZGxlKCk7XG4gICAgcmV0dXJuIGJ1bmRsZS5icmFuY2guYWhlYWRCZWhpbmQuYmVoaW5kO1xuICB9XG5cbiAgZ2V0Q29uZmlnKG9wdGlvbiwge2xvY2FsfSA9IHtsb2NhbDogZmFsc2V9KSB7XG4gICAgcmV0dXJuIHRoaXMuY2FjaGUuZ2V0T3JTZXQoS2V5cy5jb25maWcub25lV2l0aChvcHRpb24sIHtsb2NhbH0pLCAoKSA9PiB7XG4gICAgICByZXR1cm4gdGhpcy5naXQoKS5nZXRDb25maWcob3B0aW9uLCB7bG9jYWx9KTtcbiAgICB9KTtcbiAgfVxuXG4gIGRpcmVjdEdldENvbmZpZyhrZXksIG9wdGlvbnMpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRDb25maWcoa2V5LCBvcHRpb25zKTtcbiAgfVxuXG4gIC8vIERpcmVjdCBibG9iIGFjY2Vzc1xuXG4gIGdldEJsb2JDb250ZW50cyhzaGEpIHtcbiAgICByZXR1cm4gdGhpcy5jYWNoZS5nZXRPclNldChLZXlzLmJsb2Iub25lV2l0aChzaGEpLCAoKSA9PiB7XG4gICAgICByZXR1cm4gdGhpcy5naXQoKS5nZXRCbG9iQ29udGVudHMoc2hhKTtcbiAgICB9KTtcbiAgfVxuXG4gIGRpcmVjdEdldEJsb2JDb250ZW50cyhzaGEpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRCbG9iQ29udGVudHMoc2hhKTtcbiAgfVxuXG4gIC8vIERpc2NhcmQgaGlzdG9yeVxuXG4gIGhhc0Rpc2NhcmRIaXN0b3J5KHBhcnRpYWxEaXNjYXJkRmlsZVBhdGggPSBudWxsKSB7XG4gICAgcmV0dXJuIHRoaXMuZGlzY2FyZEhpc3RvcnkuaGFzSGlzdG9yeShwYXJ0aWFsRGlzY2FyZEZpbGVQYXRoKTtcbiAgfVxuXG4gIGdldERpc2NhcmRIaXN0b3J5KHBhcnRpYWxEaXNjYXJkRmlsZVBhdGggPSBudWxsKSB7XG4gICAgcmV0dXJuIHRoaXMuZGlzY2FyZEhpc3RvcnkuZ2V0SGlzdG9yeShwYXJ0aWFsRGlzY2FyZEZpbGVQYXRoKTtcbiAgfVxuXG4gIGdldExhc3RIaXN0b3J5U25hcHNob3RzKHBhcnRpYWxEaXNjYXJkRmlsZVBhdGggPSBudWxsKSB7XG4gICAgcmV0dXJuIHRoaXMuZGlzY2FyZEhpc3RvcnkuZ2V0TGFzdFNuYXBzaG90cyhwYXJ0aWFsRGlzY2FyZEZpbGVQYXRoKTtcbiAgfVxuXG4gIC8vIENhY2hlXG5cbiAgLyogaXN0YW5idWwgaWdub3JlIG5leHQgKi9cbiAgZ2V0Q2FjaGUoKSB7XG4gICAgcmV0dXJuIHRoaXMuY2FjaGU7XG4gIH1cblxuICBpbnZhbGlkYXRlKHNwZWMsIGJvZHkpIHtcbiAgICByZXR1cm4gYm9keSgpLnRoZW4oXG4gICAgICByZXN1bHQgPT4ge1xuICAgICAgICB0aGlzLmFjY2VwdEludmFsaWRhdGlvbihzcGVjKTtcbiAgICAgICAgcmV0dXJuIHJlc3VsdDtcbiAgICAgIH0sXG4gICAgICBlcnIgPT4ge1xuICAgICAgICB0aGlzLmFjY2VwdEludmFsaWRhdGlvbihzcGVjKTtcbiAgICAgICAgcmV0dXJuIFByb21pc2UucmVqZWN0KGVycik7XG4gICAgICB9LFxuICAgICk7XG4gIH1cbn1cblxuU3RhdGUucmVnaXN0ZXIoUHJlc2VudCk7XG5cbmZ1bmN0aW9uIHBhcnRpdGlvbihhcnJheSwgcHJlZGljYXRlKSB7XG4gIGNvbnN0IG1hdGNoZXMgPSBbXTtcbiAgY29uc3Qgbm9ubWF0Y2hlcyA9IFtdO1xuICBhcnJheS5mb3JFYWNoKGl0ZW0gPT4ge1xuICAgIGlmIChwcmVkaWNhdGUoaXRlbSkpIHtcbiAgICAgIG1hdGNoZXMucHVzaChpdGVtKTtcbiAgICB9IGVsc2Uge1xuICAgICAgbm9ubWF0Y2hlcy5wdXNoKGl0ZW0pO1xuICAgIH1cbiAgfSk7XG4gIHJldHVybiBbbWF0Y2hlcywgbm9ubWF0Y2hlc107XG59XG5cbmNsYXNzIENhY2hlIHtcbiAgY29uc3RydWN0b3IoKSB7XG4gICAgdGhpcy5zdG9yYWdlID0gbmV3IE1hcCgpO1xuICAgIHRoaXMuYnlHcm91cCA9IG5ldyBNYXAoKTtcblxuICAgIHRoaXMuZW1pdHRlciA9IG5ldyBFbWl0dGVyKCk7XG4gIH1cblxuICBnZXRPclNldChrZXksIG9wZXJhdGlvbikge1xuICAgIGNvbnN0IHByaW1hcnkgPSBrZXkuZ2V0UHJpbWFyeSgpO1xuICAgIGNvbnN0IGV4aXN0aW5nID0gdGhpcy5zdG9yYWdlLmdldChwcmltYXJ5KTtcbiAgICBpZiAoZXhpc3RpbmcgIT09IHVuZGVmaW5lZCkge1xuICAgICAgZXhpc3RpbmcuaGl0cysrO1xuICAgICAgcmV0dXJuIGV4aXN0aW5nLnByb21pc2U7XG4gICAgfVxuXG4gICAgY29uc3QgY3JlYXRlZCA9IG9wZXJhdGlvbigpO1xuXG4gICAgdGhpcy5zdG9yYWdlLnNldChwcmltYXJ5LCB7XG4gICAgICBjcmVhdGVkQXQ6IHBlcmZvcm1hbmNlLm5vdygpLFxuICAgICAgaGl0czogMCxcbiAgICAgIHByb21pc2U6IGNyZWF0ZWQsXG4gICAgfSk7XG5cbiAgICBjb25zdCBncm91cHMgPSBrZXkuZ2V0R3JvdXBzKCk7XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCBncm91cHMubGVuZ3RoOyBpKyspIHtcbiAgICAgIGNvbnN0IGdyb3VwID0gZ3JvdXBzW2ldO1xuICAgICAgbGV0IGdyb3VwU2V0ID0gdGhpcy5ieUdyb3VwLmdldChncm91cCk7XG4gICAgICBpZiAoZ3JvdXBTZXQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICBncm91cFNldCA9IG5ldyBTZXQoKTtcbiAgICAgICAgdGhpcy5ieUdyb3VwLnNldChncm91cCwgZ3JvdXBTZXQpO1xuICAgICAgfVxuICAgICAgZ3JvdXBTZXQuYWRkKGtleSk7XG4gICAgfVxuXG4gICAgdGhpcy5kaWRVcGRhdGUoKTtcblxuICAgIHJldHVybiBjcmVhdGVkO1xuICB9XG5cbiAgaW52YWxpZGF0ZShrZXlzKSB7XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCBrZXlzLmxlbmd0aDsgaSsrKSB7XG4gICAgICBrZXlzW2ldLnJlbW92ZUZyb21DYWNoZSh0aGlzKTtcbiAgICB9XG5cbiAgICBpZiAoa2V5cy5sZW5ndGggPiAwKSB7XG4gICAgICB0aGlzLmRpZFVwZGF0ZSgpO1xuICAgIH1cbiAgfVxuXG4gIGtleXNJbkdyb3VwKGdyb3VwKSB7XG4gICAgcmV0dXJuIHRoaXMuYnlHcm91cC5nZXQoZ3JvdXApIHx8IFtdO1xuICB9XG5cbiAgcmVtb3ZlUHJpbWFyeShwcmltYXJ5KSB7XG4gICAgdGhpcy5zdG9yYWdlLmRlbGV0ZShwcmltYXJ5KTtcbiAgICB0aGlzLmRpZFVwZGF0ZSgpO1xuICB9XG5cbiAgcmVtb3ZlRnJvbUdyb3VwKGdyb3VwLCBrZXkpIHtcbiAgICBjb25zdCBncm91cFNldCA9IHRoaXMuYnlHcm91cC5nZXQoZ3JvdXApO1xuICAgIGdyb3VwU2V0ICYmIGdyb3VwU2V0LmRlbGV0ZShrZXkpO1xuICAgIHRoaXMuZGlkVXBkYXRlKCk7XG4gIH1cblxuICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICBbU3ltYm9sLml0ZXJhdG9yXSgpIHtcbiAgICByZXR1cm4gdGhpcy5zdG9yYWdlW1N5bWJvbC5pdGVyYXRvcl0oKTtcbiAgfVxuXG4gIGNsZWFyKCkge1xuICAgIHRoaXMuc3RvcmFnZS5jbGVhcigpO1xuICAgIHRoaXMuYnlHcm91cC5jbGVhcigpO1xuICAgIHRoaXMuZGlkVXBkYXRlKCk7XG4gIH1cblxuICBkaWRVcGRhdGUoKSB7XG4gICAgdGhpcy5lbWl0dGVyLmVtaXQoJ2RpZC11cGRhdGUnKTtcbiAgfVxuXG4gIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gIG9uRGlkVXBkYXRlKGNhbGxiYWNrKSB7XG4gICAgcmV0dXJuIHRoaXMuZW1pdHRlci5vbignZGlkLXVwZGF0ZScsIGNhbGxiYWNrKTtcbiAgfVxuXG4gIGRlc3Ryb3koKSB7XG4gICAgdGhpcy5lbWl0dGVyLmRpc3Bvc2UoKTtcbiAgfVxufVxuXG5jbGFzcyBDYWNoZUtleSB7XG4gIGNvbnN0cnVjdG9yKHByaW1hcnksIGdyb3VwcyA9IFtdKSB7XG4gICAgdGhpcy5wcmltYXJ5ID0gcHJpbWFyeTtcbiAgICB0aGlzLmdyb3VwcyA9IGdyb3VwcztcbiAgfVxuXG4gIGdldFByaW1hcnkoKSB7XG4gICAgcmV0dXJuIHRoaXMucHJpbWFyeTtcbiAgfVxuXG4gIGdldEdyb3VwcygpIHtcbiAgICByZXR1cm4gdGhpcy5ncm91cHM7XG4gIH1cblxuICByZW1vdmVGcm9tQ2FjaGUoY2FjaGUsIHdpdGhvdXRHcm91cCA9IG51bGwpIHtcbiAgICBjYWNoZS5yZW1vdmVQcmltYXJ5KHRoaXMuZ2V0UHJpbWFyeSgpKTtcblxuICAgIGNvbnN0IGdyb3VwcyA9IHRoaXMuZ2V0R3JvdXBzKCk7XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCBncm91cHMubGVuZ3RoOyBpKyspIHtcbiAgICAgIGNvbnN0IGdyb3VwID0gZ3JvdXBzW2ldO1xuICAgICAgaWYgKGdyb3VwID09PSB3aXRob3V0R3JvdXApIHtcbiAgICAgICAgY29udGludWU7XG4gICAgICB9XG5cbiAgICAgIGNhY2hlLnJlbW92ZUZyb21Hcm91cChncm91cCwgdGhpcyk7XG4gICAgfVxuICB9XG5cbiAgLyogaXN0YW5idWwgaWdub3JlIG5leHQgKi9cbiAgdG9TdHJpbmcoKSB7XG4gICAgcmV0dXJuIGBDYWNoZUtleSgke3RoaXMucHJpbWFyeX0pYDtcbiAgfVxufVxuXG5jbGFzcyBHcm91cEtleSB7XG4gIGNvbnN0cnVjdG9yKGdyb3VwKSB7XG4gICAgdGhpcy5ncm91cCA9IGdyb3VwO1xuICB9XG5cbiAgcmVtb3ZlRnJvbUNhY2hlKGNhY2hlKSB7XG4gICAgZm9yIChjb25zdCBtYXRjaGluZ0tleSBvZiBjYWNoZS5rZXlzSW5Hcm91cCh0aGlzLmdyb3VwKSkge1xuICAgICAgbWF0Y2hpbmdLZXkucmVtb3ZlRnJvbUNhY2hlKGNhY2hlLCB0aGlzLmdyb3VwKTtcbiAgICB9XG4gIH1cblxuICAvKiBpc3RhbmJ1bCBpZ25vcmUgbmV4dCAqL1xuICB0b1N0cmluZygpIHtcbiAgICByZXR1cm4gYEdyb3VwS2V5KCR7dGhpcy5ncm91cH0pYDtcbiAgfVxufVxuXG5jb25zdCBLZXlzID0ge1xuICBzdGF0dXNCdW5kbGU6IG5ldyBDYWNoZUtleSgnc3RhdHVzLWJ1bmRsZScpLFxuXG4gIHN0YWdlZENoYW5nZXM6IG5ldyBDYWNoZUtleSgnc3RhZ2VkLWNoYW5nZXMnKSxcblxuICBmaWxlUGF0Y2g6IHtcbiAgICBfb3B0S2V5OiAoe3N0YWdlZH0pID0+IChzdGFnZWQgPyAncycgOiAndScpLFxuXG4gICAgb25lV2l0aDogKGZpbGVOYW1lLCBvcHRpb25zKSA9PiB7IC8vIDwtLSBLZXlzLmZpbGVQYXRjaFxuICAgICAgY29uc3Qgb3B0S2V5ID0gS2V5cy5maWxlUGF0Y2guX29wdEtleShvcHRpb25zKTtcbiAgICAgIGNvbnN0IGJhc2VDb21taXQgPSBvcHRpb25zLmJhc2VDb21taXQgfHwgJ2hlYWQnO1xuXG4gICAgICBjb25zdCBleHRyYUdyb3VwcyA9IFtdO1xuICAgICAgaWYgKG9wdGlvbnMuYmFzZUNvbW1pdCkge1xuICAgICAgICBleHRyYUdyb3Vwcy5wdXNoKGBmaWxlLXBhdGNoOmJhc2Utbm9uaGVhZDpwYXRoLSR7ZmlsZU5hbWV9YCk7XG4gICAgICAgIGV4dHJhR3JvdXBzLnB1c2goJ2ZpbGUtcGF0Y2g6YmFzZS1ub25oZWFkJyk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBleHRyYUdyb3Vwcy5wdXNoKCdmaWxlLXBhdGNoOmJhc2UtaGVhZCcpO1xuICAgICAgfVxuXG4gICAgICByZXR1cm4gbmV3IENhY2hlS2V5KGBmaWxlLXBhdGNoOiR7b3B0S2V5fToke2Jhc2VDb21taXR9OiR7ZmlsZU5hbWV9YCwgW1xuICAgICAgICAnZmlsZS1wYXRjaCcsXG4gICAgICAgIGBmaWxlLXBhdGNoOm9wdC0ke29wdEtleX1gLFxuICAgICAgICBgZmlsZS1wYXRjaDpvcHQtJHtvcHRLZXl9OnBhdGgtJHtmaWxlTmFtZX1gLFxuICAgICAgICAuLi5leHRyYUdyb3VwcyxcbiAgICAgIF0pO1xuICAgIH0sXG5cbiAgICBlYWNoV2l0aEZpbGVPcHRzOiAoZmlsZU5hbWVzLCBvcHRzKSA9PiB7XG4gICAgICBjb25zdCBrZXlzID0gW107XG4gICAgICBmb3IgKGxldCBpID0gMDsgaSA8IGZpbGVOYW1lcy5sZW5ndGg7IGkrKykge1xuICAgICAgICBmb3IgKGxldCBqID0gMDsgaiA8IG9wdHMubGVuZ3RoOyBqKyspIHtcbiAgICAgICAgICBrZXlzLnB1c2gobmV3IEdyb3VwS2V5KGBmaWxlLXBhdGNoOm9wdC0ke0tleXMuZmlsZVBhdGNoLl9vcHRLZXkob3B0c1tqXSl9OnBhdGgtJHtmaWxlTmFtZXNbaV19YCkpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgICByZXR1cm4ga2V5cztcbiAgICB9LFxuXG4gICAgZWFjaE5vbkhlYWRXaXRoRmlsZXM6IGZpbGVOYW1lcyA9PiB7XG4gICAgICByZXR1cm4gZmlsZU5hbWVzLm1hcChmaWxlTmFtZSA9PiBuZXcgR3JvdXBLZXkoYGZpbGUtcGF0Y2g6YmFzZS1ub25oZWFkOnBhdGgtJHtmaWxlTmFtZX1gKSk7XG4gICAgfSxcblxuICAgIGFsbEFnYWluc3ROb25IZWFkOiBuZXcgR3JvdXBLZXkoJ2ZpbGUtcGF0Y2g6YmFzZS1ub25oZWFkJyksXG5cbiAgICBlYWNoV2l0aE9wdHM6ICguLi5vcHRzKSA9PiBvcHRzLm1hcChvcHQgPT4gbmV3IEdyb3VwS2V5KGBmaWxlLXBhdGNoOm9wdC0ke0tleXMuZmlsZVBhdGNoLl9vcHRLZXkob3B0KX1gKSksXG5cbiAgICBhbGw6IG5ldyBHcm91cEtleSgnZmlsZS1wYXRjaCcpLFxuICB9LFxuXG4gIGluZGV4OiB7XG4gICAgb25lV2l0aDogZmlsZU5hbWUgPT4gbmV3IENhY2hlS2V5KGBpbmRleDoke2ZpbGVOYW1lfWAsIFsnaW5kZXgnXSksXG5cbiAgICBhbGw6IG5ldyBHcm91cEtleSgnaW5kZXgnKSxcbiAgfSxcblxuICBsYXN0Q29tbWl0OiBuZXcgQ2FjaGVLZXkoJ2xhc3QtY29tbWl0JyksXG5cbiAgcmVjZW50Q29tbWl0czogbmV3IENhY2hlS2V5KCdyZWNlbnQtY29tbWl0cycpLFxuXG4gIGF1dGhvcnM6IG5ldyBDYWNoZUtleSgnYXV0aG9ycycpLFxuXG4gIGJyYW5jaGVzOiBuZXcgQ2FjaGVLZXkoJ2JyYW5jaGVzJyksXG5cbiAgaGVhZERlc2NyaXB0aW9uOiBuZXcgQ2FjaGVLZXkoJ2hlYWQtZGVzY3JpcHRpb24nKSxcblxuICByZW1vdGVzOiBuZXcgQ2FjaGVLZXkoJ3JlbW90ZXMnKSxcblxuICBjb25maWc6IHtcbiAgICBfb3B0S2V5OiBvcHRpb25zID0+IChvcHRpb25zLmxvY2FsID8gJ2wnIDogJycpLFxuXG4gICAgb25lV2l0aDogKHNldHRpbmcsIG9wdGlvbnMpID0+IHtcbiAgICAgIGNvbnN0IG9wdEtleSA9IEtleXMuY29uZmlnLl9vcHRLZXkob3B0aW9ucyk7XG4gICAgICByZXR1cm4gbmV3IENhY2hlS2V5KGBjb25maWc6JHtvcHRLZXl9OiR7c2V0dGluZ31gLCBbJ2NvbmZpZycsIGBjb25maWc6JHtvcHRLZXl9YF0pO1xuICAgIH0sXG5cbiAgICBlYWNoV2l0aFNldHRpbmc6IHNldHRpbmcgPT4gW1xuICAgICAgS2V5cy5jb25maWcub25lV2l0aChzZXR0aW5nLCB7bG9jYWw6IHRydWV9KSxcbiAgICAgIEtleXMuY29uZmlnLm9uZVdpdGgoc2V0dGluZywge2xvY2FsOiBmYWxzZX0pLFxuICAgIF0sXG5cbiAgICBhbGw6IG5ldyBHcm91cEtleSgnY29uZmlnJyksXG4gIH0sXG5cbiAgYmxvYjoge1xuICAgIG9uZVdpdGg6IHNoYSA9PiBuZXcgQ2FjaGVLZXkoYGJsb2I6JHtzaGF9YCwgWydibG9iJ10pLFxuICB9LFxuXG4gIC8vIENvbW1vbiBjb2xsZWN0aW9ucyBvZiBrZXlzIGFuZCBwYXR0ZXJucyBmb3IgdXNlIHdpdGggaW52YWxpZGF0ZSgpLlxuXG4gIHdvcmtkaXJPcGVyYXRpb25LZXlzOiBmaWxlTmFtZXMgPT4gW1xuICAgIEtleXMuc3RhdHVzQnVuZGxlLFxuICAgIC4uLktleXMuZmlsZVBhdGNoLmVhY2hXaXRoRmlsZU9wdHMoZmlsZU5hbWVzLCBbe3N0YWdlZDogZmFsc2V9XSksXG4gIF0sXG5cbiAgY2FjaGVPcGVyYXRpb25LZXlzOiBmaWxlTmFtZXMgPT4gW1xuICAgIC4uLktleXMud29ya2Rpck9wZXJhdGlvbktleXMoZmlsZU5hbWVzKSxcbiAgICAuLi5LZXlzLmZpbGVQYXRjaC5lYWNoV2l0aEZpbGVPcHRzKGZpbGVOYW1lcywgW3tzdGFnZWQ6IHRydWV9XSksXG4gICAgLi4uZmlsZU5hbWVzLm1hcChLZXlzLmluZGV4Lm9uZVdpdGgpLFxuICAgIEtleXMuc3RhZ2VkQ2hhbmdlcyxcbiAgXSxcblxuICBoZWFkT3BlcmF0aW9uS2V5czogKCkgPT4gW1xuICAgIEtleXMuaGVhZERlc2NyaXB0aW9uLFxuICAgIEtleXMuYnJhbmNoZXMsXG4gICAgLi4uS2V5cy5maWxlUGF0Y2guZWFjaFdpdGhPcHRzKHtzdGFnZWQ6IHRydWV9KSxcbiAgICBLZXlzLmZpbGVQYXRjaC5hbGxBZ2FpbnN0Tm9uSGVhZCxcbiAgICBLZXlzLnN0YWdlZENoYW5nZXMsXG4gICAgS2V5cy5sYXN0Q29tbWl0LFxuICAgIEtleXMucmVjZW50Q29tbWl0cyxcbiAgICBLZXlzLmF1dGhvcnMsXG4gICAgS2V5cy5zdGF0dXNCdW5kbGUsXG4gIF0sXG59O1xuIl19