'use strict';

var _fs = _interopRequireDefault(require('fs'));

var _jestDiff = _interopRequireDefault(require('jest-diff'));

var _jestMatcherUtils = require('jest-matcher-utils');

var _snapshot_resolver = require('./snapshot_resolver');

var _State = _interopRequireDefault(require('./State'));

var _plugins = require('./plugins');

var utils = _interopRequireWildcard(require('./utils'));

function _interopRequireWildcard(obj) {
  if (obj && obj.__esModule) {
    return obj;
  } else {
    var newObj = {};
    if (obj != null) {
      for (var key in obj) {
        if (Object.prototype.hasOwnProperty.call(obj, key)) {
          var desc =
            Object.defineProperty && Object.getOwnPropertyDescriptor
              ? Object.getOwnPropertyDescriptor(obj, key)
              : {};
          if (desc.get || desc.set) {
            Object.defineProperty(newObj, key, desc);
          } else {
            newObj[key] = obj[key];
          }
        }
      }
    }
    newObj.default = obj;
    return newObj;
  }
}

function _interopRequireDefault(obj) {
  return obj && obj.__esModule ? obj : {default: obj};
}

var Symbol = global['jest-symbol-do-not-touch'] || global.Symbol;
var Symbol = global['jest-symbol-do-not-touch'] || global.Symbol;

var jestExistsFile =
  global[Symbol.for('jest-native-exists-file')] || _fs.default.existsSync;

const fileExists = (filePath, hasteFS) =>
  hasteFS.exists(filePath) || jestExistsFile(filePath);

const cleanup = (hasteFS, update, snapshotResolver) => {
  const pattern = '\\.' + _snapshot_resolver.EXTENSION + '$';
  const files = hasteFS.matchFiles(pattern);
  const filesRemoved = files.reduce((acc, snapshotFile) => {
    if (!fileExists(snapshotResolver.resolveTestPath(snapshotFile), hasteFS)) {
      if (update === 'all') {
        _fs.default.unlinkSync(snapshotFile);
      }

      return acc + 1;
    }

    return acc;
  }, 0);
  return {
    filesRemoved
  };
};

const toMatchSnapshot = function toMatchSnapshot(
  received,
  propertyMatchers,
  testName
) {
  if (arguments.length === 3 && !propertyMatchers) {
    throw new Error(
      'Property matchers must be an object.\n\nTo provide a snapshot test name without property matchers, use: toMatchSnapshot("name")'
    );
  }

  return _toMatchSnapshot({
    context: this,
    propertyMatchers,
    received,
    testName
  });
};

const toMatchInlineSnapshot = function toMatchInlineSnapshot(
  received,
  propertyMatchersOrInlineSnapshot,
  inlineSnapshot
) {
  let propertyMatchers;

  if (typeof propertyMatchersOrInlineSnapshot === 'string') {
    inlineSnapshot = propertyMatchersOrInlineSnapshot;
  } else {
    propertyMatchers = propertyMatchersOrInlineSnapshot;
  }

  return _toMatchSnapshot({
    context: this,
    inlineSnapshot: inlineSnapshot || '',
    propertyMatchers,
    received
  });
};

const _toMatchSnapshot = ({
  context,
  received,
  propertyMatchers,
  testName,
  inlineSnapshot
}) => {
  context.dontThrow && context.dontThrow();
  testName = typeof propertyMatchers === 'string' ? propertyMatchers : testName;
  const currentTestName = context.currentTestName,
    isNot = context.isNot,
    snapshotState = context.snapshotState;

  if (isNot) {
    const matcherName =
      typeof inlineSnapshot === 'string'
        ? 'toMatchInlineSnapshot'
        : 'toMatchSnapshot';
    throw new Error(
      `Jest: \`.not\` cannot be used with \`.${matcherName}()\`.`
    );
  }

  if (!snapshotState) {
    throw new Error('Jest: snapshot state must be initialized.');
  }

  const fullTestName =
    testName && currentTestName
      ? `${currentTestName}: ${testName}`
      : currentTestName || '';

  if (typeof propertyMatchers === 'object') {
    if (propertyMatchers === null) {
      throw new Error(`Property matchers must be an object.`);
    }

    const propertyPass = context.equals(received, propertyMatchers, [
      context.utils.iterableEquality,
      context.utils.subsetEquality
    ]);

    if (!propertyPass) {
      const key = snapshotState.fail(fullTestName, received);

      const report = () =>
        `${(0, _jestMatcherUtils.RECEIVED_COLOR)(
          'Received value'
        )} does not match ` +
        `${(0, _jestMatcherUtils.EXPECTED_COLOR)(
          `snapshot properties for "${key}"`
        )}.\n\n` +
        `Expected snapshot to match properties:\n` +
        `  ${context.utils.printExpected(propertyMatchers)}` +
        `\nReceived:\n` +
        `  ${context.utils.printReceived(received)}`;

      return {
        message: () =>
          (0, _jestMatcherUtils.matcherHint)(
            '.toMatchSnapshot',
            'value',
            'properties'
          ) +
          '\n\n' +
          report(),
        name: 'toMatchSnapshot',
        pass: false,
        report
      };
    } else {
      received = utils.deepMerge(received, propertyMatchers);
    }
  }

  const result = snapshotState.match({
    error: context.error,
    inlineSnapshot,
    received,
    testName: fullTestName
  });
  const pass = result.pass;
  let actual = result.actual,
    expected = result.expected;
  let report;

  if (pass) {
    return {
      message: () => '',
      pass: true
    };
  } else if (!expected) {
    report = () =>
      `New snapshot was ${(0, _jestMatcherUtils.RECEIVED_COLOR)(
        'not written'
      )}. The update flag ` +
      `must be explicitly passed to write a new snapshot.\n\n` +
      `This is likely because this test is run in a continuous integration ` +
      `(CI) environment in which snapshots are not written by default.\n\n` +
      `${(0, _jestMatcherUtils.RECEIVED_COLOR)('Received value')} ` +
      `${actual}`;
  } else {
    expected = (expected || '').trim();
    actual = (actual || '').trim();
    const diffMessage = (0, _jestDiff.default)(expected, actual, {
      aAnnotation: 'Snapshot',
      bAnnotation: 'Received',
      expand: snapshotState.expand
    });

    report = () =>
      `${(0, _jestMatcherUtils.RECEIVED_COLOR)(
        'Received value'
      )} does not match ` +
      `${(0, _jestMatcherUtils.EXPECTED_COLOR)(
        `stored snapshot "${result.key}"`
      )}.\n\n` +
      (diffMessage ||
        (0, _jestMatcherUtils.EXPECTED_COLOR)('- ' + (expected || '')) +
          '\n' +
          (0, _jestMatcherUtils.RECEIVED_COLOR)('+ ' + actual));
  } // Passing the actual and expected objects so that a custom reporter
  // could access them, for example in order to display a custom visual diff,
  // or create a different error message

  return {
    actual,
    expected,
    message: () =>
      (0, _jestMatcherUtils.matcherHint)('.toMatchSnapshot', 'value', '') +
      '\n\n' +
      report(),
    name: 'toMatchSnapshot',
    pass: false,
    report
  };
};

const toThrowErrorMatchingSnapshot = function toThrowErrorMatchingSnapshot(
  received,
  testName,
  fromPromise
) {
  return _toThrowErrorMatchingSnapshot({
    context: this,
    fromPromise,
    received,
    testName
  });
};

const toThrowErrorMatchingInlineSnapshot = function toThrowErrorMatchingInlineSnapshot(
  received,
  inlineSnapshot,
  fromPromise
) {
  return _toThrowErrorMatchingSnapshot({
    context: this,
    fromPromise,
    inlineSnapshot: inlineSnapshot || '',
    received
  });
};

const _toThrowErrorMatchingSnapshot = ({
  context,
  received,
  testName,
  fromPromise,
  inlineSnapshot
}) => {
  context.dontThrow && context.dontThrow();
  const isNot = context.isNot;
  const matcherName =
    typeof inlineSnapshot === 'string'
      ? 'toThrowErrorMatchingInlineSnapshot'
      : 'toThrowErrorMatchingSnapshot';

  if (isNot) {
    throw new Error(
      `Jest: \`.not\` cannot be used with \`.${matcherName}()\`.`
    );
  }

  let error;

  if (fromPromise) {
    error = received;
  } else {
    try {
      received();
    } catch (e) {
      error = e;
    }
  }

  if (error === undefined) {
    throw new Error(
      (0, _jestMatcherUtils.matcherHint)(`.${matcherName}`, '() => {}', '') +
        '\n\n' +
        `Expected the function to throw an error.\n` +
        `But it didn't throw anything.`
    );
  }

  return _toMatchSnapshot({
    context,
    inlineSnapshot,
    received: error.message,
    testName
  });
};

module.exports = {
  EXTENSION: _snapshot_resolver.EXTENSION,
  SnapshotState: _State.default,
  addSerializer: _plugins.addSerializer,
  buildSnapshotResolver: _snapshot_resolver.buildSnapshotResolver,
  cleanup,
  getSerializers: _plugins.getSerializers,
  isSnapshotPath: _snapshot_resolver.isSnapshotPath,
  toMatchInlineSnapshot,
  toMatchSnapshot,
  toThrowErrorMatchingInlineSnapshot,
  toThrowErrorMatchingSnapshot,
  utils
};
