"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
class WebpackFileSystemHostAdapter {
    constructor(_host) {
        this._host = _host;
        this._syncHost = null;
    }
    _doHostCall(o, callback) {
        const token = Symbol();
        let value = token;
        let error = false;
        try {
            o.subscribe({
                error(err) {
                    error = true;
                    callback(err);
                },
                next(v) {
                    value = v;
                },
                complete() {
                    if (value !== token) {
                        callback(null, value);
                    }
                    else {
                        callback(new Error('Unknown error happened.'));
                    }
                },
            });
        }
        catch (err) {
            // In some occasions, the error handler above will be called, then an exception will be
            // thrown (by design in observable constructors in RxJS 5). Don't call the callback
            // twice.
            if (!error) {
                callback(err);
            }
        }
    }
    stat(path, callback) {
        const p = core_1.normalize('/' + path);
        const result = this._host.stat(p);
        if (result === null) {
            const o = this._host.exists(p).pipe(operators_1.switchMap(exists => {
                if (!exists) {
                    throw new core_1.FileDoesNotExistException(p);
                }
                return this._host.isDirectory(p).pipe(operators_1.mergeMap(isDirectory => {
                    return (isDirectory ? rxjs_1.of(0) : this._host.read(p).pipe(operators_1.map(content => content.byteLength))).pipe(operators_1.map(size => [isDirectory, size]));
                }));
            }), operators_1.map(([isDirectory, size]) => {
                return {
                    isFile() { return !isDirectory; },
                    isDirectory() { return isDirectory; },
                    size,
                    atime: new Date(),
                    mtime: new Date(),
                    ctime: new Date(),
                    birthtime: new Date(),
                };
            }));
            this._doHostCall(o, callback);
        }
        else {
            this._doHostCall(result, callback);
        }
    }
    readdir(path, callback) {
        return this._doHostCall(this._host.list(core_1.normalize('/' + path)), callback);
    }
    readFile(path, callback) {
        const o = this._host.read(core_1.normalize('/' + path)).pipe(operators_1.map(content => Buffer.from(content)));
        return this._doHostCall(o, callback);
    }
    readJson(path, callback) {
        const o = this._host.read(core_1.normalize('/' + path)).pipe(operators_1.map(content => JSON.parse(core_1.virtualFs.fileBufferToString(content))));
        return this._doHostCall(o, callback);
    }
    readlink(path, callback) {
        const err = new Error('Not a symlink.');
        err.code = 'EINVAL';
        callback(err);
    }
    statSync(path) {
        if (!this._syncHost) {
            this._syncHost = new core_1.virtualFs.SyncDelegateHost(this._host);
        }
        const result = this._syncHost.stat(core_1.normalize('/' + path));
        if (result) {
            return result;
        }
        else {
            return {};
        }
    }
    readdirSync(path) {
        if (!this._syncHost) {
            this._syncHost = new core_1.virtualFs.SyncDelegateHost(this._host);
        }
        return this._syncHost.list(core_1.normalize('/' + path));
    }
    readFileSync(path) {
        if (!this._syncHost) {
            this._syncHost = new core_1.virtualFs.SyncDelegateHost(this._host);
        }
        return Buffer.from(this._syncHost.read(core_1.normalize('/' + path)));
    }
    readJsonSync(path) {
        if (!this._syncHost) {
            this._syncHost = new core_1.virtualFs.SyncDelegateHost(this._host);
        }
        const data = this._syncHost.read(core_1.normalize('/' + path));
        return JSON.parse(core_1.virtualFs.fileBufferToString(data));
    }
    readlinkSync(path) {
        const err = new Error('Not a symlink.');
        err.code = 'EINVAL';
        throw err;
    }
    purge(_changes) { }
}
exports.WebpackFileSystemHostAdapter = WebpackFileSystemHostAdapter;
//# sourceMappingURL=data:application/json;base64,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