import { getSetoid as getArraySetoid, traverse as arrayTraverse, empty } from './Array';
import { concat, identity, toString } from './function';
import { fromEquals } from './Setoid';
export const URI = 'Tree';
/**
 * @since 1.6.0
 */
export class Tree {
    constructor(value, forest) {
        this.value = value;
        this.forest = forest;
    }
    map(f) {
        return new Tree(f(this.value), this.forest.map(tree => tree.map(f)));
    }
    ap(fab) {
        return fab.chain(f => this.map(f)); // <- derived
    }
    /**
     * Flipped version of `ap`
     * @since 1.6.0
     */
    ap_(fb) {
        return fb.ap(this);
    }
    chain(f) {
        const { value, forest } = f(this.value);
        return new Tree(value, concat(forest, this.forest.map(t => t.chain(f))));
    }
    extract() {
        return this.value;
    }
    extend(f) {
        return new Tree(f(this), this.forest.map(t => t.extend(f)));
    }
    reduce(b, f) {
        let r = f(b, this.value);
        const len = this.forest.length;
        for (let i = 0; i < len; i++) {
            r = this.forest[i].reduce(r, f);
        }
        return r;
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `new Tree(${toString(this.value)}, ${toString(this.forest)})`;
    }
}
/**
 * @since 1.17.0
 */
export const getShow = (S) => {
    const show = (t) => {
        return `new Tree(${S.show(t.value)}, [${t.forest.map(show).join(', ')}])`;
    };
    return {
        show
    };
};
const map = (fa, f) => {
    return fa.map(f);
};
const of = (a) => {
    return new Tree(a, empty);
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const chain = (fa, f) => {
    return fa.chain(f);
};
const extract = (fa) => {
    return fa.extract();
};
const extend = (fa, f) => {
    return fa.extend(f);
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => (fa, f) => {
    return fa.reduce(M.empty, (acc, a) => M.concat(acc, f(a)));
};
const foldr = (fa, b, f) => {
    let r = b;
    const len = fa.forest.length;
    for (let i = len - 1; i >= 0; i--) {
        r = foldr(fa.forest[i], r, f);
    }
    return f(fa.value, r);
};
function traverse(F) {
    const traverseF = arrayTraverse(F);
    const r = (ta, f) => F.ap(F.map(f(ta.value), (value) => (forest) => new Tree(value, forest)), traverseF(ta.forest, t => r(t, f)));
    return r;
}
function sequence(F) {
    const traverseF = traverse(F);
    return ta => traverseF(ta, identity);
}
/**
 * @since 1.6.0
 */
export const getSetoid = (S) => {
    let SA;
    const R = fromEquals((x, y) => S.equals(x.value, y.value) && SA.equals(x.forest, y.forest));
    SA = getArraySetoid(R);
    return R;
};
/**
 * @since 1.6.0
 */
export const tree = {
    URI,
    map,
    of,
    ap,
    chain,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence,
    extract,
    extend
};
const draw = (indentation, forest) => {
    let r = '';
    const len = forest.length;
    let tree;
    for (let i = 0; i < len; i++) {
        tree = forest[i];
        const isLast = i === len - 1;
        r += indentation + (isLast ? '└' : '├') + '─ ' + tree.value;
        r += draw(indentation + (len > 1 && !isLast ? '│  ' : '   '), tree.forest);
    }
    return r;
};
/**
 * Neat 2-dimensional drawing of a forest
 *
 * @since 1.6.0
 */
export const drawForest = (forest) => {
    return draw('\n', forest);
};
/**
 * Neat 2-dimensional drawing of a tree
 *
 * @example
 * import { Tree, drawTree, tree } from 'fp-ts/lib/Tree'
 *
 * const fa = new Tree('a', [
 *   tree.of('b'),
 *   tree.of('c'),
 *   new Tree('d', [tree.of('e'), tree.of('f')])
 * ])
 *
 * assert.strictEqual(drawTree(fa), `a
 * ├─ b
 * ├─ c
 * └─ d
 *    ├─ e
 *    └─ f`)
 *
 *
 * @since 1.6.0
 */
export const drawTree = (tree) => {
    return tree.value + drawForest(tree.forest);
};
/**
 * Build a tree from a seed value
 *
 * @since 1.6.0
 */
export const unfoldTree = (b, f) => {
    const [a, bs] = f(b);
    return new Tree(a, unfoldForest(bs, f));
};
/**
 * Build a tree from a seed value
 *
 * @since 1.6.0
 */
export const unfoldForest = (bs, f) => {
    return bs.map(b => unfoldTree(b, f));
};
export function unfoldTreeM(M) {
    const unfoldForestMM = unfoldForestM(M);
    return (b, f) => M.chain(f(b), ([a, bs]) => M.chain(unfoldForestMM(bs, f), ts => M.of(new Tree(a, ts))));
}
export function unfoldForestM(M) {
    const traverseM = arrayTraverse(M);
    let unfoldTree;
    return (bs, f) => {
        // tslint:disable-next-line
        if (unfoldTree === undefined) {
            unfoldTree = unfoldTreeM(M);
        }
        return traverseM(bs, b => unfoldTree(b, f));
    };
}
/**
 * @since 1.14.0
 */
export function elem(S) {
    const go = (a, fa) => {
        if (S.equals(a, fa.value)) {
            return true;
        }
        return fa.forest.some(tree => go(a, tree));
    };
    return go;
}
