(function() {
  var CSON, ContextMenuItemSet, ContextMenuManager, Disposable, MenuHelpers, _, calculateSpecificity, fs, path, platformContextMenu, ref, ref1, ref2, remote, sortMenuItems, validateSelector;

  path = require('path');

  CSON = require('season');

  fs = require('fs-plus');

  ref = require('clear-cut'), calculateSpecificity = ref.calculateSpecificity, validateSelector = ref.validateSelector;

  Disposable = require('event-kit').Disposable;

  remote = require('electron').remote;

  MenuHelpers = require('./menu-helpers');

  sortMenuItems = require('./menu-sort-helpers').sortMenuItems;

  _ = require('underscore-plus');

  platformContextMenu = (ref1 = require('../package.json')) != null ? (ref2 = ref1._atomMenu) != null ? ref2['context-menu'] : void 0 : void 0;

  module.exports = ContextMenuManager = (function() {
    function ContextMenuManager(arg) {
      this.keymapManager = arg.keymapManager;
      this.definitions = {
        '.overlayer': []
      };
      this.clear();
      this.keymapManager.onDidLoadBundledKeymaps((function(_this) {
        return function() {
          return _this.loadPlatformItems();
        };
      })(this));
    }

    ContextMenuManager.prototype.initialize = function(arg) {
      this.resourcePath = arg.resourcePath, this.devMode = arg.devMode;
    };

    ContextMenuManager.prototype.loadPlatformItems = function() {
      var map, menusDirPath, platformMenuPath, ref3;
      if (platformContextMenu != null) {
        return this.add(platformContextMenu, (ref3 = this.devMode) != null ? ref3 : false);
      } else {
        menusDirPath = path.join(this.resourcePath, 'menus');
        platformMenuPath = fs.resolve(menusDirPath, process.platform, ['cson', 'json']);
        map = CSON.readFileSync(platformMenuPath);
        return this.add(map['context-menu']);
      }
    };

    ContextMenuManager.prototype.add = function(itemsBySelector, throwOnInvalidSelector) {
      var addedItemSets, itemSet, items, selector;
      if (throwOnInvalidSelector == null) {
        throwOnInvalidSelector = true;
      }
      addedItemSets = [];
      for (selector in itemsBySelector) {
        items = itemsBySelector[selector];
        if (throwOnInvalidSelector) {
          validateSelector(selector);
        }
        itemSet = new ContextMenuItemSet(selector, items);
        addedItemSets.push(itemSet);
        this.itemSets.push(itemSet);
      }
      return new Disposable((function(_this) {
        return function() {
          var i, len;
          for (i = 0, len = addedItemSets.length; i < len; i++) {
            itemSet = addedItemSets[i];
            _this.itemSets.splice(_this.itemSets.indexOf(itemSet), 1);
          }
        };
      })(this));
    };

    ContextMenuManager.prototype.templateForElement = function(target) {
      return this.templateForEvent({
        target: target
      });
    };

    ContextMenuManager.prototype.templateForEvent = function(event) {
      var currentTarget, currentTargetItems, i, item, itemForEvent, itemSet, j, k, len, len1, len2, matchingItemSets, ref3, template;
      template = [];
      currentTarget = event.target;
      while (currentTarget != null) {
        currentTargetItems = [];
        matchingItemSets = this.itemSets.filter(function(itemSet) {
          return currentTarget.webkitMatchesSelector(itemSet.selector);
        });
        for (i = 0, len = matchingItemSets.length; i < len; i++) {
          itemSet = matchingItemSets[i];
          ref3 = itemSet.items;
          for (j = 0, len1 = ref3.length; j < len1; j++) {
            item = ref3[j];
            itemForEvent = this.cloneItemForEvent(item, event);
            if (itemForEvent) {
              MenuHelpers.merge(currentTargetItems, itemForEvent, itemSet.specificity);
            }
          }
        }
        for (k = 0, len2 = currentTargetItems.length; k < len2; k++) {
          item = currentTargetItems[k];
          MenuHelpers.merge(template, item, false);
        }
        currentTarget = currentTarget.parentElement;
      }
      this.pruneRedundantSeparators(template);
      this.addAccelerators(template);
      return this.sortTemplate(template);
    };

    ContextMenuManager.prototype.addAccelerators = function(template) {
      var id, item, keymaps, keystrokes, ref3, results;
      results = [];
      for (id in template) {
        item = template[id];
        if (item.command) {
          keymaps = this.keymapManager.findKeyBindings({
            command: item.command,
            target: document.activeElement
          });
          keystrokes = keymaps != null ? (ref3 = keymaps[0]) != null ? ref3.keystrokes : void 0 : void 0;
          if (keystrokes) {
            if (keystrokes.includes(' ')) {
              item.label += " [" + (_.humanizeKeystroke(keystrokes)) + "]";
            } else {
              item.accelerator = MenuHelpers.acceleratorForKeystroke(keystrokes);
            }
          }
        }
        if (Array.isArray(item.submenu)) {
          results.push(this.addAccelerators(item.submenu));
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    ContextMenuManager.prototype.pruneRedundantSeparators = function(menu) {
      var index, keepNextItemIfSeparator, results;
      keepNextItemIfSeparator = false;
      index = 0;
      results = [];
      while (index < menu.length) {
        if (menu[index].type === 'separator') {
          if (!keepNextItemIfSeparator || index === menu.length - 1) {
            results.push(menu.splice(index, 1));
          } else {
            results.push(index++);
          }
        } else {
          keepNextItemIfSeparator = true;
          results.push(index++);
        }
      }
      return results;
    };

    ContextMenuManager.prototype.sortTemplate = function(template) {
      var id, item;
      template = sortMenuItems(template);
      for (id in template) {
        item = template[id];
        if (Array.isArray(item.submenu)) {
          item.submenu = this.sortTemplate(item.submenu);
        }
      }
      return template;
    };

    ContextMenuManager.prototype.cloneItemForEvent = function(item, event) {
      if (item.devMode && !this.devMode) {
        return null;
      }
      item = Object.create(item);
      if (typeof item.shouldDisplay === 'function') {
        if (!item.shouldDisplay(event)) {
          return null;
        }
      }
      if (typeof item.created === "function") {
        item.created(event);
      }
      if (Array.isArray(item.submenu)) {
        item.submenu = item.submenu.map((function(_this) {
          return function(submenuItem) {
            return _this.cloneItemForEvent(submenuItem, event);
          };
        })(this)).filter(function(submenuItem) {
          return submenuItem !== null;
        });
      }
      return item;
    };

    ContextMenuManager.prototype.showForEvent = function(event) {
      var menuTemplate;
      this.activeElement = event.target;
      menuTemplate = this.templateForEvent(event);
      if (!((menuTemplate != null ? menuTemplate.length : void 0) > 0)) {
        return;
      }
      remote.getCurrentWindow().emit('context-menu', menuTemplate);
    };

    ContextMenuManager.prototype.clear = function() {
      var inspectElement;
      this.activeElement = null;
      this.itemSets = [];
      inspectElement = {
        'atom-workspace': [
          {
            label: 'Inspect Element',
            command: 'application:inspect',
            devMode: true,
            created: function(event) {
              var pageX, pageY;
              pageX = event.pageX, pageY = event.pageY;
              return this.commandDetail = {
                x: pageX,
                y: pageY
              };
            }
          }
        ]
      };
      return this.add(inspectElement, false);
    };

    return ContextMenuManager;

  })();

  ContextMenuItemSet = (function() {
    function ContextMenuItemSet(selector1, items1) {
      this.selector = selector1;
      this.items = items1;
      this.specificity = calculateSpecificity(this.selector);
    }

    return ContextMenuItemSet;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiL2J1aWxkL2F0b20vc3JjL2F0b20tMS40MS4wL291dC9hcHAvc3JjL2NvbnRleHQtbWVudS1tYW5hZ2VyLmNvZmZlZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTtBQUFBLE1BQUE7O0VBQUEsSUFBQSxHQUFPLE9BQUEsQ0FBUSxNQUFSOztFQUNQLElBQUEsR0FBTyxPQUFBLENBQVEsUUFBUjs7RUFDUCxFQUFBLEdBQUssT0FBQSxDQUFRLFNBQVI7O0VBQ0wsTUFBMkMsT0FBQSxDQUFRLFdBQVIsQ0FBM0MsRUFBQywrQ0FBRCxFQUF1Qjs7RUFDdEIsYUFBYyxPQUFBLENBQVEsV0FBUjs7RUFDZCxTQUFVLE9BQUEsQ0FBUSxVQUFSOztFQUNYLFdBQUEsR0FBYyxPQUFBLENBQVEsZ0JBQVI7O0VBQ2IsZ0JBQWlCLE9BQUEsQ0FBUSxxQkFBUjs7RUFDbEIsQ0FBQSxHQUFJLE9BQUEsQ0FBUSxpQkFBUjs7RUFFSixtQkFBQSx1RkFBNkQsQ0FBQSxjQUFBOztFQWdDN0QsTUFBTSxDQUFDLE9BQVAsR0FDTTtJQUNTLDRCQUFDLEdBQUQ7TUFBRSxJQUFDLENBQUEsZ0JBQUYsSUFBRTtNQUNkLElBQUMsQ0FBQSxXQUFELEdBQWU7UUFBQyxZQUFBLEVBQWMsRUFBZjs7TUFDZixJQUFDLENBQUEsS0FBRCxDQUFBO01BRUEsSUFBQyxDQUFBLGFBQWEsQ0FBQyx1QkFBZixDQUF1QyxDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUE7aUJBQUcsS0FBQyxDQUFBLGlCQUFELENBQUE7UUFBSDtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBdkM7SUFKVzs7aUNBTWIsVUFBQSxHQUFZLFNBQUMsR0FBRDtNQUFFLElBQUMsQ0FBQSxtQkFBQSxjQUFjLElBQUMsQ0FBQSxjQUFBO0lBQWxCOztpQ0FFWixpQkFBQSxHQUFtQixTQUFBO0FBQ2pCLFVBQUE7TUFBQSxJQUFHLDJCQUFIO2VBQ0UsSUFBQyxDQUFBLEdBQUQsQ0FBSyxtQkFBTCx5Q0FBcUMsS0FBckMsRUFERjtPQUFBLE1BQUE7UUFHRSxZQUFBLEdBQWUsSUFBSSxDQUFDLElBQUwsQ0FBVSxJQUFDLENBQUEsWUFBWCxFQUF5QixPQUF6QjtRQUNmLGdCQUFBLEdBQW1CLEVBQUUsQ0FBQyxPQUFILENBQVcsWUFBWCxFQUF5QixPQUFPLENBQUMsUUFBakMsRUFBMkMsQ0FBQyxNQUFELEVBQVMsTUFBVCxDQUEzQztRQUNuQixHQUFBLEdBQU0sSUFBSSxDQUFDLFlBQUwsQ0FBa0IsZ0JBQWxCO2VBQ04sSUFBQyxDQUFBLEdBQUQsQ0FBSyxHQUFJLENBQUEsY0FBQSxDQUFULEVBTkY7O0lBRGlCOztpQ0E2RG5CLEdBQUEsR0FBSyxTQUFDLGVBQUQsRUFBa0Isc0JBQWxCO0FBQ0gsVUFBQTs7UUFEcUIseUJBQXlCOztNQUM5QyxhQUFBLEdBQWdCO0FBRWhCLFdBQUEsMkJBQUE7O1FBQ0UsSUFBOEIsc0JBQTlCO1VBQUEsZ0JBQUEsQ0FBaUIsUUFBakIsRUFBQTs7UUFDQSxPQUFBLEdBQVUsSUFBSSxrQkFBSixDQUF1QixRQUF2QixFQUFpQyxLQUFqQztRQUNWLGFBQWEsQ0FBQyxJQUFkLENBQW1CLE9BQW5CO1FBQ0EsSUFBQyxDQUFBLFFBQVEsQ0FBQyxJQUFWLENBQWUsT0FBZjtBQUpGO2FBTUEsSUFBSSxVQUFKLENBQWUsQ0FBQSxTQUFBLEtBQUE7ZUFBQSxTQUFBO0FBQ2IsY0FBQTtBQUFBLGVBQUEsK0NBQUE7O1lBQ0UsS0FBQyxDQUFBLFFBQVEsQ0FBQyxNQUFWLENBQWlCLEtBQUMsQ0FBQSxRQUFRLENBQUMsT0FBVixDQUFrQixPQUFsQixDQUFqQixFQUE2QyxDQUE3QztBQURGO1FBRGE7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQWY7SUFURzs7aUNBY0wsa0JBQUEsR0FBb0IsU0FBQyxNQUFEO2FBQ2xCLElBQUMsQ0FBQSxnQkFBRCxDQUFrQjtRQUFDLFFBQUEsTUFBRDtPQUFsQjtJQURrQjs7aUNBR3BCLGdCQUFBLEdBQWtCLFNBQUMsS0FBRDtBQUNoQixVQUFBO01BQUEsUUFBQSxHQUFXO01BQ1gsYUFBQSxHQUFnQixLQUFLLENBQUM7QUFFdEIsYUFBTSxxQkFBTjtRQUNFLGtCQUFBLEdBQXFCO1FBQ3JCLGdCQUFBLEdBQ0UsSUFBQyxDQUFBLFFBQVEsQ0FBQyxNQUFWLENBQWlCLFNBQUMsT0FBRDtpQkFBYSxhQUFhLENBQUMscUJBQWQsQ0FBb0MsT0FBTyxDQUFDLFFBQTVDO1FBQWIsQ0FBakI7QUFFRixhQUFBLGtEQUFBOztBQUNFO0FBQUEsZUFBQSx3Q0FBQTs7WUFDRSxZQUFBLEdBQWUsSUFBQyxDQUFBLGlCQUFELENBQW1CLElBQW5CLEVBQXlCLEtBQXpCO1lBQ2YsSUFBRyxZQUFIO2NBQ0UsV0FBVyxDQUFDLEtBQVosQ0FBa0Isa0JBQWxCLEVBQXNDLFlBQXRDLEVBQW9ELE9BQU8sQ0FBQyxXQUE1RCxFQURGOztBQUZGO0FBREY7QUFNQSxhQUFBLHNEQUFBOztVQUNFLFdBQVcsQ0FBQyxLQUFaLENBQWtCLFFBQWxCLEVBQTRCLElBQTVCLEVBQWtDLEtBQWxDO0FBREY7UUFHQSxhQUFBLEdBQWdCLGFBQWEsQ0FBQztNQWRoQztNQWdCQSxJQUFDLENBQUEsd0JBQUQsQ0FBMEIsUUFBMUI7TUFDQSxJQUFDLENBQUEsZUFBRCxDQUFpQixRQUFqQjtBQUVBLGFBQU8sSUFBQyxDQUFBLFlBQUQsQ0FBYyxRQUFkO0lBdkJTOztpQ0EyQmxCLGVBQUEsR0FBaUIsU0FBQyxRQUFEO0FBQ2YsVUFBQTtBQUFBO1dBQUEsY0FBQTs7UUFDRSxJQUFHLElBQUksQ0FBQyxPQUFSO1VBQ0UsT0FBQSxHQUFVLElBQUMsQ0FBQSxhQUFhLENBQUMsZUFBZixDQUErQjtZQUFDLE9BQUEsRUFBUyxJQUFJLENBQUMsT0FBZjtZQUF3QixNQUFBLEVBQVEsUUFBUSxDQUFDLGFBQXpDO1dBQS9CO1VBQ1YsVUFBQSx1REFBd0IsQ0FBRTtVQUMxQixJQUFHLFVBQUg7WUFJRSxJQUFHLFVBQVUsQ0FBQyxRQUFYLENBQW9CLEdBQXBCLENBQUg7Y0FDRSxJQUFJLENBQUMsS0FBTCxJQUFjLElBQUEsR0FBSSxDQUFDLENBQUMsQ0FBQyxpQkFBRixDQUFvQixVQUFwQixDQUFELENBQUosR0FBcUMsSUFEckQ7YUFBQSxNQUFBO2NBR0UsSUFBSSxDQUFDLFdBQUwsR0FBbUIsV0FBVyxDQUFDLHVCQUFaLENBQW9DLFVBQXBDLEVBSHJCO2FBSkY7V0FIRjs7UUFXQSxJQUFHLEtBQUssQ0FBQyxPQUFOLENBQWMsSUFBSSxDQUFDLE9BQW5CLENBQUg7dUJBQ0UsSUFBQyxDQUFBLGVBQUQsQ0FBaUIsSUFBSSxDQUFDLE9BQXRCLEdBREY7U0FBQSxNQUFBOytCQUFBOztBQVpGOztJQURlOztpQ0FnQmpCLHdCQUFBLEdBQTBCLFNBQUMsSUFBRDtBQUN4QixVQUFBO01BQUEsdUJBQUEsR0FBMEI7TUFDMUIsS0FBQSxHQUFRO0FBQ1I7YUFBTSxLQUFBLEdBQVEsSUFBSSxDQUFDLE1BQW5CO1FBQ0UsSUFBRyxJQUFLLENBQUEsS0FBQSxDQUFNLENBQUMsSUFBWixLQUFvQixXQUF2QjtVQUNFLElBQUcsQ0FBSSx1QkFBSixJQUErQixLQUFBLEtBQVMsSUFBSSxDQUFDLE1BQUwsR0FBYyxDQUF6RDt5QkFDRSxJQUFJLENBQUMsTUFBTCxDQUFZLEtBQVosRUFBbUIsQ0FBbkIsR0FERjtXQUFBLE1BQUE7eUJBR0UsS0FBQSxJQUhGO1dBREY7U0FBQSxNQUFBO1VBTUUsdUJBQUEsR0FBMEI7dUJBQzFCLEtBQUEsSUFQRjs7TUFERixDQUFBOztJQUh3Qjs7aUNBYTFCLFlBQUEsR0FBYyxTQUFDLFFBQUQ7QUFDWixVQUFBO01BQUEsUUFBQSxHQUFXLGFBQUEsQ0FBYyxRQUFkO0FBQ1gsV0FBQSxjQUFBOztRQUNFLElBQUcsS0FBSyxDQUFDLE9BQU4sQ0FBYyxJQUFJLENBQUMsT0FBbkIsQ0FBSDtVQUNFLElBQUksQ0FBQyxPQUFMLEdBQWUsSUFBQyxDQUFBLFlBQUQsQ0FBYyxJQUFJLENBQUMsT0FBbkIsRUFEakI7O0FBREY7QUFHQSxhQUFPO0lBTEs7O2lDQVFkLGlCQUFBLEdBQW1CLFNBQUMsSUFBRCxFQUFPLEtBQVA7TUFDakIsSUFBZSxJQUFJLENBQUMsT0FBTCxJQUFpQixDQUFJLElBQUMsQ0FBQSxPQUFyQztBQUFBLGVBQU8sS0FBUDs7TUFDQSxJQUFBLEdBQU8sTUFBTSxDQUFDLE1BQVAsQ0FBYyxJQUFkO01BQ1AsSUFBRyxPQUFPLElBQUksQ0FBQyxhQUFaLEtBQTZCLFVBQWhDO1FBQ0UsSUFBQSxDQUFtQixJQUFJLENBQUMsYUFBTCxDQUFtQixLQUFuQixDQUFuQjtBQUFBLGlCQUFPLEtBQVA7U0FERjs7O1FBRUEsSUFBSSxDQUFDLFFBQVM7O01BQ2QsSUFBRyxLQUFLLENBQUMsT0FBTixDQUFjLElBQUksQ0FBQyxPQUFuQixDQUFIO1FBQ0UsSUFBSSxDQUFDLE9BQUwsR0FBZSxJQUFJLENBQUMsT0FDbEIsQ0FBQyxHQURZLENBQ1IsQ0FBQSxTQUFBLEtBQUE7aUJBQUEsU0FBQyxXQUFEO21CQUFpQixLQUFDLENBQUEsaUJBQUQsQ0FBbUIsV0FBbkIsRUFBZ0MsS0FBaEM7VUFBakI7UUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBRFEsQ0FFYixDQUFDLE1BRlksQ0FFTCxTQUFDLFdBQUQ7aUJBQWlCLFdBQUEsS0FBaUI7UUFBbEMsQ0FGSyxFQURqQjs7QUFJQSxhQUFPO0lBVlU7O2lDQVluQixZQUFBLEdBQWMsU0FBQyxLQUFEO0FBQ1osVUFBQTtNQUFBLElBQUMsQ0FBQSxhQUFELEdBQWlCLEtBQUssQ0FBQztNQUN2QixZQUFBLEdBQWUsSUFBQyxDQUFBLGdCQUFELENBQWtCLEtBQWxCO01BRWYsSUFBQSxDQUFBLHlCQUFjLFlBQVksQ0FBRSxnQkFBZCxHQUF1QixDQUFyQyxDQUFBO0FBQUEsZUFBQTs7TUFDQSxNQUFNLENBQUMsZ0JBQVAsQ0FBQSxDQUF5QixDQUFDLElBQTFCLENBQStCLGNBQS9CLEVBQStDLFlBQS9DO0lBTFk7O2lDQVFkLEtBQUEsR0FBTyxTQUFBO0FBQ0wsVUFBQTtNQUFBLElBQUMsQ0FBQSxhQUFELEdBQWlCO01BQ2pCLElBQUMsQ0FBQSxRQUFELEdBQVk7TUFDWixjQUFBLEdBQWlCO1FBQ2YsZ0JBQUEsRUFBa0I7VUFBQztZQUNqQixLQUFBLEVBQU8saUJBRFU7WUFFakIsT0FBQSxFQUFTLHFCQUZRO1lBR2pCLE9BQUEsRUFBUyxJQUhRO1lBSWpCLE9BQUEsRUFBUyxTQUFDLEtBQUQ7QUFDUCxrQkFBQTtjQUFDLG1CQUFELEVBQVE7cUJBQ1IsSUFBQyxDQUFBLGFBQUQsR0FBaUI7Z0JBQUMsQ0FBQSxFQUFHLEtBQUo7Z0JBQVcsQ0FBQSxFQUFHLEtBQWQ7O1lBRlYsQ0FKUTtXQUFEO1NBREg7O2FBVWpCLElBQUMsQ0FBQSxHQUFELENBQUssY0FBTCxFQUFxQixLQUFyQjtJQWJLOzs7Ozs7RUFlSDtJQUNTLDRCQUFDLFNBQUQsRUFBWSxNQUFaO01BQUMsSUFBQyxDQUFBLFdBQUQ7TUFBVyxJQUFDLENBQUEsUUFBRDtNQUN2QixJQUFDLENBQUEsV0FBRCxHQUFlLG9CQUFBLENBQXFCLElBQUMsQ0FBQSxRQUF0QjtJQURKOzs7OztBQXRPZiIsInNvdXJjZXNDb250ZW50IjpbInBhdGggPSByZXF1aXJlICdwYXRoJ1xuQ1NPTiA9IHJlcXVpcmUgJ3NlYXNvbidcbmZzID0gcmVxdWlyZSAnZnMtcGx1cydcbntjYWxjdWxhdGVTcGVjaWZpY2l0eSwgdmFsaWRhdGVTZWxlY3Rvcn0gPSByZXF1aXJlICdjbGVhci1jdXQnXG57RGlzcG9zYWJsZX0gPSByZXF1aXJlICdldmVudC1raXQnXG57cmVtb3RlfSA9IHJlcXVpcmUgJ2VsZWN0cm9uJ1xuTWVudUhlbHBlcnMgPSByZXF1aXJlICcuL21lbnUtaGVscGVycydcbntzb3J0TWVudUl0ZW1zfSA9IHJlcXVpcmUgJy4vbWVudS1zb3J0LWhlbHBlcnMnXG5fID0gcmVxdWlyZSAndW5kZXJzY29yZS1wbHVzJ1xuXG5wbGF0Zm9ybUNvbnRleHRNZW51ID0gcmVxdWlyZSgnLi4vcGFja2FnZS5qc29uJyk/Ll9hdG9tTWVudT9bJ2NvbnRleHQtbWVudSddXG5cbiMgRXh0ZW5kZWQ6IFByb3ZpZGVzIGEgcmVnaXN0cnkgZm9yIGNvbW1hbmRzIHRoYXQgeW91J2QgbGlrZSB0byBhcHBlYXIgaW4gdGhlXG4jIGNvbnRleHQgbWVudS5cbiNcbiMgQW4gaW5zdGFuY2Ugb2YgdGhpcyBjbGFzcyBpcyBhbHdheXMgYXZhaWxhYmxlIGFzIHRoZSBgYXRvbS5jb250ZXh0TWVudWBcbiMgZ2xvYmFsLlxuI1xuIyAjIyBDb250ZXh0IE1lbnUgQ1NPTiBGb3JtYXRcbiNcbiMgYGBgY29mZmVlXG4jICdhdG9tLXdvcmtzcGFjZSc6IFt7bGFiZWw6ICdIZWxwJywgY29tbWFuZDogJ2FwcGxpY2F0aW9uOm9wZW4tZG9jdW1lbnRhdGlvbid9XVxuIyAnYXRvbS10ZXh0LWVkaXRvcic6IFt7XG4jICAgbGFiZWw6ICdIaXN0b3J5JyxcbiMgICBzdWJtZW51OiBbXG4jICAgICB7bGFiZWw6ICdVbmRvJywgY29tbWFuZDonY29yZTp1bmRvJ31cbiMgICAgIHtsYWJlbDogJ1JlZG8nLCBjb21tYW5kOidjb3JlOnJlZG8nfVxuIyAgIF1cbiMgfV1cbiMgYGBgXG4jXG4jIEluIHlvdXIgcGFja2FnZSdzIG1lbnUgYC5jc29uYCBmaWxlIHlvdSBuZWVkIHRvIHNwZWNpZnkgaXQgdW5kZXIgYVxuIyBgY29udGV4dC1tZW51YCBrZXk6XG4jXG4jIGBgYGNvZmZlZVxuIyAnY29udGV4dC1tZW51JzpcbiMgICAnYXRvbS13b3Jrc3BhY2UnOiBbe2xhYmVsOiAnSGVscCcsIGNvbW1hbmQ6ICdhcHBsaWNhdGlvbjpvcGVuLWRvY3VtZW50YXRpb24nfV1cbiMgICAuLi5cbiMgYGBgXG4jXG4jIFRoZSBmb3JtYXQgZm9yIHVzZSBpbiB7OjphZGR9IGlzIHRoZSBzYW1lIG1pbnVzIHRoZSBgY29udGV4dC1tZW51YCBrZXkuIFNlZVxuIyB7OjphZGR9IGZvciBtb3JlIGluZm9ybWF0aW9uLlxubW9kdWxlLmV4cG9ydHMgPVxuY2xhc3MgQ29udGV4dE1lbnVNYW5hZ2VyXG4gIGNvbnN0cnVjdG9yOiAoe0BrZXltYXBNYW5hZ2VyfSkgLT5cbiAgICBAZGVmaW5pdGlvbnMgPSB7Jy5vdmVybGF5ZXInOiBbXX0gIyBUT0RPOiBSZW1vdmUgb25jZSBjb2xvciBwaWNrZXIgcGFja2FnZSBzdG9wcyB0b3VjaGluZyBwcml2YXRlIGRhdGFcbiAgICBAY2xlYXIoKVxuXG4gICAgQGtleW1hcE1hbmFnZXIub25EaWRMb2FkQnVuZGxlZEtleW1hcHMgPT4gQGxvYWRQbGF0Zm9ybUl0ZW1zKClcblxuICBpbml0aWFsaXplOiAoe0ByZXNvdXJjZVBhdGgsIEBkZXZNb2RlfSkgLT5cblxuICBsb2FkUGxhdGZvcm1JdGVtczogLT5cbiAgICBpZiBwbGF0Zm9ybUNvbnRleHRNZW51P1xuICAgICAgQGFkZChwbGF0Zm9ybUNvbnRleHRNZW51LCBAZGV2TW9kZSA/IGZhbHNlKVxuICAgIGVsc2VcbiAgICAgIG1lbnVzRGlyUGF0aCA9IHBhdGguam9pbihAcmVzb3VyY2VQYXRoLCAnbWVudXMnKVxuICAgICAgcGxhdGZvcm1NZW51UGF0aCA9IGZzLnJlc29sdmUobWVudXNEaXJQYXRoLCBwcm9jZXNzLnBsYXRmb3JtLCBbJ2Nzb24nLCAnanNvbiddKVxuICAgICAgbWFwID0gQ1NPTi5yZWFkRmlsZVN5bmMocGxhdGZvcm1NZW51UGF0aClcbiAgICAgIEBhZGQobWFwWydjb250ZXh0LW1lbnUnXSlcblxuICAjIFB1YmxpYzogQWRkIGNvbnRleHQgbWVudSBpdGVtcyBzY29wZWQgYnkgQ1NTIHNlbGVjdG9ycy5cbiAgI1xuICAjICMjIEV4YW1wbGVzXG4gICNcbiAgIyBUbyBhZGQgYSBjb250ZXh0IG1lbnUsIHBhc3MgYSBzZWxlY3RvciBtYXRjaGluZyB0aGUgZWxlbWVudHMgdG8gd2hpY2ggeW91XG4gICMgd2FudCB0aGUgbWVudSB0byBhcHBseSBhcyB0aGUgdG9wIGxldmVsIGtleSwgZm9sbG93ZWQgYnkgYSBtZW51IGRlc2NyaXB0b3IuXG4gICMgVGhlIGludm9jYXRpb24gYmVsb3cgYWRkcyBhIGdsb2JhbCAnSGVscCcgY29udGV4dCBtZW51IGl0ZW0gYW5kIGEgJ0hpc3RvcnknXG4gICMgc3VibWVudSBvbiB0aGUgZWRpdG9yIHN1cHBvcnRpbmcgdW5kby9yZWRvLiBUaGlzIGlzIGp1c3QgZm9yIGV4YW1wbGVcbiAgIyBwdXJwb3NlcyBhbmQgbm90IHRoZSB3YXkgdGhlIG1lbnUgaXMgYWN0dWFsbHkgY29uZmlndXJlZCBpbiBBdG9tIGJ5IGRlZmF1bHQuXG4gICNcbiAgIyBgYGBjb2ZmZWVcbiAgIyBhdG9tLmNvbnRleHRNZW51LmFkZCB7XG4gICMgICAnYXRvbS13b3Jrc3BhY2UnOiBbe2xhYmVsOiAnSGVscCcsIGNvbW1hbmQ6ICdhcHBsaWNhdGlvbjpvcGVuLWRvY3VtZW50YXRpb24nfV1cbiAgIyAgICdhdG9tLXRleHQtZWRpdG9yJzogW3tcbiAgIyAgICAgbGFiZWw6ICdIaXN0b3J5JyxcbiAgIyAgICAgc3VibWVudTogW1xuICAjICAgICAgIHtsYWJlbDogJ1VuZG8nLCBjb21tYW5kOidjb3JlOnVuZG8nfVxuICAjICAgICAgIHtsYWJlbDogJ1JlZG8nLCBjb21tYW5kOidjb3JlOnJlZG8nfVxuICAjICAgICBdXG4gICMgICB9XVxuICAjIH1cbiAgIyBgYGBcbiAgI1xuICAjICMjIEFyZ3VtZW50c1xuICAjXG4gICMgKiBgaXRlbXNCeVNlbGVjdG9yYCBBbiB7T2JqZWN0fSB3aG9zZSBrZXlzIGFyZSBDU1Mgc2VsZWN0b3JzIGFuZCB3aG9zZVxuICAjICAgdmFsdWVzIGFyZSB7QXJyYXl9cyBvZiBpdGVtIHtPYmplY3R9cyBjb250YWluaW5nIHRoZSBmb2xsb3dpbmcga2V5czpcbiAgIyAgICogYGxhYmVsYCAob3B0aW9uYWwpIEEge1N0cmluZ30gY29udGFpbmluZyB0aGUgbWVudSBpdGVtJ3MgbGFiZWwuXG4gICMgICAqIGBjb21tYW5kYCAob3B0aW9uYWwpIEEge1N0cmluZ30gY29udGFpbmluZyB0aGUgY29tbWFuZCB0byBpbnZva2Ugb24gdGhlXG4gICMgICAgIHRhcmdldCBvZiB0aGUgcmlnaHQgY2xpY2sgdGhhdCBpbnZva2VkIHRoZSBjb250ZXh0IG1lbnUuXG4gICMgICAqIGBlbmFibGVkYCAob3B0aW9uYWwpIEEge0Jvb2xlYW59IGluZGljYXRpbmcgd2hldGhlciB0aGUgbWVudSBpdGVtXG4gICMgICAgIHNob3VsZCBiZSBjbGlja2FibGUuIERpc2FibGVkIG1lbnUgaXRlbXMgdHlwaWNhbGx5IGFwcGVhciBncmF5ZWQgb3V0LlxuICAjICAgICBEZWZhdWx0cyB0byBgdHJ1ZWAuXG4gICMgICAqIGBzdWJtZW51YCAob3B0aW9uYWwpIEFuIHtBcnJheX0gb2YgYWRkaXRpb25hbCBpdGVtcy5cbiAgIyAgICogYHR5cGVgIChvcHRpb25hbCkgSWYgeW91IHdhbnQgdG8gY3JlYXRlIGEgc2VwYXJhdG9yLCBwcm92aWRlIGFuIGl0ZW1cbiAgIyAgICAgIHdpdGggYHR5cGU6ICdzZXBhcmF0b3InYCBhbmQgbm8gb3RoZXIga2V5cy5cbiAgIyAgICogYHZpc2libGVgIChvcHRpb25hbCkgQSB7Qm9vbGVhbn0gaW5kaWNhdGluZyB3aGV0aGVyIHRoZSBtZW51IGl0ZW1cbiAgIyAgICAgc2hvdWxkIGFwcGVhciBpbiB0aGUgbWVudS4gRGVmYXVsdHMgdG8gYHRydWVgLlxuICAjICAgKiBgY3JlYXRlZGAgKG9wdGlvbmFsKSBBIHtGdW5jdGlvbn0gdGhhdCBpcyBjYWxsZWQgb24gdGhlIGl0ZW0gZWFjaCB0aW1lIGFcbiAgIyAgICAgY29udGV4dCBtZW51IGlzIGNyZWF0ZWQgdmlhIGEgcmlnaHQgY2xpY2suIFlvdSBjYW4gYXNzaWduIHByb3BlcnRpZXMgdG9cbiAgIyAgICBgdGhpc2AgdG8gZHluYW1pY2FsbHkgY29tcHV0ZSB0aGUgY29tbWFuZCwgbGFiZWwsIGV0Yy4gVGhpcyBtZXRob2QgaXNcbiAgIyAgICBhY3R1YWxseSBjYWxsZWQgb24gYSBjbG9uZSBvZiB0aGUgb3JpZ2luYWwgaXRlbSB0ZW1wbGF0ZSB0byBwcmV2ZW50IHN0YXRlXG4gICMgICAgZnJvbSBsZWFraW5nIGFjcm9zcyBjb250ZXh0IG1lbnUgZGVwbG95bWVudHMuIENhbGxlZCB3aXRoIHRoZSBmb2xsb3dpbmdcbiAgIyAgICBhcmd1bWVudDpcbiAgIyAgICAgKiBgZXZlbnRgIFRoZSBjbGljayBldmVudCB0aGF0IGRlcGxveWVkIHRoZSBjb250ZXh0IG1lbnUuXG4gICMgICAqIGBzaG91bGREaXNwbGF5YCAob3B0aW9uYWwpIEEge0Z1bmN0aW9ufSB0aGF0IGlzIGNhbGxlZCB0byBkZXRlcm1pbmVcbiAgIyAgICAgd2hldGhlciB0byBkaXNwbGF5IHRoaXMgaXRlbSBvbiBhIGdpdmVuIGNvbnRleHQgbWVudSBkZXBsb3ltZW50LiBDYWxsZWRcbiAgIyAgICAgd2l0aCB0aGUgZm9sbG93aW5nIGFyZ3VtZW50OlxuICAjICAgICAqIGBldmVudGAgVGhlIGNsaWNrIGV2ZW50IHRoYXQgZGVwbG95ZWQgdGhlIGNvbnRleHQgbWVudS5cbiAgI1xuICAjIFJldHVybnMgYSB7RGlzcG9zYWJsZX0gb24gd2hpY2ggYC5kaXNwb3NlKClgIGNhbiBiZSBjYWxsZWQgdG8gcmVtb3ZlIHRoZVxuICAjIGFkZGVkIG1lbnUgaXRlbXMuXG4gIGFkZDogKGl0ZW1zQnlTZWxlY3RvciwgdGhyb3dPbkludmFsaWRTZWxlY3RvciA9IHRydWUpIC0+XG4gICAgYWRkZWRJdGVtU2V0cyA9IFtdXG5cbiAgICBmb3Igc2VsZWN0b3IsIGl0ZW1zIG9mIGl0ZW1zQnlTZWxlY3RvclxuICAgICAgdmFsaWRhdGVTZWxlY3RvcihzZWxlY3RvcikgaWYgdGhyb3dPbkludmFsaWRTZWxlY3RvclxuICAgICAgaXRlbVNldCA9IG5ldyBDb250ZXh0TWVudUl0ZW1TZXQoc2VsZWN0b3IsIGl0ZW1zKVxuICAgICAgYWRkZWRJdGVtU2V0cy5wdXNoKGl0ZW1TZXQpXG4gICAgICBAaXRlbVNldHMucHVzaChpdGVtU2V0KVxuXG4gICAgbmV3IERpc3Bvc2FibGUgPT5cbiAgICAgIGZvciBpdGVtU2V0IGluIGFkZGVkSXRlbVNldHNcbiAgICAgICAgQGl0ZW1TZXRzLnNwbGljZShAaXRlbVNldHMuaW5kZXhPZihpdGVtU2V0KSwgMSlcbiAgICAgIHJldHVyblxuXG4gIHRlbXBsYXRlRm9yRWxlbWVudDogKHRhcmdldCkgLT5cbiAgICBAdGVtcGxhdGVGb3JFdmVudCh7dGFyZ2V0fSlcblxuICB0ZW1wbGF0ZUZvckV2ZW50OiAoZXZlbnQpIC0+XG4gICAgdGVtcGxhdGUgPSBbXVxuICAgIGN1cnJlbnRUYXJnZXQgPSBldmVudC50YXJnZXRcblxuICAgIHdoaWxlIGN1cnJlbnRUYXJnZXQ/XG4gICAgICBjdXJyZW50VGFyZ2V0SXRlbXMgPSBbXVxuICAgICAgbWF0Y2hpbmdJdGVtU2V0cyA9XG4gICAgICAgIEBpdGVtU2V0cy5maWx0ZXIgKGl0ZW1TZXQpIC0+IGN1cnJlbnRUYXJnZXQud2Via2l0TWF0Y2hlc1NlbGVjdG9yKGl0ZW1TZXQuc2VsZWN0b3IpXG5cbiAgICAgIGZvciBpdGVtU2V0IGluIG1hdGNoaW5nSXRlbVNldHNcbiAgICAgICAgZm9yIGl0ZW0gaW4gaXRlbVNldC5pdGVtc1xuICAgICAgICAgIGl0ZW1Gb3JFdmVudCA9IEBjbG9uZUl0ZW1Gb3JFdmVudChpdGVtLCBldmVudClcbiAgICAgICAgICBpZiBpdGVtRm9yRXZlbnRcbiAgICAgICAgICAgIE1lbnVIZWxwZXJzLm1lcmdlKGN1cnJlbnRUYXJnZXRJdGVtcywgaXRlbUZvckV2ZW50LCBpdGVtU2V0LnNwZWNpZmljaXR5KVxuXG4gICAgICBmb3IgaXRlbSBpbiBjdXJyZW50VGFyZ2V0SXRlbXNcbiAgICAgICAgTWVudUhlbHBlcnMubWVyZ2UodGVtcGxhdGUsIGl0ZW0sIGZhbHNlKVxuXG4gICAgICBjdXJyZW50VGFyZ2V0ID0gY3VycmVudFRhcmdldC5wYXJlbnRFbGVtZW50XG5cbiAgICBAcHJ1bmVSZWR1bmRhbnRTZXBhcmF0b3JzKHRlbXBsYXRlKVxuICAgIEBhZGRBY2NlbGVyYXRvcnModGVtcGxhdGUpXG5cbiAgICByZXR1cm4gQHNvcnRUZW1wbGF0ZSh0ZW1wbGF0ZSlcblxuICAjIEFkZHMgYW4gYGFjY2VsZXJhdG9yYCBwcm9wZXJ0eSB0byBpdGVtcyB0aGF0IGhhdmUga2V5IGJpbmRpbmdzLiBFbGVjdHJvblxuICAjIHVzZXMgdGhpcyBwcm9wZXJ0eSB0byBzdXJmYWNlIHRoZSByZWxldmFudCBrZXltYXBzIGluIHRoZSBjb250ZXh0IG1lbnUuXG4gIGFkZEFjY2VsZXJhdG9yczogKHRlbXBsYXRlKSAtPlxuICAgIGZvciBpZCwgaXRlbSBvZiB0ZW1wbGF0ZVxuICAgICAgaWYgaXRlbS5jb21tYW5kXG4gICAgICAgIGtleW1hcHMgPSBAa2V5bWFwTWFuYWdlci5maW5kS2V5QmluZGluZ3Moe2NvbW1hbmQ6IGl0ZW0uY29tbWFuZCwgdGFyZ2V0OiBkb2N1bWVudC5hY3RpdmVFbGVtZW50fSlcbiAgICAgICAga2V5c3Ryb2tlcyA9IGtleW1hcHM/WzBdPy5rZXlzdHJva2VzXG4gICAgICAgIGlmIGtleXN0cm9rZXNcbiAgICAgICAgICAjIEVsZWN0cm9uIGRvZXMgbm90IHN1cHBvcnQgbXVsdGkta2V5c3Ryb2tlIGFjY2VsZXJhdG9ycy4gVGhlcmVmb3JlLFxuICAgICAgICAgICMgd2hlbiB0aGUgY29tbWFuZCBtYXBzIHRvIGEgbXVsdGktc3Ryb2tlIGtleSBiaW5kaW5nLCBzaG93IHRoZVxuICAgICAgICAgICMga2V5c3Ryb2tlcyBuZXh0IHRvIHRoZSBpdGVtJ3MgbGFiZWwuXG4gICAgICAgICAgaWYga2V5c3Ryb2tlcy5pbmNsdWRlcygnICcpXG4gICAgICAgICAgICBpdGVtLmxhYmVsICs9IFwiIFsje18uaHVtYW5pemVLZXlzdHJva2Uoa2V5c3Ryb2tlcyl9XVwiXG4gICAgICAgICAgZWxzZVxuICAgICAgICAgICAgaXRlbS5hY2NlbGVyYXRvciA9IE1lbnVIZWxwZXJzLmFjY2VsZXJhdG9yRm9yS2V5c3Ryb2tlKGtleXN0cm9rZXMpXG4gICAgICBpZiBBcnJheS5pc0FycmF5KGl0ZW0uc3VibWVudSlcbiAgICAgICAgQGFkZEFjY2VsZXJhdG9ycyhpdGVtLnN1Ym1lbnUpXG5cbiAgcHJ1bmVSZWR1bmRhbnRTZXBhcmF0b3JzOiAobWVudSkgLT5cbiAgICBrZWVwTmV4dEl0ZW1JZlNlcGFyYXRvciA9IGZhbHNlXG4gICAgaW5kZXggPSAwXG4gICAgd2hpbGUgaW5kZXggPCBtZW51Lmxlbmd0aFxuICAgICAgaWYgbWVudVtpbmRleF0udHlwZSBpcyAnc2VwYXJhdG9yJ1xuICAgICAgICBpZiBub3Qga2VlcE5leHRJdGVtSWZTZXBhcmF0b3Igb3IgaW5kZXggaXMgbWVudS5sZW5ndGggLSAxXG4gICAgICAgICAgbWVudS5zcGxpY2UoaW5kZXgsIDEpXG4gICAgICAgIGVsc2VcbiAgICAgICAgICBpbmRleCsrXG4gICAgICBlbHNlXG4gICAgICAgIGtlZXBOZXh0SXRlbUlmU2VwYXJhdG9yID0gdHJ1ZVxuICAgICAgICBpbmRleCsrXG5cbiAgc29ydFRlbXBsYXRlOiAodGVtcGxhdGUpIC0+XG4gICAgdGVtcGxhdGUgPSBzb3J0TWVudUl0ZW1zKHRlbXBsYXRlKVxuICAgIGZvciBpZCwgaXRlbSBvZiB0ZW1wbGF0ZVxuICAgICAgaWYgQXJyYXkuaXNBcnJheShpdGVtLnN1Ym1lbnUpXG4gICAgICAgIGl0ZW0uc3VibWVudSA9IEBzb3J0VGVtcGxhdGUoaXRlbS5zdWJtZW51KVxuICAgIHJldHVybiB0ZW1wbGF0ZVxuXG4gICMgUmV0dXJucyBhbiBvYmplY3QgY29tcGF0aWJsZSB3aXRoIGA6OmFkZCgpYCBvciBgbnVsbGAuXG4gIGNsb25lSXRlbUZvckV2ZW50OiAoaXRlbSwgZXZlbnQpIC0+XG4gICAgcmV0dXJuIG51bGwgaWYgaXRlbS5kZXZNb2RlIGFuZCBub3QgQGRldk1vZGVcbiAgICBpdGVtID0gT2JqZWN0LmNyZWF0ZShpdGVtKVxuICAgIGlmIHR5cGVvZiBpdGVtLnNob3VsZERpc3BsYXkgaXMgJ2Z1bmN0aW9uJ1xuICAgICAgcmV0dXJuIG51bGwgdW5sZXNzIGl0ZW0uc2hvdWxkRGlzcGxheShldmVudClcbiAgICBpdGVtLmNyZWF0ZWQ/KGV2ZW50KVxuICAgIGlmIEFycmF5LmlzQXJyYXkoaXRlbS5zdWJtZW51KVxuICAgICAgaXRlbS5zdWJtZW51ID0gaXRlbS5zdWJtZW51XG4gICAgICAgIC5tYXAoKHN1Ym1lbnVJdGVtKSA9PiBAY2xvbmVJdGVtRm9yRXZlbnQoc3VibWVudUl0ZW0sIGV2ZW50KSlcbiAgICAgICAgLmZpbHRlcigoc3VibWVudUl0ZW0pIC0+IHN1Ym1lbnVJdGVtIGlzbnQgbnVsbClcbiAgICByZXR1cm4gaXRlbVxuXG4gIHNob3dGb3JFdmVudDogKGV2ZW50KSAtPlxuICAgIEBhY3RpdmVFbGVtZW50ID0gZXZlbnQudGFyZ2V0XG4gICAgbWVudVRlbXBsYXRlID0gQHRlbXBsYXRlRm9yRXZlbnQoZXZlbnQpXG5cbiAgICByZXR1cm4gdW5sZXNzIG1lbnVUZW1wbGF0ZT8ubGVuZ3RoID4gMFxuICAgIHJlbW90ZS5nZXRDdXJyZW50V2luZG93KCkuZW1pdCgnY29udGV4dC1tZW51JywgbWVudVRlbXBsYXRlKVxuICAgIHJldHVyblxuXG4gIGNsZWFyOiAtPlxuICAgIEBhY3RpdmVFbGVtZW50ID0gbnVsbFxuICAgIEBpdGVtU2V0cyA9IFtdXG4gICAgaW5zcGVjdEVsZW1lbnQgPSB7XG4gICAgICAnYXRvbS13b3Jrc3BhY2UnOiBbe1xuICAgICAgICBsYWJlbDogJ0luc3BlY3QgRWxlbWVudCdcbiAgICAgICAgY29tbWFuZDogJ2FwcGxpY2F0aW9uOmluc3BlY3QnXG4gICAgICAgIGRldk1vZGU6IHRydWVcbiAgICAgICAgY3JlYXRlZDogKGV2ZW50KSAtPlxuICAgICAgICAgIHtwYWdlWCwgcGFnZVl9ID0gZXZlbnRcbiAgICAgICAgICBAY29tbWFuZERldGFpbCA9IHt4OiBwYWdlWCwgeTogcGFnZVl9XG4gICAgICB9XVxuICAgIH1cbiAgICBAYWRkKGluc3BlY3RFbGVtZW50LCBmYWxzZSlcblxuY2xhc3MgQ29udGV4dE1lbnVJdGVtU2V0XG4gIGNvbnN0cnVjdG9yOiAoQHNlbGVjdG9yLCBAaXRlbXMpIC0+XG4gICAgQHNwZWNpZmljaXR5ID0gY2FsY3VsYXRlU3BlY2lmaWNpdHkoQHNlbGVjdG9yKVxuIl19
