"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var natural = require("natural");
var regexHelper = require("./regex-helper");
/**
 * Checks the contents of a buffer against a spelling manager, producing a
 * tokenized list and the check status for each one.
 */
var BufferSpellingChecker = /** @class */ (function () {
    function BufferSpellingChecker(
    // Save the member variables.
    spellingManager, tokenizer) {
        if (tokenizer === void 0) { tokenizer = null; }
        // We weren't provided a tokenizer, we create a 'best guess' one that
        // handles most of the conditions. This attempts to include Unicode
        // breaks also to identify those words. We also avoid punctuation only
        // elements but do include punctuation inside a word so we can treat
        // "didn't" as a single word.
        if (!tokenizer) {
            tokenizer = new natural.RegexpTokenizer({
                pattern: regexHelper.wordTokenzier,
            });
        }
        this.spellingManager = spellingManager;
        this.tokenizer = tokenizer;
    }
    BufferSpellingChecker.prototype.check = function (buffer) {
        // If we have a blank or empty string, then just return an empty list.
        if (!buffer || buffer.trim() === "") {
            return new Array();
        }
        // Since we have useful values, we need to now tokenize them. This
        // doesn't give us positional information, but we'll build that up as we
        // figure out the spelling status.
        var startSearch = 0;
        var tokens = this.tokenizer.tokenize(buffer);
        var results = new Array();
        for (var _i = 0, tokens_1 = tokens; _i < tokens_1.length; _i++) {
            var token = tokens_1[_i];
            // If we don't have at least one character, skip it.
            if (!regexHelper.isWord.test(token)) {
                continue;
            }
            // Figure out where this token appears in the buffer.
            var tokenIndex = buffer.indexOf(token, startSearch);
            if (tokenIndex < 0) {
                // We should never get to this.
                throw new Error("Cannot find token '"
                    + token
                    + "' starting at position "
                    + startSearch
                    + "}.");
            }
            startSearch = tokenIndex + token.length;
            // Figure out the spelling status.
            var checkStatus = this.spellingManager.check(token);
            // Build up the token status.
            var tokenStatus = {
                end: tokenIndex + token.length,
                start: tokenIndex,
                status: checkStatus,
                token: token,
            };
            results.push(tokenStatus);
        }
        // Return the results, which includes all tokens.
        return results;
    };
    return BufferSpellingChecker;
}());
exports.BufferSpellingChecker = BufferSpellingChecker;
//# sourceMappingURL=BufferSpellingChecker.js.map