"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = IssueishLink;
exports.handleClickEvent = handleClickEvent;
exports.openIssueishLinkInNewTab = openIssueishLinkInNewTab;
exports.openLinkInBrowser = openLinkInBrowser;
exports.getDataFromGithubUrl = getDataFromGithubUrl;

var _url = _interopRequireDefault(require("url"));

var _electron = require("electron");

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _issueishDetailItem = _interopRequireDefault(require("../items/issueish-detail-item"));

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _objectWithoutProperties(source, excluded) { if (source == null) return {}; var target = _objectWithoutPropertiesLoose(source, excluded); var key, i; if (Object.getOwnPropertySymbols) { var sourceSymbolKeys = Object.getOwnPropertySymbols(source); for (i = 0; i < sourceSymbolKeys.length; i++) { key = sourceSymbolKeys[i]; if (excluded.indexOf(key) >= 0) continue; if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue; target[key] = source[key]; } } return target; }

function _objectWithoutPropertiesLoose(source, excluded) { if (source == null) return {}; var target = {}; var sourceKeys = Object.keys(source); var key, i; for (i = 0; i < sourceKeys.length; i++) { key = sourceKeys[i]; if (excluded.indexOf(key) >= 0) continue; target[key] = source[key]; } return target; }

// eslint-disable-next-line no-shadow
function IssueishLink(_ref) {
  let {
    url,
    children
  } = _ref,
      others = _objectWithoutProperties(_ref, ["url", "children"]);

  function clickHandler(event) {
    handleClickEvent(event, url);
  }

  return _react["default"].createElement("a", _extends({}, others, {
    onClick: clickHandler
  }), children);
}

IssueishLink.propTypes = {
  url: _propTypes["default"].string.isRequired,
  children: _propTypes["default"].node
}; // eslint-disable-next-line no-shadow

function handleClickEvent(event, url) {
  event.preventDefault();
  event.stopPropagation();

  if (!event.shiftKey) {
    return openIssueishLinkInNewTab(url, {
      activate: !(event.metaKey || event.ctrlKey)
    });
  } else {
    // Open in browser if shift key held
    return openLinkInBrowser(url);
  }
} // eslint-disable-next-line no-shadow


function openIssueishLinkInNewTab(url, options = {}) {
  const uri = getAtomUriForGithubUrl(url);

  if (uri) {
    return openInNewTab(uri, options);
  } else {
    return null;
  }
}

function openLinkInBrowser(uri) {
  return new Promise((resolve, reject) => {
    _electron.shell.openExternal(uri, {}, err => {
      if (err) {
        reject(err);
      } else {
        (0, _reporterProxy.addEvent)('open-issueish-in-browser', {
          "package": 'github',
          from: 'issueish-link'
        });
        resolve();
      }
    });
  });
}

function getAtomUriForGithubUrl(githubUrl) {
  return getUriForData(getDataFromGithubUrl(githubUrl));
}

function getDataFromGithubUrl(githubUrl) {
  const {
    hostname,
    pathname
  } = _url["default"].parse(githubUrl);

  const [repoOwner, repoName, type, issueishNumber] = pathname.split('/').filter(s => s);
  return {
    hostname,
    repoOwner,
    repoName,
    type,
    issueishNumber: parseInt(issueishNumber, 10)
  };
}

function getUriForData({
  hostname,
  repoOwner,
  repoName,
  type,
  issueishNumber
}) {
  if (hostname !== 'github.com' || !['pull', 'issues'].includes(type) || !issueishNumber || isNaN(issueishNumber)) {
    return null;
  } else {
    return _issueishDetailItem["default"].buildURI({
      host: 'github.com',
      owner: repoOwner,
      repo: repoName,
      number: issueishNumber
    });
  }
}

function openInNewTab(uri, {
  activate
} = {
  activate: true
}) {
  return atom.workspace.open(uri, {
    activateItem: activate
  }).then(() => {
    (0, _reporterProxy.addEvent)('open-issueish-in-pane', {
      "package": 'github',
      from: 'issueish-link',
      target: 'new-tab'
    });
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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