'use strict';

Object.defineProperty(exports, '__esModule', {
  value: true
});
exports.default = exports.serialize = exports.test = void 0;

var _markup = require('./lib/markup');

/**
 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *
 */
const ELEMENT_NODE = 1;
const TEXT_NODE = 3;
const COMMENT_NODE = 8;
const FRAGMENT_NODE = 11;
const ELEMENT_REGEXP = /^((HTML|SVG)\w*)?Element$/;

const testNode = (nodeType, name) =>
  (nodeType === ELEMENT_NODE && ELEMENT_REGEXP.test(name)) ||
  (nodeType === TEXT_NODE && name === 'Text') ||
  (nodeType === COMMENT_NODE && name === 'Comment') ||
  (nodeType === FRAGMENT_NODE && name === 'DocumentFragment');

const test = val =>
  val &&
  val.constructor &&
  val.constructor.name &&
  testNode(val.nodeType, val.constructor.name); // Convert array of attribute objects to keys array and props object.

exports.test = test;

const keysMapper = attribute => attribute.name;

const propsReducer = (props, attribute) => {
  props[attribute.name] = attribute.value;
  return props;
};

const serialize = (node, config, indentation, depth, refs, printer) => {
  if (node.nodeType === TEXT_NODE) {
    return (0, _markup.printText)(node.data, config);
  }

  if (node.nodeType === COMMENT_NODE) {
    return (0, _markup.printComment)(node.data, config);
  }

  const type =
    node.nodeType === FRAGMENT_NODE
      ? `DocumentFragment`
      : node.tagName.toLowerCase();

  if (++depth > config.maxDepth) {
    return (0, _markup.printElementAsLeaf)(type, config);
  }

  return (0, _markup.printElement)(
    type,
    (0, _markup.printProps)(
      Array.prototype.map.call(node.attributes || [], keysMapper).sort(),
      Array.prototype.reduce.call(node.attributes || [], propsReducer, {}),
      config,
      indentation + config.indent,
      depth,
      refs,
      printer
    ),
    (0, _markup.printChildren)(
      Array.prototype.slice.call(node.childNodes || node.children),
      config,
      indentation + config.indent,
      depth,
      refs,
      printer
    ),
    config,
    indentation
  );
};

exports.serialize = serialize;
var _default = {
  serialize,
  test
};
exports.default = _default;
