"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var fs = require("fs");
var indentation_1 = require("./validators/indentation");
var attributes_1 = require("./validators/attributes");
var glob = require("glob");
var logger_1 = require("./utils/logger");
var chalk_1 = require("chalk");
var Linter = /** @class */ (function () {
    function Linter() {
    }
    Linter.lint = function (config, fileList) {
        var _this = this;
        return new Promise(function (resolve, reject) {
            var filePaths = fileList || config.files;
            if (!Array.isArray(filePaths) || filePaths.length === 0) {
                logger_1.default.logHelp('files must be an array in config file, or a comma-separated list on command line');
                resolve(1);
            }
            else {
                var promises_1 = [];
                filePaths.forEach(function (filePath) {
                    promises_1.push(_this.checkFile(filePath, config));
                });
                Promise.all(promises_1).then(function (results) {
                    var totalLintErrors = results.reduce(function (prev, curr) { return prev + curr; }, 0);
                    if (totalLintErrors === 0) {
                        logger_1.default.log(chalk_1.default.green('All files pass linting'));
                    }
                    resolve(totalLintErrors);
                });
            }
        });
    };
    Linter.checkFile = function (filePath, config) {
        return new Promise(function (resolve) {
            glob(filePath, function (globError, files) {
                if (globError) {
                    logger_1.default.logResults(filePath, [
                        {
                            line: 0,
                            column: 0,
                            message: globError.toString()
                        }
                    ]);
                }
                else {
                    var filesRead_1 = 0;
                    var lintErrors_1 = 0;
                    files.forEach(function (file) {
                        fs.readFile(file, function (fileError, data) {
                            if (fileError) {
                                logger_1.default.logResults(file, [
                                    {
                                        line: 0,
                                        column: 0,
                                        message: fileError.toString()
                                    }
                                ]);
                            }
                            else {
                                var errors = [];
                                var fileString = data.toString();
                                var lines = fileString.split('\n');
                                if (config.indentation) {
                                    errors.push.apply(errors, indentation_1.Indentation.validate(lines, config.indentation));
                                }
                                if (config.attributes) {
                                    errors.push.apply(errors, attributes_1.Attributes.validate(lines, config.attributes));
                                }
                                if (errors.length > 0) {
                                    lintErrors_1 += errors.length;
                                    logger_1.default.logResults(file, errors);
                                }
                            }
                            if (++filesRead_1 === files.length) {
                                resolve(lintErrors_1);
                            }
                        });
                    });
                }
            });
        });
    };
    return Linter;
}());
exports.Linter = Linter;
