"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

let key = 0;

class StubItem {
  // StubItems should only be created by `create` and never constructed directly.
  static create(name, props, uri = '') {
    const stub = new StubItem(name, props, uri);
    const override = {
      _getStub: () => stub,
      getElement: () => stub.getElement(),
      destroy: stub.destroy.bind(stub)
    };
    const proxy = new Proxy(override, {
      get(target, propName) {
        const item = stub.getRealItem();

        if (Reflect.has(target, propName)) {
          return target[propName];
        } else if (item && Reflect.has(item, propName)) {
          let val = item[propName];

          if (typeof val === 'function') {
            val = val.bind(item);
          }

          return val;
        } else {
          let val = stub[propName];

          if (typeof val === 'function') {
            val = val.bind(stub);
          }

          return val;
        }
      }

    });
    return proxy;
  }

  constructor(name, props = {}, uri) {
    this.emitter = new _eventKit.Emitter();
    this.subscriptions = new _eventKit.CompositeDisposable();
    this.name = name;
    this.props = props;
    this.uri = uri;
    this.key = ++key;
    this.element = document.createElement('div');
    this.element.classList.add(`github-StubItem-${name}`);
    this.realItem = null;
    this.realItemPromise = new Promise(res => {
      this.resolveRealItemPromise = res;
    });
  }

  setRealItem(item) {
    this.realItem = item;

    if (this.realItem.getRealItemPromise) {
      this.realItem.getRealItemPromise().then(this.resolveRealItemPromise);
    } else {
      this.resolveRealItemPromise(this.realItem);
    }

    this.emitter.emit('did-change-title');
    this.emitter.emit('did-change-icon');

    if (item.onDidChangeTitle) {
      this.subscriptions.add(item.onDidChangeTitle((...args) => this.emitter.emit('did-change-title', ...args)));
    }

    if (item.onDidChangeIcon) {
      this.subscriptions.add(item.onDidChangeIcon((...args) => this.emitter.emit('did-change-icon', ...args)));
    }

    if (item.onDidDestroy) {
      this.subscriptions.add(item.onDidDestroy((...args) => {
        this.realItem = null;
        this.emitter.emit('did-destroy', ...args);
      }));
    }
  }

  getRealItemPromise() {
    return this.realItemPromise;
  }

  getRealItem() {
    return this.realItem;
  }

  getURI() {
    return this.uri;
  }

  getTitle() {
    return this.props.title || null;
  }

  getIconName() {
    return this.props.iconName || null;
  }

  onDidChangeTitle(cb) {
    return this.emitter.on('did-change-title', cb);
  }

  onDidChangeIcon(cb) {
    return this.emitter.on('did-change-icon', cb);
  }

  getElement() {
    return this.element;
  }

  onDidDestroy(cb) {
    return this.emitter.on('did-destroy', cb);
  }

  destroy() {
    this.resolveRealItemPromise(null);
    this.subscriptions.dispose();
    this.emitter.dispose();

    if (this.realItem) {
      this.realItem.destroy && this.realItem.destroy();
    }
  }

}

exports["default"] = StubItem;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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