/**
 * @file The `Setoid` type class represents types which support decidable equality.
 *
 * Instances must satisfy the following laws:
 *
 * 1. Reflexivity: `S.equals(a, a) === true`
 * 2. Symmetry: `S.equals(a, b) === S.equals(b, a)`
 * 3. Transitivity: if `S.equals(a, b) === true` and `S.equals(b, c) === true`, then `S.equals(a, c) === true`
 *
 * See [Getting started with fp-ts: Setoid](https://dev.to/gcanti/getting-started-with-fp-ts-setoid-39f3)
 */
/**
 * @since 1.14.0
 */
export const fromEquals = (equals) => {
    return {
        equals: (x, y) => x === y || equals(x, y)
    };
};
/**
 * @since 1.0.0
 */
export const strictEqual = (a, b) => {
    return a === b;
};
const setoidStrict = { equals: strictEqual };
/**
 * @since 1.0.0
 */
export const setoidString = setoidStrict;
/**
 * @since 1.0.0
 */
export const setoidNumber = setoidStrict;
/**
 * @since 1.0.0
 */
export const setoidBoolean = setoidStrict;
/**
 * @since 1.0.0
 */
export const getArraySetoid = (S) => {
    return fromEquals((xs, ys) => xs.length === ys.length && xs.every((x, i) => S.equals(x, ys[i])));
};
/**
 * @since 1.14.2
 */
export const getStructSetoid = (setoids) => {
    return fromEquals((x, y) => {
        for (const k in setoids) {
            if (!setoids[k].equals(x[k], y[k])) {
                return false;
            }
        }
        return true;
    });
};
/**
 * Use `getStructSetoid` instead
 * @since 1.0.0
 * @deprecated
 */
export const getRecordSetoid = (setoids) => {
    return getStructSetoid(setoids);
};
/**
 * Given a tuple of `Setoid`s returns a `Setoid` for the tuple
 *
 * @example
 * import { getTupleSetoid, setoidString, setoidNumber, setoidBoolean } from 'fp-ts/lib/Setoid'
 *
 * const S = getTupleSetoid(setoidString, setoidNumber, setoidBoolean)
 * assert.strictEqual(S.equals(['a', 1, true], ['a', 1, true]), true)
 * assert.strictEqual(S.equals(['a', 1, true], ['b', 1, true]), false)
 * assert.strictEqual(S.equals(['a', 1, true], ['a', 2, true]), false)
 * assert.strictEqual(S.equals(['a', 1, true], ['a', 1, false]), false)
 *
 * @since 1.14.2
 */
export const getTupleSetoid = (...setoids) => {
    return fromEquals((x, y) => setoids.every((S, i) => S.equals(x[i], y[i])));
};
/**
 * Use `getTupleSetoid` instead
 * @since 1.0.0
 * @deprecated
 */
export const getProductSetoid = (SA, SB) => {
    return getTupleSetoid(SA, SB);
};
/**
 * Returns the `Setoid` corresponding to the partitions of `B` induced by `f`
 *
 * @since 1.2.0
 */
export const contramap = (f, fa) => {
    return fromEquals((x, y) => fa.equals(f(x), f(y)));
};
/**
 * @since 1.4.0
 */
export const setoidDate = contramap(date => date.valueOf(), setoidNumber);
