"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const uuid_1 = require("./uuid");
const indexeddb_1 = require("./databases/indexeddb");
const util_1 = require("./util");
// if you're running a local instance of central, use
// "http://localhost:4000/api/usage/" instead.
const baseUsageApi = "https://central.github.com/api/usage/";
exports.LastDailyStatsReportKey = "last-daily-stats-report";
/** The localStorage key for whether the user has opted out. */
exports.StatsOptOutKey = "stats-opt-out";
/** The localStorage key that indicates that we're sending the stats to the server. */
exports.SendingStatsKey = "metrics:stats-being-sent";
/** Have we successfully sent the stats opt-in? */
exports.HasSentOptInPingKey = "has-sent-stats-opt-in-ping";
/** milliseconds in a minute (for readability, dawg) */
const minutes = 60 * 1000;
/** milliseconds in an hour (for readability, dawg) */
const hours = 60 * minutes;
/** How often do we want to check to report stats (also the delay until the first report). */
const MaxReportingFrequency = 10 * minutes;
/** How often daily stats should be submitted (i.e., 24 hours). */
exports.DailyStatsReportIntervalInMs = hours * 24;
/** After this amount of time we'll. */
exports.TimeoutForReportingLock = hours * 2;
/** The goal is for this package to be app-agnostic so we can add
 * other editors in the future.
 */
var AppName;
(function (AppName) {
    AppName["Atom"] = "atom";
})(AppName = exports.AppName || (exports.AppName = {}));
class StatsStore {
    constructor(appName, version, isDevMode, getAccessToken = () => "", options = {}) {
        this.database = new indexeddb_1.default();
        this.version = version;
        this.appUrl = baseUsageApi + appName;
        const optOutValue = localStorage.getItem(exports.StatsOptOutKey);
        this.isDevMode = !options.logInDevMode && isDevMode;
        this.getAccessToken = getAccessToken;
        this.gitHubUser = null;
        this.reportingFrequency = options.reportingFrequency || exports.DailyStatsReportIntervalInMs;
        // We set verbose mode when logging in Dev mode to prevent users from forgetting to turn
        // off the logging in dev mode.
        this.verboseMode = (!!options.logInDevMode && isDevMode) || !!options.verboseMode;
        this.timer = this.getTimer(Math.min(this.reportingFrequency / 6, MaxReportingFrequency));
        if (optOutValue) {
            this.optOut = !!parseInt(optOutValue, 10);
            // If the user has set an opt out value but we haven't sent the ping yet,
            // give it a shot now.
            if (!localStorage.getItem(exports.HasSentOptInPingKey)) {
                this.sendOptInStatusPing(!this.optOut);
            }
        }
        else {
            this.optOut = false;
        }
    }
    end() {
        clearInterval(this.timer);
    }
    setGitHubUser(gitHubUser) {
        this.gitHubUser = gitHubUser;
    }
    /** Set whether the user has opted out of stats reporting. */
    async setOptOut(optOut) {
        const changed = this.optOut !== optOut;
        this.optOut = optOut;
        localStorage.setItem(exports.StatsOptOutKey, optOut ? "1" : "0");
        if (changed) {
            await this.sendOptInStatusPing(!optOut);
        }
    }
    async reportStats() {
        if (this.optOut || this.isDevMode) {
            return;
        }
        // If multiple instances of `telemetry` are being run from different
        // renderer process, we want to avoid two instances to send the same
        // stats.
        // We use a timed mutex so if for some reason the lock is not released
        // after reporting the metrics the metrics can still be sent after some
        // timeout.
        // Remember to not perform async operations between the `localStorage.setItem()`
        // and the `localStorage.getItem()` calls. This way we can take advantage of the
        // LocalStorage mutex on Chrome: https://www.w3.org/TR/webstorage/#threads
        if (!this.isDateBefore(localStorage.getItem(exports.SendingStatsKey), exports.TimeoutForReportingLock)) {
            return;
        }
        localStorage.setItem(exports.SendingStatsKey, Date.now().toString());
        try {
            const stats = await this.getDailyStats();
            const response = await this.post(stats);
            if (response.status !== 200) {
                throw new Error(`Stats reporting failure: ${response.status})`);
            }
            else {
                await localStorage.setItem(exports.LastDailyStatsReportKey, Date.now().toString());
                await this.database.clearData();
                console.log("stats successfully reported");
            }
        }
        catch (err) {
            // todo (tt, 5/2018): would be good to log these errors to Haystack/Datadog
            // so we have some kind of visibility into how often things are failing.
            console.log(err);
        }
        finally {
            // Delete the "mutex" used to ensure that stats are not sent at the same time by
            // two different processes.
            localStorage.removeItem(exports.SendingStatsKey);
        }
    }
    async clearData() {
        await this.database.clearData();
    }
    /* send a ping to indicate that the user has changed their opt-in preferences.
    * public for testing purposes only.
    */
    async sendOptInStatusPing(optIn) {
        if (this.isDevMode) {
            return;
        }
        const direction = optIn ? "in" : "out";
        try {
            const response = await this.post({
                eventType: "ping",
                dimensions: {
                    optIn,
                },
            });
            if (response.status !== 200) {
                throw new Error(`Error sending opt in ping: ${response.status}`);
            }
            localStorage.setItem(exports.HasSentOptInPingKey, "1");
            console.log(`Opt ${direction} reported.`);
        }
        catch (err) {
            // todo (tt, 5/2018): would be good to log these errors to Haystack/Datadog
            // so we have some kind of visibility into how often things are failing.
            console.log(`Error reporting opt ${direction}`, err);
        }
    }
    // public for testing purposes only
    async getDailyStats() {
        return {
            measures: await this.database.getCounters(),
            customEvents: await this.database.getCustomEvents(),
            timings: await this.database.getTimings(),
            dimensions: {
                appVersion: this.version,
                platform: process.platform,
                guid: uuid_1.getGUID(),
                eventType: "usage",
                date: util_1.getISODate(),
                language: process.env.LANG || "",
                gitHubUser: this.gitHubUser,
            },
        };
    }
    async addCustomEvent(eventType, event) {
        await this.database.addCustomEvent(eventType, event);
    }
    /**
     * Add timing data to the stats store, to be sent with the daily metrics requests.
     */
    async addTiming(eventType, durationInMilliseconds, metadata = {}) {
        // don't increment in dev mode because localStorage
        // is shared across dev and non dev windows and there's
        // no way to keep dev and non-dev metrics separate.
        // don't increment if the user has opted out, because
        // we want to respect user privacy.
        if (this.isDevMode || this.optOut) {
            return;
        }
        await this.database.addTiming(eventType, durationInMilliseconds, metadata);
    }
    /**
     * Increment a counter.  This is used to track usage statistics.
     */
    async incrementCounter(counterName) {
        // don't increment in dev mode because localStorage
        // is shared across dev and non dev windows and there's
        // no way to keep dev and non-dev metrics separate.
        // don't increment if the user has opted out, because
        // we want to respect user privacy.
        if (this.isDevMode || this.optOut) {
            return;
        }
        await this.database.incrementCounter(counterName);
    }
    /** Post some data to our stats endpoint.
     * This is public for testing purposes only.
     */
    async post(body) {
        if (this.verboseMode) {
            console.log("Sending metrics", body);
        }
        const requestHeaders = { "Content-Type": "application/json" };
        const token = this.getAccessToken();
        if (token) {
            requestHeaders.Authorization = `token ${token}`;
        }
        const options = {
            method: "POST",
            headers: requestHeaders,
            body: JSON.stringify(body),
        };
        return this.fetch(this.appUrl, options);
    }
    /** Exists to enable us to mock fetch in tests
     * This is public for testing purposes only.
     */
    async fetch(url, options) {
        return fetch(url, options);
    }
    /** Should the app report its daily stats?
     * Public for testing purposes only.
     */
    hasReportingIntervalElapsed() {
        return this.isDateBefore(localStorage.getItem(exports.LastDailyStatsReportKey), this.reportingFrequency);
    }
    /** Set a timer so we can report the stats when the time comes. */
    getTimer(loopInterval) {
        // todo (tt, 5/2018): maybe we shouldn't even set up the timer
        // in dev mode or if the user has opted out.
        const timer = setInterval(() => {
            if (this.hasReportingIntervalElapsed()) {
                this.reportStats();
            }
        }, loopInterval);
        if (timer.unref !== undefined) {
            // make sure we don't block node from exiting in tests
            // https://stackoverflow.com/questions/48172363/mocha-test-suite-never-ends-when-setinterval-running
            timer.unref();
        }
        return timer;
    }
    /**
     * Helper method that returns whether the difference between the current date and the specified date is
     * less than the specified amount of milliseconds.
     *
     * The pass date should be a string representing the number of milliseconds elapsed since
     * January 1, 1970 00:00:00 UTC.
     */
    isDateBefore(dateAsString, numMilliseconds) {
        if (!dateAsString || dateAsString.length === 0) {
            return true;
        }
        const date = parseInt(dateAsString, 10);
        if (isNaN(date)) {
            return true;
        }
        return (Date.now() - date) > numMilliseconds;
    }
}
exports.StatsStore = StatsStore;
//# sourceMappingURL=index.js.map