import { compose, identity, concat } from './function';
import { fold as foldSemigroup, getDictionarySemigroup, getDualSemigroup, getFunctionSemigroup, getJoinSemigroup, getMeetSemigroup, semigroupAll, semigroupAny, semigroupProduct, semigroupString, semigroupSum, semigroupVoid, getStructSemigroup, getTupleSemigroup } from './Semigroup';
/**
 * @since 1.0.0
 */
export const fold = (M) => {
    return foldSemigroup(M)(M.empty);
};
/**
 * Given a tuple of monoids returns a monoid for the tuple
 *
 * @example
 * import { getTupleMonoid, monoidString, monoidSum, monoidAll } from 'fp-ts/lib/Monoid'
 *
 * const M1 = getTupleMonoid(monoidString, monoidSum)
 * assert.deepStrictEqual(M1.concat(['a', 1], ['b', 2]), ['ab', 3])
 *
 * const M2 = getTupleMonoid(monoidString, monoidSum, monoidAll)
 * assert.deepStrictEqual(M2.concat(['a', 1, true], ['b', 2, false]), ['ab', 3, false])
 *
 * @since 1.0.0
 */
export const getTupleMonoid = (...monoids) => {
    return Object.assign({}, getTupleSemigroup(...monoids), { empty: monoids.map(m => m.empty) });
};
/**
 * Use `getTupleMonoid` instead
 * @since 1.0.0
 * @deprecated
 */
export const getProductMonoid = (MA, MB) => {
    return getTupleMonoid(MA, MB);
};
/**
 * @since 1.0.0
 */
export const getDualMonoid = (M) => {
    return Object.assign({}, getDualSemigroup(M), { empty: M.empty });
};
/**
 * Boolean monoid under conjunction
 * @since 1.0.0
 */
export const monoidAll = Object.assign({}, semigroupAll, { empty: true });
/**
 * Boolean monoid under disjunction
 * @since 1.0.0
 */
export const monoidAny = Object.assign({}, semigroupAny, { empty: false });
const emptyArray = [];
/**
 * @since 1.0.0
 */
export const unsafeMonoidArray = {
    concat,
    empty: emptyArray
};
/**
 * `Monoid` under array concatenation
 *
 * @since 1.0.0
 */
export const getArrayMonoid = () => {
    return unsafeMonoidArray;
};
const emptyObject = {};
export function getDictionaryMonoid(S) {
    return Object.assign({}, getDictionarySemigroup(S), { empty: emptyObject });
}
/**
 * Number monoid under addition
 * @since 1.0.0
 */
export const monoidSum = Object.assign({}, semigroupSum, { empty: 0 });
/**
 * Number monoid under multiplication
 * @since 1.0.0
 */
export const monoidProduct = Object.assign({}, semigroupProduct, { empty: 1 });
/**
 * @since 1.0.0
 */
export const monoidString = Object.assign({}, semigroupString, { empty: '' });
/**
 * @since 1.0.0
 */
export const monoidVoid = Object.assign({}, semigroupVoid, { empty: undefined });
/**
 * @since 1.0.0
 */
export const getFunctionMonoid = (M) => () => {
    return Object.assign({}, getFunctionSemigroup(M)(), { empty: () => M.empty });
};
/**
 * @since 1.0.0
 */
export const getEndomorphismMonoid = () => {
    return {
        concat: compose,
        empty: identity
    };
};
/**
 * @since 1.14.0
 */
export const getStructMonoid = (monoids) => {
    const empty = {};
    for (const key of Object.keys(monoids)) {
        empty[key] = monoids[key].empty;
    }
    return Object.assign({}, getStructSemigroup(monoids), { empty });
};
/**
 * Use `getStructMonoid` instead
 * @since 1.0.0
 * @deprecated
 */
export const getRecordMonoid = (monoids) => {
    return getStructMonoid(monoids);
};
/**
 * @since 1.9.0
 */
export const getMeetMonoid = (B) => {
    return Object.assign({}, getMeetSemigroup(B), { empty: B.top });
};
/**
 * @since 1.9.0
 */
export const getJoinMonoid = (B) => {
    return Object.assign({}, getJoinSemigroup(B), { empty: B.bottom });
};
