/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,missingReturn,unusedPrivateMembers,uselessCode} checked by tsc
 */
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import './ng_dev_mode';
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects, stringify } from '../util';
import { EMPTY_ARRAY, EMPTY_OBJ } from './empty';
import { NG_COMPONENT_DEF, NG_DIRECTIVE_DEF, NG_MODULE_DEF, NG_PIPE_DEF } from './fields';
/** @type {?} */
let _renderCompCount = 0;
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ngComponentDef = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @template T
 * @param {?} componentDefinition
 * @return {?}
 */
export function defineComponent(componentDefinition) {
    /** @type {?} */
    const type = componentDefinition.type;
    /** @type {?} */
    const typePrototype = type.prototype;
    /** @type {?} */
    const declaredInputs = (/** @type {?} */ ({}));
    /** @type {?} */
    const def = {
        type: type,
        providersResolver: null,
        consts: componentDefinition.consts,
        vars: componentDefinition.vars,
        factory: componentDefinition.factory,
        template: componentDefinition.template || (/** @type {?} */ (null)),
        hostBindings: componentDefinition.hostBindings || null,
        contentQueries: componentDefinition.contentQueries || null,
        contentQueriesRefresh: componentDefinition.contentQueriesRefresh || null,
        attributes: componentDefinition.attributes || null,
        declaredInputs: declaredInputs,
        inputs: (/** @type {?} */ (null)),
        // assigned in noSideEffects
        outputs: (/** @type {?} */ (null)),
        // assigned in noSideEffects
        exportAs: componentDefinition.exportAs || null,
        onInit: typePrototype.ngOnInit || null,
        doCheck: typePrototype.ngDoCheck || null,
        afterContentInit: typePrototype.ngAfterContentInit || null,
        afterContentChecked: typePrototype.ngAfterContentChecked || null,
        afterViewInit: typePrototype.ngAfterViewInit || null,
        afterViewChecked: typePrototype.ngAfterViewChecked || null,
        onDestroy: typePrototype.ngOnDestroy || null,
        onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
        directiveDefs: (/** @type {?} */ (null)),
        // assigned in noSideEffects
        pipeDefs: (/** @type {?} */ (null)),
        // assigned in noSideEffects
        selectors: componentDefinition.selectors,
        viewQuery: componentDefinition.viewQuery || null,
        features: componentDefinition.features || null,
        data: componentDefinition.data || {},
        // TODO(misko): convert ViewEncapsulation into const enum so that it can be used directly in the
        // next line. Also `None` should be 0 not 2.
        encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
        id: 'c',
        styles: componentDefinition.styles || EMPTY_ARRAY,
        _: (/** @type {?} */ (null)),
    };
    def._ = (/** @type {?} */ (noSideEffects(() => {
        /** @type {?} */
        const directiveTypes = (/** @type {?} */ (componentDefinition.directives));
        /** @type {?} */
        const feature = componentDefinition.features;
        /** @type {?} */
        const pipeTypes = (/** @type {?} */ (componentDefinition.pipes));
        def.id += _renderCompCount++;
        def.inputs = invertObject(componentDefinition.inputs, declaredInputs),
            def.outputs = invertObject(componentDefinition.outputs),
            feature && feature.forEach((fn) => fn(def));
        def.directiveDefs = directiveTypes ?
            () => (typeof directiveTypes === 'function' ? directiveTypes() : directiveTypes)
                .map(extractDirectiveDef) :
            null;
        def.pipeDefs = pipeTypes ?
            () => (typeof pipeTypes === 'function' ? pipeTypes() : pipeTypes).map(extractPipeDef) :
            null;
    })));
    return (/** @type {?} */ (def));
}
/**
 * @param {?} type
 * @return {?}
 */
export function extractDirectiveDef(type) {
    /** @type {?} */
    const def = getComponentDef(type) || getDirectiveDef(type);
    if (ngDevMode && !def) {
        throw new Error(`'${type.name}' is neither 'ComponentType' or 'DirectiveType'.`);
    }
    return (/** @type {?} */ (def));
}
/**
 * @param {?} type
 * @return {?}
 */
export function extractPipeDef(type) {
    /** @type {?} */
    const def = getPipeDef(type);
    if (ngDevMode && !def) {
        throw new Error(`'${type.name}' is not a 'PipeType'.`);
    }
    return (/** @type {?} */ (def));
}
/**
 * @template T
 * @param {?} def
 * @return {?}
 */
export function defineNgModule(def) {
    /** @type {?} */
    const res = {
        type: def.type,
        bootstrap: def.bootstrap || EMPTY_ARRAY,
        declarations: def.declarations || EMPTY_ARRAY,
        imports: def.imports || EMPTY_ARRAY,
        exports: def.exports || EMPTY_ARRAY,
        transitiveCompileScopes: null,
    };
    return (/** @type {?} */ (res));
}
/**
 * Inverts an inputs or outputs lookup such that the keys, which were the
 * minified keys, are part of the values, and the values are parsed so that
 * the publicName of the property is the new key
 *
 * e.g. for
 *
 * ```
 * class Comp {
 * \@Input()
 *   propName1: string;
 *
 * \@Input('publicName2')
 *   declaredPropName2: number;
 * }
 * ```
 *
 * will be serialized as
 *
 * ```
 * {
 *   propName1: 'propName1',
 *   declaredPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * which is than translated by the minifier as:
 *
 * ```
 * {
 *   minifiedPropName1: 'propName1',
 *   minifiedPropName2: ['publicName2', 'declaredPropName2'],
 * }
 * ```
 *
 * becomes: (public name => minifiedName)
 *
 * ```
 * {
 *  'propName1': 'minifiedPropName1',
 *  'publicName2': 'minifiedPropName2',
 * }
 * ```
 *
 * Optionally the function can take `secondary` which will result in: (public name => declared name)
 *
 * ```
 * {
 *  'propName1': 'propName1',
 *  'publicName2': 'declaredPropName2',
 * }
 * ```
 *
 * @param {?} obj
 * @param {?=} secondary
 * @return {?}
 */
function invertObject(obj, secondary) {
    if (obj == null)
        return EMPTY_OBJ;
    /** @type {?} */
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            /** @type {?} */
            let publicName = obj[minifiedKey];
            /** @type {?} */
            let declaredName = publicName;
            if (Array.isArray(publicName)) {
                declaredName = publicName[1];
                publicName = publicName[0];
            }
            newLookup[publicName] = minifiedKey;
            if (secondary) {
                (secondary[publicName] = declaredName);
            }
        }
    }
    return newLookup;
}
/**
 * Create a base definition
 *
 * # Example
 * ```
 * class ShouldBeInherited {
 *   static ngBaseDef = defineBase({
 *      ...
 *   })
 * }
 * @template T
 * @param {?} baseDefinition The base definition parameters
 * @return {?}
 */
export function defineBase(baseDefinition) {
    /** @type {?} */
    const declaredInputs = (/** @type {?} */ ({}));
    return {
        inputs: invertObject(baseDefinition.inputs, declaredInputs),
        declaredInputs: declaredInputs,
        outputs: invertObject(baseDefinition.outputs),
    };
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ngDirectiveDef = defineDirective({
 *     ...
 *   });
 * }
 * ```
 * @type {?}
 */
export const defineDirective = (/** @type {?} */ ((/** @type {?} */ (defineComponent))));
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ngPipeDef = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @template T
 * @param {?} pipeDef Pipe definition generated by the compiler
 * @return {?}
 */
export function definePipe(pipeDef) {
    return (/** @type {?} */ (((/** @type {?} */ ({
        name: pipeDef.name,
        factory: pipeDef.factory,
        pure: pipeDef.pure !== false,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null
    })))));
}
/**
 * The following getter methods retrieve the definition form the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 * @template T
 * @param {?} type
 * @return {?}
 */
export function getComponentDef(type) {
    return ((/** @type {?} */ (type)))[NG_COMPONENT_DEF] || null;
}
/**
 * @template T
 * @param {?} type
 * @return {?}
 */
export function getDirectiveDef(type) {
    return ((/** @type {?} */ (type)))[NG_DIRECTIVE_DEF] || null;
}
/**
 * @template T
 * @param {?} type
 * @return {?}
 */
export function getPipeDef(type) {
    return ((/** @type {?} */ (type)))[NG_PIPE_DEF] || null;
}
/**
 * @template T
 * @param {?} type
 * @param {?=} throwNotFound
 * @return {?}
 */
export function getNgModuleDef(type, throwNotFound) {
    /** @type {?} */
    /** @nocollapse */ const ngModuleDef = ((/** @type {?} */ (type)))[NG_MODULE_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ngModuleDef' property.`);
    }
    return ngModuleDef;
}
//# sourceMappingURL=data:application/json;base64,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