"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullBranch = exports["default"] = void 0;
const DETACHED = Symbol('detached');
const REMOTE_TRACKING = Symbol('remote-tracking');

class Branch {
  constructor(name, upstream = nullBranch, push = upstream, head = false, attributes = {}) {
    this.name = name;
    this.upstream = upstream;
    this.push = push;
    this.head = head;
    this.attributes = attributes;
  }

  static createDetached(describe) {
    return new Branch(describe, nullBranch, nullBranch, true, {
      [DETACHED]: true
    });
  }

  static createRemoteTracking(refName, remoteName, remoteRef) {
    return new Branch(refName, nullBranch, nullBranch, false, {
      [REMOTE_TRACKING]: {
        remoteName,
        remoteRef
      }
    });
  }

  getName() {
    return this.name;
  }

  getShortRef() {
    return this.getName().replace(/^(refs\/)?((heads|remotes)\/)?/, '');
  }

  getFullRef() {
    if (this.isDetached()) {
      return '';
    }

    if (this.isRemoteTracking()) {
      if (this.name.startsWith('refs/')) {
        return this.name;
      } else if (this.name.startsWith('remotes/')) {
        return `refs/${this.name}`;
      }

      return `refs/remotes/${this.name}`;
    }

    if (this.name.startsWith('refs/')) {
      return this.name;
    } else if (this.name.startsWith('heads/')) {
      return `refs/${this.name}`;
    } else {
      return `refs/heads/${this.name}`;
    }
  }

  getRemoteName() {
    if (!this.isRemoteTracking()) {
      return '';
    }

    return this.attributes[REMOTE_TRACKING].remoteName || '';
  }

  getRemoteRef() {
    if (!this.isRemoteTracking()) {
      return '';
    }

    return this.attributes[REMOTE_TRACKING].remoteRef || '';
  }

  getShortRemoteRef() {
    return this.getRemoteRef().replace(/^(refs\/)?((heads|remotes)\/)?/, '');
  }

  getRefSpec(action) {
    if (this.isRemoteTracking()) {
      return '';
    }

    const remoteBranch = action === 'PUSH' ? this.push : this.upstream;
    const remoteBranchName = remoteBranch.getShortRemoteRef();
    const localBranchName = this.getName();

    if (remoteBranchName && remoteBranchName !== localBranchName) {
      if (action === 'PUSH') {
        return `${localBranchName}:${remoteBranchName}`;
      } else if (action === 'PULL') {
        return `${remoteBranchName}:${localBranchName}`;
      }
    }

    return localBranchName;
  }

  getSha() {
    return this.attributes.sha || '';
  }

  getUpstream() {
    return this.upstream;
  }

  getPush() {
    return this.push;
  }

  isHead() {
    return this.head;
  }

  isDetached() {
    return this.attributes[DETACHED] !== undefined;
  }

  isRemoteTracking() {
    return this.attributes[REMOTE_TRACKING] !== undefined;
  }

  isPresent() {
    return true;
  }

}

exports["default"] = Branch;
const nullBranch = {
  getName() {
    return '';
  },

  getShortRef() {
    return '';
  },

  getFullRef() {
    return '';
  },

  getSha() {
    return '';
  },

  getUpstream() {
    return this;
  },

  getPush() {
    return this;
  },

  isHead() {
    return false;
  },

  getRemoteName() {
    return '';
  },

  getRemoteRef() {
    return '';
  },

  getShortRemoteRef() {
    return '';
  },

  isDetached() {
    return false;
  },

  isRemoteTracking() {
    return false;
  },

  isPresent() {
    return false;
  },

  inspect(depth, options) {
    return '{nullBranch}';
  }

};
exports.nullBranch = nullBranch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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