"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const delegate_1 = require("./delegate");
const interface_1 = require("./interface");
class ScopedFileEntry {
    constructor(_base, scope) {
        this._base = _base;
        this.scope = scope;
    }
    get path() {
        return core_1.join(core_1.NormalizedRoot, core_1.relative(this.scope, this._base.path));
    }
    get content() { return this._base.content; }
}
class ScopedDirEntry {
    constructor(_base, scope) {
        this._base = _base;
        this.scope = scope;
    }
    get parent() {
        if (!this._base.parent || this._base.path == this.scope) {
            return null;
        }
        return new ScopedDirEntry(this._base.parent, this.scope);
    }
    get path() {
        return core_1.join(core_1.NormalizedRoot, core_1.relative(this.scope, this._base.path));
    }
    get subdirs() {
        return this._base.subdirs;
    }
    get subfiles() {
        return this._base.subfiles;
    }
    dir(name) {
        const entry = this._base.dir(name);
        return entry && new ScopedDirEntry(entry, this.scope);
    }
    file(name) {
        const entry = this._base.file(name);
        return entry && new ScopedFileEntry(entry, this.scope);
    }
    visit(visitor) {
        return this._base.visit((path, entry) => {
            visitor(core_1.join(core_1.NormalizedRoot, core_1.relative(this.scope, path)), entry && new ScopedFileEntry(entry, this.scope));
        });
    }
}
class ScopedTree {
    constructor(_base, scope) {
        this._base = _base;
        const normalizedScope = core_1.normalize('/' + scope);
        this._root = new ScopedDirEntry(this._base.getDir(normalizedScope), normalizedScope);
    }
    get root() { return this._root; }
    branch() { return new ScopedTree(this._base.branch(), this._root.scope); }
    merge(other, strategy) {
        const self = this;
        const delegate = new class extends delegate_1.DelegateTree {
            get actions() {
                return other.actions.map(action => self._fullPathAction(action));
            }
        }(other);
        this._base.merge(delegate, strategy);
    }
    // Readonly.
    read(path) { return this._base.read(this._fullPath(path)); }
    exists(path) { return this._base.exists(this._fullPath(path)); }
    get(path) {
        const entry = this._base.get(this._fullPath(path));
        return entry && new ScopedFileEntry(entry, this._root.scope);
    }
    getDir(path) {
        const entry = this._base.getDir(this._fullPath(path));
        return entry && new ScopedDirEntry(entry, this._root.scope);
    }
    visit(visitor) { return this._root.visit(visitor); }
    // Change content of host files.
    overwrite(path, content) {
        return this._base.overwrite(this._fullPath(path), content);
    }
    beginUpdate(path) {
        return this._base.beginUpdate(this._fullPath(path));
    }
    commitUpdate(record) { return this._base.commitUpdate(record); }
    // Structural methods.
    create(path, content) {
        return this._base.create(this._fullPath(path), content);
    }
    delete(path) { return this._base.delete(this._fullPath(path)); }
    rename(from, to) {
        return this._base.rename(this._fullPath(from), this._fullPath(to));
    }
    apply(action, strategy) {
        return this._base.apply(this._fullPathAction(action), strategy);
    }
    get actions() {
        const scopedActions = [];
        for (const action of this._base.actions) {
            if (!action.path.startsWith(this._root.scope + '/')) {
                continue;
            }
            if (action.kind !== 'r') {
                scopedActions.push(Object.assign({}, action, { path: core_1.join(core_1.NormalizedRoot, core_1.relative(this._root.scope, action.path)) }));
            }
            else if (action.to.startsWith(this._root.scope + '/')) {
                scopedActions.push(Object.assign({}, action, { path: core_1.join(core_1.NormalizedRoot, core_1.relative(this._root.scope, action.path)), to: core_1.join(core_1.NormalizedRoot, core_1.relative(this._root.scope, action.to)) }));
            }
        }
        return scopedActions;
    }
    [interface_1.TreeSymbol]() {
        return this;
    }
    _fullPath(path) {
        return core_1.join(this._root.scope, core_1.normalize('/' + path));
    }
    _fullPathAction(action) {
        let fullPathAction;
        if (action.kind === 'r') {
            fullPathAction = Object.assign({}, action, { path: this._fullPath(action.path), to: this._fullPath(action.to) });
        }
        else {
            fullPathAction = Object.assign({}, action, { path: this._fullPath(action.path) });
        }
        return fullPathAction;
    }
}
exports.ScopedTree = ScopedTree;
//# sourceMappingURL=data:application/json;base64,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