"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var _a;
var d3_array_1 = require("d3-array");
var d3_scale_1 = require("d3-scale");
var luxon_1 = require("luxon");
var commons_1 = require("../commons");
var scales_1 = require("./scales");
var SCALES = (_a = {},
    _a[scales_1.ScaleType.Linear] = d3_scale_1.scaleLinear,
    _a[scales_1.ScaleType.Log] = d3_scale_1.scaleLog,
    _a[scales_1.ScaleType.Sqrt] = d3_scale_1.scaleSqrt,
    _a[scales_1.ScaleType.Time] = d3_scale_1.scaleUtc,
    _a);
function limitToMin(value, positive) {
    if (value === 0) {
        return positive ? 1 : -1;
    }
    return value;
}
exports.limitToMin = limitToMin;
/**
 * As log(0) = -Infinite, a log scale domain must be strictly-positive
 * or strictly-negative; the domain must not include or cross zero value.
 * We need to limit the domain scale to the right value on all possible cases.
 * @param domain the domain to limit
 */
function limitLogScaleDomain(domain) {
    if (domain[0] === 0) {
        if (domain[1] > 0) {
            return [1, domain[1]];
        }
        else if (domain[1] < 0) {
            return [-1, domain[1]];
        }
        else {
            return [1, 1];
        }
    }
    if (domain[1] === 0) {
        if (domain[0] > 0) {
            return [domain[0], 1];
        }
        else if (domain[0] < 0) {
            return [domain[0], -1];
        }
        else {
            return [1, 1];
        }
    }
    if (domain[0] < 0 && domain[1] > 0) {
        var isD0Min = Math.abs(domain[1]) - Math.abs(domain[0]) >= 0;
        if (isD0Min) {
            return [1, domain[1]];
        }
        else {
            return [domain[0], -1];
        }
    }
    if (domain[0] > 0 && domain[1] < 0) {
        var isD0Max = Math.abs(domain[0]) - Math.abs(domain[1]) >= 0;
        if (isD0Max) {
            return [domain[0], 1];
        }
        else {
            return [-1, domain[1]];
        }
    }
    return domain;
}
exports.limitLogScaleDomain = limitLogScaleDomain;
var defaultScaleOptions = {
    bandwidth: 0,
    minInterval: 0,
    timeZone: 'utc',
    totalBarsInCluster: 1,
    barsPadding: 0,
    ticks: 10,
    isSingleValueHistogram: false,
};
var ScaleContinuous = /** @class */ (function () {
    function ScaleContinuous(scaleData, options) {
        var _this = this;
        var type = scaleData.type, domain = scaleData.domain, range = scaleData.range;
        var _a = commons_1.mergePartial(defaultScaleOptions, options), bandwidth = _a.bandwidth, minInterval = _a.minInterval, timeZone = _a.timeZone, totalBarsInCluster = _a.totalBarsInCluster, barsPadding = _a.barsPadding, ticks = _a.ticks, isSingleValueHistogram = _a.isSingleValueHistogram;
        this.d3Scale = SCALES[type]();
        if (type === scales_1.ScaleType.Log) {
            this.domain = limitLogScaleDomain(domain);
            this.d3Scale.domain(this.domain);
        }
        else {
            this.domain = domain;
            this.d3Scale.domain(domain);
        }
        var safeBarPadding = commons_1.clamp(barsPadding, 0, 1);
        this.barsPadding = safeBarPadding;
        this.bandwidth = bandwidth * (1 - safeBarPadding);
        this.bandwidthPadding = bandwidth * safeBarPadding;
        this.d3Scale.range(range);
        this.step = 0;
        this.type = type;
        this.range = range;
        this.minInterval = minInterval;
        this.isInverted = this.domain[0] > this.domain[1];
        this.timeZone = timeZone;
        this.totalBarsInCluster = totalBarsInCluster;
        this.isSingleValueHistogram = isSingleValueHistogram;
        if (type === scales_1.ScaleType.Time) {
            var startDomain = luxon_1.DateTime.fromMillis(this.domain[0], { zone: this.timeZone });
            var endDomain = luxon_1.DateTime.fromMillis(this.domain[1], { zone: this.timeZone });
            var offset_1 = startDomain.offset;
            var shiftedDomainMin = startDomain.plus({ minutes: offset_1 }).toMillis();
            var shiftedDomainMax = endDomain.plus({ minutes: offset_1 }).toMillis();
            var tzShiftedScale = d3_scale_1.scaleUtc().domain([shiftedDomainMin, shiftedDomainMax]);
            var rawTicks = tzShiftedScale.ticks(ticks);
            var timePerTick = (shiftedDomainMax - shiftedDomainMin) / rawTicks.length;
            var hasHourTicks_1 = timePerTick < 1000 * 60 * 60 * 12;
            this.tickValues = rawTicks.map(function (d) {
                var currentDateTime = luxon_1.DateTime.fromJSDate(d, { zone: _this.timeZone });
                var currentOffset = hasHourTicks_1 ? offset_1 : currentDateTime.offset;
                return currentDateTime.minus({ minutes: currentOffset }).toMillis();
            });
        }
        else {
            if (this.minInterval > 0) {
                var intervalCount = Math.floor((this.domain[1] - this.domain[0]) / this.minInterval);
                this.tickValues = new Array(intervalCount + 1).fill(0).map(function (d, i) {
                    return _this.domain[0] + i * _this.minInterval;
                });
            }
            else {
                this.tickValues = this.d3Scale.ticks(ticks);
            }
        }
    }
    ScaleContinuous.prototype.scale = function (value) {
        return this.d3Scale(value) + (this.bandwidthPadding / 2) * this.totalBarsInCluster;
    };
    ScaleContinuous.prototype.pureScale = function (value) {
        if (this.bandwidth === 0) {
            return this.d3Scale(value);
        }
        return this.d3Scale(value + this.minInterval / 2);
    };
    ScaleContinuous.prototype.ticks = function () {
        return this.tickValues;
    };
    ScaleContinuous.prototype.invert = function (value) {
        var invertedValue = this.d3Scale.invert(value);
        if (this.type === scales_1.ScaleType.Time) {
            invertedValue = luxon_1.DateTime.fromJSDate(invertedValue).toMillis();
        }
        return invertedValue;
    };
    ScaleContinuous.prototype.invertWithStep = function (value, data) {
        var invertedValue = this.invert(value);
        var bisectValue = this.bandwidth === 0 ? invertedValue + this.minInterval / 2 : invertedValue;
        var leftIndex = d3_array_1.bisectLeft(data, bisectValue);
        if (leftIndex === 0) {
            if (invertedValue < data[0]) {
                return {
                    value: data[0] - this.minInterval * Math.ceil((data[0] - invertedValue) / this.minInterval),
                    withinBandwidth: false,
                };
            }
            return {
                value: data[0],
                withinBandwidth: true,
            };
        }
        var currentValue = data[leftIndex - 1];
        // pure linear scale
        if (this.minInterval === 0) {
            var nextValue = data[leftIndex];
            var nextDiff = Math.abs(nextValue - invertedValue);
            var prevDiff = Math.abs(invertedValue - currentValue);
            return {
                value: nextDiff <= prevDiff ? nextValue : currentValue,
                withinBandwidth: true,
            };
        }
        if (invertedValue - currentValue <= this.minInterval) {
            return {
                value: currentValue,
                withinBandwidth: true,
            };
        }
        return {
            value: currentValue + this.minInterval * Math.floor((invertedValue - currentValue) / this.minInterval),
            withinBandwidth: false,
        };
    };
    ScaleContinuous.prototype.isSingleValue = function () {
        if (this.isSingleValueHistogram) {
            return true;
        }
        if (this.domain.length < 2) {
            return true;
        }
        var min = this.domain[0];
        var max = this.domain[this.domain.length - 1];
        return max === min;
    };
    return ScaleContinuous;
}());
exports.ScaleContinuous = ScaleContinuous;
function isContinuousScale(scale) {
    return scale.type !== scales_1.ScaleType.Ordinal;
}
exports.isContinuousScale = isContinuousScale;
function isLogarithmicScale(scale) {
    return scale.type === scales_1.ScaleType.Log;
}
exports.isLogarithmicScale = isLogarithmicScale;
//# sourceMappingURL=scale_continuous.js.map