/**
 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *
 */
'use strict';

Object.defineProperty(exports, '__esModule', {
  value: true
});
exports.default = void 0;

var _types = require('../types');

function _path() {
  const data = _interopRequireDefault(require('path'));

  _path = function _path() {
    return data;
  };

  return data;
}

function _interopRequireDefault(obj) {
  return obj && obj.__esModule ? obj : {default: obj};
}

function _objectSpread(target) {
  for (var i = 1; i < arguments.length; i++) {
    var source = arguments[i] != null ? arguments[i] : {};
    var ownKeys = Object.keys(source);
    if (typeof Object.getOwnPropertySymbols === 'function') {
      ownKeys = ownKeys.concat(
        Object.getOwnPropertySymbols(source).filter(function(sym) {
          return Object.getOwnPropertyDescriptor(source, sym).enumerable;
        })
      );
    }
    ownKeys.forEach(function(key) {
      _defineProperty(target, key, source[key]);
    });
  }
  return target;
}

function _defineProperty(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }
  return obj;
}

// $FlowFixMe: Flow doesn't know about experimental features of Node
const _require = require('worker_threads'),
  Worker = _require.Worker;

class ExperimentalWorker {
  constructor(options) {
    this._options = options;
    this.initialize();
  }

  initialize() {
    this._worker = new Worker(
      _path().default.resolve(__dirname, './threadChild.js'),
      {
        eval: false,
        stderr: true,
        stdout: true,
        workerData: _objectSpread(
          {
            cwd: process.cwd(),
            env: _objectSpread({}, process.env, {
              JEST_WORKER_ID: this._options.workerId
            }),
            // Suppress --debug / --inspect flags while preserving others (like --harmony).
            execArgv: process.execArgv.filter(
              v => !/^--(debug|inspect)/.test(v)
            ),
            silent: true
          },
          this._options.forkOptions
        )
      }
    );

    this._worker.on('message', this.onMessage.bind(this));

    this._worker.on('exit', this.onExit.bind(this));

    this._worker.postMessage([
      _types.CHILD_MESSAGE_INITIALIZE,
      false,
      this._options.workerPath,
      this._options.setupArgs
    ]);

    this._retries++; // If we exceeded the amount of retries, we will emulate an error reply
    // coming from the child. This avoids code duplication related with cleaning
    // the queue, and scheduling the next call.

    if (this._retries > this._options.maxRetries) {
      const error = new Error('Call retries were exceeded');
      this.onMessage([
        _types.PARENT_MESSAGE_CLIENT_ERROR,
        error.name,
        error.message,
        error.stack,
        {
          type: 'WorkerError'
        }
      ]);
    }
  }

  onMessage(
    response
    /* Should be ParentMessage */
  ) {
    let error;

    switch (response[0]) {
      case _types.PARENT_MESSAGE_OK:
        this._onProcessEnd(null, response[1]);

        break;

      case _types.PARENT_MESSAGE_CLIENT_ERROR:
        error = response[4];

        if (error != null && typeof error === 'object') {
          const extra = error;
          const NativeCtor = global[response[1]];
          const Ctor = typeof NativeCtor === 'function' ? NativeCtor : Error;
          error = new Ctor(response[2]); // $FlowFixMe: adding custom properties to errors.

          error.type = response[1];
          error.stack = response[3];

          for (const key in extra) {
            // $FlowFixMe: adding custom properties to errors.
            error[key] = extra[key];
          }
        }

        this._onProcessEnd(error, null);

        break;

      case _types.PARENT_MESSAGE_SETUP_ERROR:
        error = new Error('Error when calling setup: ' + response[2]); // $FlowFixMe: adding custom properties to errors.

        error.type = response[1];
        error.stack = response[3];

        this._onProcessEnd(error, null);

        break;

      default:
        throw new TypeError('Unexpected response from worker: ' + response[0]);
    }
  }

  onExit(exitCode) {
    if (exitCode !== 0) {
      this.initialize();
    }
  }

  send(request, onProcessStart, onProcessEnd) {
    onProcessStart(this);
    this._onProcessEnd = onProcessEnd;
    this._retries = 0;

    this._worker.postMessage(request);
  }

  getWorkerId() {
    return this._options.workerId;
  }

  getStdout() {
    return this._worker.stdout;
  }

  getStderr() {
    return this._worker.stderr;
  }
}

exports.default = ExperimentalWorker;
