"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const util_1 = require("../util");
const memory_1 = require("./memory");
const idb_1 = require("idb");
class IndexedDBDatabase {
    constructor() {
        this.dbPromise = idb_1.openDB("atom-telemetry-store", 1, {
            upgrade(db) {
                db.createObjectStore("counters", { keyPath: "name" });
                db.createObjectStore("customEvents", { autoIncrement: true });
                db.createObjectStore("timingEvents", { autoIncrement: true });
            },
        }).catch(() => {
            console.warn("Could not open IndexedDB database to store telemetry events. This session events won't be recorded.");
            return new memory_1.default();
        });
    }
    async addCustomEvent(eventType, customEvent) {
        const db = await this.dbPromise;
        if (db instanceof memory_1.default) {
            db.addCustomEvent(eventType, customEvent);
            return;
        }
        const eventToInsert = Object.assign({}, customEvent, { date: util_1.getISODate(), eventType });
        db.put("customEvents", eventToInsert);
    }
    async incrementCounter(counterName) {
        const db = await this.dbPromise;
        if (db instanceof memory_1.default) {
            db.incrementCounter(counterName);
            return;
        }
        const tx = db.transaction("counters", "readwrite");
        const entry = await tx.store.get(counterName);
        const currentValue = entry ? entry.value : 0;
        await tx.store.put({ name: counterName, value: currentValue + 1 });
        await tx.done;
    }
    async addTiming(eventType, durationInMilliseconds, metadata = {}) {
        const db = await this.dbPromise;
        if (db instanceof memory_1.default) {
            db.addTiming(eventType, durationInMilliseconds, metadata);
            return;
        }
        const timingData = {
            eventType,
            durationInMilliseconds,
            metadata,
            date: util_1.getISODate(),
        };
        db.put("timingEvents", timingData);
    }
    async clearData() {
        const db = await this.dbPromise;
        if (db instanceof memory_1.default) {
            db.clearData();
            return;
        }
        const tx = db.transaction(["counters", "customEvents", "timingEvents"], "readwrite");
        await tx.objectStore("counters").clear();
        await tx.objectStore("customEvents").clear();
        await tx.objectStore("timingEvents").clear();
        await tx.done;
    }
    async getTimings() {
        const db = await this.dbPromise;
        if (db instanceof memory_1.default) {
            return db.getTimings();
        }
        return db.getAll("timingEvents");
    }
    async getCustomEvents() {
        const db = await this.dbPromise;
        if (db instanceof memory_1.default) {
            return db.getCustomEvents();
        }
        return db.getAll("customEvents");
    }
    /**
     * Get all counters.
     * Returns something like { commits: 7, coAuthoredCommits: 8 }.
     */
    async getCounters() {
        const db = await this.dbPromise;
        if (db instanceof memory_1.default) {
            return db.getCounters();
        }
        const counters = Object.create(null);
        const entries = await db.getAll("counters");
        for (const { name, value } of entries) {
            counters[name] = value;
        }
        return counters;
    }
}
exports.default = IndexedDBDatabase;
//# sourceMappingURL=indexeddb.js.map