"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _eventKit = require("event-kit");

var _electron = require("electron");

var _atom = require("atom");

var _react = _interopRequireWildcard(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _path = _interopRequireDefault(require("path"));

var _propTypes2 = require("../prop-types");

var _filePatchListItemView = _interopRequireDefault(require("./file-patch-list-item-view"));

var _observeModel = _interopRequireDefault(require("./observe-model"));

var _mergeConflictListItemView = _interopRequireDefault(require("./merge-conflict-list-item-view"));

var _compositeListSelection = _interopRequireDefault(require("../models/composite-list-selection"));

var _resolutionProgress = _interopRequireDefault(require("../models/conflicts/resolution-progress"));

var _commitView = _interopRequireDefault(require("./commit-view"));

var _refHolder = _interopRequireDefault(require("../models/ref-holder"));

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _commands = _interopRequireWildcard(require("../atom/commands"));

var _helpers = require("../helpers");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const {
  Menu,
  MenuItem
} = _electron.remote;

const debounce = (fn, wait) => {
  let timeout;
  return (...args) => {
    return new Promise(resolve => {
      clearTimeout(timeout);
      timeout = setTimeout(() => {
        resolve(fn(...args));
      }, wait);
    });
  };
};

function calculateTruncatedLists(lists) {
  return Object.keys(lists).reduce((acc, key) => {
    const list = lists[key];
    acc.source[key] = list;

    if (list.length <= MAXIMUM_LISTED_ENTRIES) {
      acc[key] = list;
    } else {
      acc[key] = list.slice(0, MAXIMUM_LISTED_ENTRIES);
    }

    return acc;
  }, {
    source: {}
  });
}

const noop = () => {};

const MAXIMUM_LISTED_ENTRIES = 1000;

class StagingView extends _react["default"].Component {
  constructor(props) {
    super(props);

    _defineProperty(this, "undoLastDiscardFromCoreUndo", () => {
      this.undoLastDiscard({
        eventSource: {
          command: 'core:undo'
        }
      });
    });

    _defineProperty(this, "undoLastDiscardFromCommand", () => {
      this.undoLastDiscard({
        eventSource: {
          command: 'github:undo-last-discard-in-git-tab'
        }
      });
    });

    _defineProperty(this, "undoLastDiscardFromButton", () => {
      this.undoLastDiscard({
        eventSource: 'button'
      });
    });

    _defineProperty(this, "undoLastDiscardFromHeaderMenu", () => {
      this.undoLastDiscard({
        eventSource: 'header-menu'
      });
    });

    _defineProperty(this, "discardChangesFromCommand", () => {
      this.discardChanges({
        eventSource: {
          command: 'github:discard-changes-in-selected-files'
        }
      });
    });

    _defineProperty(this, "discardAllFromCommand", () => {
      this.discardAll({
        eventSource: {
          command: 'github:discard-all-changes'
        }
      });
    });

    _defineProperty(this, "confirmSelectedItems", async () => {
      const itemPaths = this.getSelectedItemFilePaths();
      await this.props.attemptFileStageOperation(itemPaths, this.state.selection.getActiveListKey());
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.coalesce()
        }), resolve);
      });
    });

    (0, _helpers.autobind)(this, 'dblclickOnItem', 'contextMenuOnItem', 'mousedownOnItem', 'mousemoveOnItem', 'mouseup', 'registerItemElement', 'renderBody', 'openFile', 'discardChanges', 'activateNextList', 'activatePreviousList', 'activateLastList', 'stageAll', 'unstageAll', 'stageAllMergeConflicts', 'discardAll', 'confirmSelectedItems', 'selectAll', 'selectFirst', 'selectLast', 'diveIntoSelection', 'showDiffView', 'showBulkResolveMenu', 'showActionsMenu', 'resolveCurrentAsOurs', 'resolveCurrentAsTheirs', 'quietlySelectItem', 'didChangeSelectedItems');
    this.subs = new _eventKit.CompositeDisposable(atom.config.observe('github.keyboardNavigationDelay', value => {
      if (value === 0) {
        this.debouncedDidChangeSelectedItem = this.didChangeSelectedItems;
      } else {
        this.debouncedDidChangeSelectedItem = debounce(this.didChangeSelectedItems, value);
      }
    }));
    this.state = _objectSpread({}, calculateTruncatedLists({
      unstagedChanges: this.props.unstagedChanges,
      stagedChanges: this.props.stagedChanges,
      mergeConflicts: this.props.mergeConflicts
    }), {
      selection: new _compositeListSelection["default"]({
        listsByKey: [['unstaged', this.props.unstagedChanges], ['conflicts', this.props.mergeConflicts], ['staged', this.props.stagedChanges]],
        idForItem: item => item.filePath
      })
    });
    this.mouseSelectionInProgress = false;
    this.listElementsByItem = new WeakMap();
    this.refRoot = new _refHolder["default"]();
  }

  static getDerivedStateFromProps(nextProps, prevState) {
    let nextState = {};

    if (['unstagedChanges', 'stagedChanges', 'mergeConflicts'].some(key => prevState.source[key] !== nextProps[key])) {
      const nextLists = calculateTruncatedLists({
        unstagedChanges: nextProps.unstagedChanges,
        stagedChanges: nextProps.stagedChanges,
        mergeConflicts: nextProps.mergeConflicts
      });
      nextState = _objectSpread({}, nextLists, {
        selection: prevState.selection.updateLists([['unstaged', nextLists.unstagedChanges], ['conflicts', nextLists.mergeConflicts], ['staged', nextLists.stagedChanges]])
      });
    }

    return nextState;
  }

  componentDidMount() {
    window.addEventListener('mouseup', this.mouseup);
    this.subs.add(new _eventKit.Disposable(() => window.removeEventListener('mouseup', this.mouseup)), this.props.workspace.onDidChangeActivePaneItem(() => {
      this.syncWithWorkspace();
    }));

    if (this.isPopulated(this.props)) {
      this.syncWithWorkspace();
    }
  }

  componentDidUpdate(prevProps, prevState) {
    const isRepoSame = prevProps.workingDirectoryPath === this.props.workingDirectoryPath;
    const hasSelectionsPresent = prevState.selection.getSelectedItems().size > 0 && this.state.selection.getSelectedItems().size > 0;
    const selectionChanged = this.state.selection !== prevState.selection;

    if (isRepoSame && hasSelectionsPresent && selectionChanged) {
      this.debouncedDidChangeSelectedItem();
    }

    const headItem = this.state.selection.getHeadItem();

    if (headItem) {
      const element = this.listElementsByItem.get(headItem);

      if (element) {
        element.scrollIntoViewIfNeeded();
      }
    }

    if (!this.isPopulated(prevProps) && this.isPopulated(this.props)) {
      this.syncWithWorkspace();
    }
  }

  render() {
    return _react["default"].createElement(_observeModel["default"], {
      model: this.props.resolutionProgress,
      fetchData: noop
    }, this.renderBody);
  }

  renderBody() {
    const selectedItems = this.state.selection.getSelectedItems();
    return _react["default"].createElement("div", {
      ref: this.refRoot.setter,
      className: `github-StagingView ${this.state.selection.getActiveListKey()}-changes-focused`,
      tabIndex: "-1"
    }, this.renderCommands(), _react["default"].createElement("div", {
      className: `github-StagingView-group github-UnstagedChanges ${this.getFocusClass('unstaged')}`
    }, _react["default"].createElement("header", {
      className: "github-StagingView-header"
    }, _react["default"].createElement("span", {
      className: "icon icon-list-unordered"
    }), _react["default"].createElement("span", {
      className: "github-StagingView-title"
    }, "Unstaged Changes"), this.renderActionsMenu(), _react["default"].createElement("button", {
      className: "github-StagingView-headerButton icon icon-move-down",
      disabled: this.props.unstagedChanges.length === 0,
      onClick: this.stageAll
    }, "Stage All")), _react["default"].createElement("div", {
      className: "github-StagingView-list github-FilePatchListView github-StagingView-unstaged"
    }, this.state.unstagedChanges.map(filePatch => _react["default"].createElement(_filePatchListItemView["default"], {
      key: filePatch.filePath,
      registerItemElement: this.registerItemElement,
      filePatch: filePatch,
      onDoubleClick: event => this.dblclickOnItem(event, filePatch),
      onContextMenu: event => this.contextMenuOnItem(event, filePatch),
      onMouseDown: event => this.mousedownOnItem(event, filePatch),
      onMouseMove: event => this.mousemoveOnItem(event, filePatch),
      selected: selectedItems.has(filePatch)
    }))), this.renderTruncatedMessage(this.props.unstagedChanges)), this.renderMergeConflicts(), _react["default"].createElement("div", {
      className: `github-StagingView-group github-StagedChanges ${this.getFocusClass('staged')}`
    }, _react["default"].createElement("header", {
      className: "github-StagingView-header"
    }, _react["default"].createElement("span", {
      className: "icon icon-tasklist"
    }), _react["default"].createElement("span", {
      className: "github-StagingView-title"
    }, "Staged Changes"), _react["default"].createElement("button", {
      className: "github-StagingView-headerButton icon icon-move-up",
      disabled: this.props.stagedChanges.length === 0,
      onClick: this.unstageAll
    }, "Unstage All")), _react["default"].createElement("div", {
      className: "github-StagingView-list github-FilePatchListView github-StagingView-staged"
    }, this.state.stagedChanges.map(filePatch => _react["default"].createElement(_filePatchListItemView["default"], {
      key: filePatch.filePath,
      filePatch: filePatch,
      registerItemElement: this.registerItemElement,
      onDoubleClick: event => this.dblclickOnItem(event, filePatch),
      onContextMenu: event => this.contextMenuOnItem(event, filePatch),
      onMouseDown: event => this.mousedownOnItem(event, filePatch),
      onMouseMove: event => this.mousemoveOnItem(event, filePatch),
      selected: selectedItems.has(filePatch)
    }))), this.renderTruncatedMessage(this.props.stagedChanges)));
  }

  renderCommands() {
    return _react["default"].createElement(_react.Fragment, null, _react["default"].createElement(_commands["default"], {
      registry: this.props.commandRegistry,
      target: ".github-StagingView"
    }, _react["default"].createElement(_commands.Command, {
      command: "core:move-up",
      callback: () => this.selectPrevious()
    }), _react["default"].createElement(_commands.Command, {
      command: "core:move-down",
      callback: () => this.selectNext()
    }), _react["default"].createElement(_commands.Command, {
      command: "core:move-left",
      callback: this.diveIntoSelection
    }), _react["default"].createElement(_commands.Command, {
      command: "github:show-diff-view",
      callback: this.showDiffView
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-up",
      callback: () => this.selectPrevious(true)
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-down",
      callback: () => this.selectNext(true)
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-all",
      callback: this.selectAll
    }), _react["default"].createElement(_commands.Command, {
      command: "core:move-to-top",
      callback: this.selectFirst
    }), _react["default"].createElement(_commands.Command, {
      command: "core:move-to-bottom",
      callback: this.selectLast
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-to-top",
      callback: () => this.selectFirst(true)
    }), _react["default"].createElement(_commands.Command, {
      command: "core:select-to-bottom",
      callback: () => this.selectLast(true)
    }), _react["default"].createElement(_commands.Command, {
      command: "core:confirm",
      callback: this.confirmSelectedItems
    }), _react["default"].createElement(_commands.Command, {
      command: "github:activate-next-list",
      callback: this.activateNextList
    }), _react["default"].createElement(_commands.Command, {
      command: "github:activate-previous-list",
      callback: this.activatePreviousList
    }), _react["default"].createElement(_commands.Command, {
      command: "github:jump-to-file",
      callback: this.openFile
    }), _react["default"].createElement(_commands.Command, {
      command: "github:resolve-file-as-ours",
      callback: this.resolveCurrentAsOurs
    }), _react["default"].createElement(_commands.Command, {
      command: "github:resolve-file-as-theirs",
      callback: this.resolveCurrentAsTheirs
    }), _react["default"].createElement(_commands.Command, {
      command: "github:discard-changes-in-selected-files",
      callback: this.discardChangesFromCommand
    }), _react["default"].createElement(_commands.Command, {
      command: "core:undo",
      callback: this.undoLastDiscardFromCoreUndo
    })), _react["default"].createElement(_commands["default"], {
      registry: this.props.commandRegistry,
      target: "atom-workspace"
    }, _react["default"].createElement(_commands.Command, {
      command: "github:stage-all-changes",
      callback: this.stageAll
    }), _react["default"].createElement(_commands.Command, {
      command: "github:unstage-all-changes",
      callback: this.unstageAll
    }), _react["default"].createElement(_commands.Command, {
      command: "github:discard-all-changes",
      callback: this.discardAllFromCommand
    }), _react["default"].createElement(_commands.Command, {
      command: "github:undo-last-discard-in-git-tab",
      callback: this.undoLastDiscardFromCommand
    })));
  }

  renderActionsMenu() {
    if (this.props.unstagedChanges.length || this.props.hasUndoHistory) {
      return _react["default"].createElement("button", {
        className: "github-StagingView-headerButton github-StagingView-headerButton--iconOnly icon icon-ellipses",
        onClick: this.showActionsMenu
      });
    } else {
      return null;
    }
  }

  renderUndoButton() {
    return _react["default"].createElement("button", {
      className: "github-StagingView-headerButton github-StagingView-headerButton--fullWidth icon icon-history",
      onClick: this.undoLastDiscardFromButton
    }, "Undo Discard");
  }

  renderTruncatedMessage(list) {
    if (list.length > MAXIMUM_LISTED_ENTRIES) {
      return _react["default"].createElement("div", {
        className: "github-StagingView-group-truncatedMsg"
      }, "List truncated to the first ", MAXIMUM_LISTED_ENTRIES, " items");
    } else {
      return null;
    }
  }

  renderMergeConflicts() {
    const mergeConflicts = this.state.mergeConflicts;

    if (mergeConflicts && mergeConflicts.length > 0) {
      const selectedItems = this.state.selection.getSelectedItems();
      const resolutionProgress = this.props.resolutionProgress;
      const anyUnresolved = mergeConflicts.map(conflict => _path["default"].join(this.props.workingDirectoryPath, conflict.filePath)).some(conflictPath => resolutionProgress.getRemaining(conflictPath) !== 0);
      const bulkResolveDropdown = anyUnresolved ? _react["default"].createElement("span", {
        className: "inline-block icon icon-ellipses",
        onClick: this.showBulkResolveMenu
      }) : null;
      return _react["default"].createElement("div", {
        className: `github-StagingView-group github-MergeConflictPaths ${this.getFocusClass('conflicts')}`
      }, _react["default"].createElement("header", {
        className: "github-StagingView-header"
      }, _react["default"].createElement("span", {
        className: 'github-FilePatchListView-icon icon icon-alert status-modified'
      }), _react["default"].createElement("span", {
        className: "github-StagingView-title"
      }, "Merge Conflicts"), bulkResolveDropdown, _react["default"].createElement("button", {
        className: "github-StagingView-headerButton icon icon-move-down",
        disabled: anyUnresolved,
        onClick: this.stageAllMergeConflicts
      }, "Stage All")), _react["default"].createElement("div", {
        className: "github-StagingView-list github-FilePatchListView github-StagingView-merge"
      }, mergeConflicts.map(mergeConflict => {
        const fullPath = _path["default"].join(this.props.workingDirectoryPath, mergeConflict.filePath);

        return _react["default"].createElement(_mergeConflictListItemView["default"], {
          key: fullPath,
          mergeConflict: mergeConflict,
          remainingConflicts: resolutionProgress.getRemaining(fullPath),
          registerItemElement: this.registerItemElement,
          onDoubleClick: event => this.dblclickOnItem(event, mergeConflict),
          onContextMenu: event => this.contextMenuOnItem(event, mergeConflict),
          onMouseDown: event => this.mousedownOnItem(event, mergeConflict),
          onMouseMove: event => this.mousemoveOnItem(event, mergeConflict),
          selected: selectedItems.has(mergeConflict)
        });
      })), this.renderTruncatedMessage(mergeConflicts));
    } else {
      return _react["default"].createElement("noscript", null);
    }
  }

  componentWillUnmount() {
    this.subs.dispose();
  }

  getSelectedItemFilePaths() {
    return Array.from(this.state.selection.getSelectedItems(), item => item.filePath);
  }

  getSelectedConflictPaths() {
    if (this.state.selection.getActiveListKey() !== 'conflicts') {
      return [];
    }

    return this.getSelectedItemFilePaths();
  }

  openFile() {
    const filePaths = this.getSelectedItemFilePaths();
    return this.props.openFiles(filePaths);
  }

  discardChanges({
    eventSource
  } = {}) {
    const filePaths = this.getSelectedItemFilePaths();
    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      "package": 'github',
      component: 'StagingView',
      fileCount: filePaths.length,
      type: 'selected',
      eventSource
    });
    return this.props.discardWorkDirChangesForPaths(filePaths);
  }

  activateNextList() {
    return new Promise(resolve => {
      let advanced = false;
      this.setState(prevState => {
        const next = prevState.selection.activateNextSelection();

        if (prevState.selection === next) {
          return {};
        }

        advanced = true;
        return {
          selection: next.coalesce()
        };
      }, () => resolve(advanced));
    });
  }

  activatePreviousList() {
    return new Promise(resolve => {
      let retreated = false;
      this.setState(prevState => {
        const next = prevState.selection.activatePreviousSelection();

        if (prevState.selection === next) {
          return {};
        }

        retreated = true;
        return {
          selection: next.coalesce()
        };
      }, () => resolve(retreated));
    });
  }

  activateLastList() {
    return new Promise(resolve => {
      let emptySelection = false;
      this.setState(prevState => {
        const next = prevState.selection.activateLastSelection();
        emptySelection = next.getSelectedItems().size > 0;

        if (prevState.selection === next) {
          return {};
        }

        return {
          selection: next.coalesce()
        };
      }, () => resolve(emptySelection));
    });
  }

  stageAll() {
    if (this.props.unstagedChanges.length === 0) {
      return null;
    }

    return this.props.attemptStageAllOperation('unstaged');
  }

  unstageAll() {
    if (this.props.stagedChanges.length === 0) {
      return null;
    }

    return this.props.attemptStageAllOperation('staged');
  }

  stageAllMergeConflicts() {
    if (this.props.mergeConflicts.length === 0) {
      return null;
    }

    const filePaths = this.props.mergeConflicts.map(conflict => conflict.filePath);
    return this.props.attemptFileStageOperation(filePaths, 'unstaged');
  }

  discardAll({
    eventSource
  } = {}) {
    if (this.props.unstagedChanges.length === 0) {
      return null;
    }

    const filePaths = this.props.unstagedChanges.map(filePatch => filePatch.filePath);
    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      "package": 'github',
      component: 'StagingView',
      fileCount: filePaths.length,
      type: 'all',
      eventSource
    });
    return this.props.discardWorkDirChangesForPaths(filePaths);
  }

  getNextListUpdatePromise() {
    return this.state.selection.getNextUpdatePromise();
  }

  selectPrevious(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectPreviousItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectNext(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectNextItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectAll() {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectAllItems().coalesce()
      }), resolve);
    });
  }

  selectFirst(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectFirstItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  selectLast(preserveTail = false) {
    return new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.selectLastItem(preserveTail).coalesce()
      }), resolve);
    });
  }

  async diveIntoSelection() {
    const selectedItems = this.state.selection.getSelectedItems();

    if (selectedItems.size !== 1) {
      return;
    }

    const selectedItem = selectedItems.values().next().value;
    const stagingStatus = this.state.selection.getActiveListKey();

    if (stagingStatus === 'conflicts') {
      this.showMergeConflictFileForPath(selectedItem.filePath, {
        activate: true
      });
    } else {
      await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
        activate: true
      });
    }
  }

  async syncWithWorkspace() {
    const item = this.props.workspace.getActivePaneItem();

    if (!item) {
      return;
    }

    const realItemPromise = item.getRealItemPromise && item.getRealItemPromise();
    const realItem = await realItemPromise;

    if (!realItem) {
      return;
    }

    const isFilePatchItem = realItem.isFilePatchItem && realItem.isFilePatchItem();
    const isMatch = realItem.getWorkingDirectory && realItem.getWorkingDirectory() === this.props.workingDirectoryPath;

    if (isFilePatchItem && isMatch) {
      this.quietlySelectItem(realItem.getFilePath(), realItem.getStagingStatus());
    }
  }

  async showDiffView() {
    const selectedItems = this.state.selection.getSelectedItems();

    if (selectedItems.size !== 1) {
      return;
    }

    const selectedItem = selectedItems.values().next().value;
    const stagingStatus = this.state.selection.getActiveListKey();

    if (stagingStatus === 'conflicts') {
      this.showMergeConflictFileForPath(selectedItem.filePath);
    } else {
      await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey());
    }
  }

  showBulkResolveMenu(event) {
    const conflictPaths = this.props.mergeConflicts.map(c => c.filePath);
    event.preventDefault();
    const menu = new Menu();
    menu.append(new MenuItem({
      label: 'Resolve All as Ours',
      click: () => this.props.resolveAsOurs(conflictPaths)
    }));
    menu.append(new MenuItem({
      label: 'Resolve All as Theirs',
      click: () => this.props.resolveAsTheirs(conflictPaths)
    }));
    menu.popup(_electron.remote.getCurrentWindow());
  }

  showActionsMenu(event) {
    event.preventDefault();
    const menu = new Menu();
    const selectedItemCount = this.state.selection.getSelectedItems().size;
    const pluralization = selectedItemCount > 1 ? 's' : '';
    menu.append(new MenuItem({
      label: 'Discard All Changes',
      click: () => this.discardAll({
        eventSource: 'header-menu'
      }),
      enabled: this.props.unstagedChanges.length > 0
    }));
    menu.append(new MenuItem({
      label: 'Discard Changes in Selected File' + pluralization,
      click: () => this.discardChanges({
        eventSource: 'header-menu'
      }),
      enabled: !!(this.props.unstagedChanges.length && selectedItemCount)
    }));
    menu.append(new MenuItem({
      label: 'Undo Last Discard',
      click: () => this.undoLastDiscard({
        eventSource: 'header-menu'
      }),
      enabled: this.props.hasUndoHistory
    }));
    menu.popup(_electron.remote.getCurrentWindow());
  }

  resolveCurrentAsOurs() {
    this.props.resolveAsOurs(this.getSelectedConflictPaths());
  }

  resolveCurrentAsTheirs() {
    this.props.resolveAsTheirs(this.getSelectedConflictPaths());
  } // Directly modify the selection to include only the item identified by the file path and stagingStatus tuple.
  // Re-render the component, but don't notify didSelectSingleItem() or other callback functions. This is useful to
  // avoid circular callback loops for actions originating in FilePatchView or TextEditors with merge conflicts.


  quietlySelectItem(filePath, stagingStatus) {
    return new Promise(resolve => {
      this.setState(prevState => {
        const item = prevState.selection.findItem((each, key) => each.filePath === filePath && key === stagingStatus);

        if (!item) {
          // FIXME: make staging view display no selected item
          // eslint-disable-next-line no-console
          console.log(`Unable to find item at path ${filePath} with staging status ${stagingStatus}`);
          return null;
        }

        return {
          selection: prevState.selection.selectItem(item)
        };
      }, resolve);
    });
  }

  getSelectedItems() {
    const stagingStatus = this.state.selection.getActiveListKey();
    return Array.from(this.state.selection.getSelectedItems(), item => {
      return {
        filePath: item.filePath,
        stagingStatus
      };
    });
  }

  didChangeSelectedItems(openNew) {
    const selectedItems = Array.from(this.state.selection.getSelectedItems());

    if (selectedItems.length === 1) {
      this.didSelectSingleItem(selectedItems[0], openNew);
    }
  }

  async didSelectSingleItem(selectedItem, openNew = false) {
    if (!this.hasFocus()) {
      return;
    }

    if (this.state.selection.getActiveListKey() === 'conflicts') {
      if (openNew) {
        await this.showMergeConflictFileForPath(selectedItem.filePath, {
          activate: true
        });
      }
    } else {
      if (openNew) {
        // User explicitly asked to view diff, such as via click
        await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
          activate: false
        });
      } else {
        const panesWithStaleItemsToUpdate = this.getPanesWithStalePendingFilePatchItem();

        if (panesWithStaleItemsToUpdate.length > 0) {
          // Update stale items to reflect new selection
          await Promise.all(panesWithStaleItemsToUpdate.map(async pane => {
            await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
              activate: false,
              pane
            });
          }));
        } else {
          // Selection was changed via keyboard navigation, update pending item in active pane
          const activePane = this.props.workspace.getCenter().getActivePane();
          const activePendingItem = activePane.getPendingItem();

          const activePaneHasPendingFilePatchItem = activePendingItem && activePendingItem.getRealItem && activePendingItem.getRealItem() instanceof _changedFileItem["default"];

          if (activePaneHasPendingFilePatchItem) {
            await this.showFilePatchItem(selectedItem.filePath, this.state.selection.getActiveListKey(), {
              activate: false,
              pane: activePane
            });
          }
        }
      }
    }
  }

  getPanesWithStalePendingFilePatchItem() {
    // "stale" meaning there is no longer a changed file associated with item
    // due to changes being fully staged/unstaged/stashed/deleted/etc
    return this.props.workspace.getPanes().filter(pane => {
      const pendingItem = pane.getPendingItem();

      if (!pendingItem || !pendingItem.getRealItem) {
        return false;
      }

      const realItem = pendingItem.getRealItem();

      if (!(realItem instanceof _changedFileItem["default"])) {
        return false;
      } // We only want to update pending diff views for currently active repo


      const isInActiveRepo = realItem.getWorkingDirectory() === this.props.workingDirectoryPath;
      const isStale = !this.changedFileExists(realItem.getFilePath(), realItem.getStagingStatus());
      return isInActiveRepo && isStale;
    });
  }

  changedFileExists(filePath, stagingStatus) {
    return this.state.selection.findItem((item, key) => {
      return key === stagingStatus && item.filePath === filePath;
    });
  }

  async showFilePatchItem(filePath, stagingStatus, {
    activate,
    pane
  } = {
    activate: false
  }) {
    const uri = _changedFileItem["default"].buildURI(filePath, this.props.workingDirectoryPath, stagingStatus);

    const changedFileItem = await this.props.workspace.open(uri, {
      pending: true,
      activatePane: activate,
      activateItem: activate,
      pane
    });

    if (activate) {
      const itemRoot = changedFileItem.getElement();
      const focusRoot = itemRoot.querySelector('[tabIndex]');

      if (focusRoot) {
        focusRoot.focus();
      }
    } else {
      // simply make item visible
      this.props.workspace.paneForItem(changedFileItem).activateItem(changedFileItem);
    }
  }

  async showMergeConflictFileForPath(relativeFilePath, {
    activate
  } = {
    activate: false
  }) {
    const absolutePath = _path["default"].join(this.props.workingDirectoryPath, relativeFilePath);

    if (await this.fileExists(absolutePath)) {
      return this.props.workspace.open(absolutePath, {
        activatePane: activate,
        activateItem: activate,
        pending: true
      });
    } else {
      this.props.notificationManager.addInfo('File has been deleted.');
      return null;
    }
  }

  fileExists(absolutePath) {
    return new _atom.File(absolutePath).exists();
  }

  dblclickOnItem(event, item) {
    return this.props.attemptFileStageOperation([item.filePath], this.state.selection.listKeyForItem(item));
  }

  async contextMenuOnItem(event, item) {
    if (!this.state.selection.getSelectedItems().has(item)) {
      event.stopPropagation();
      event.persist();
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.selectItem(item, event.shiftKey)
        }), resolve);
      });
      const newEvent = new MouseEvent(event.type, event);
      requestAnimationFrame(() => {
        if (!event.target.parentNode) {
          return;
        }

        event.target.parentNode.dispatchEvent(newEvent);
      });
    }
  }

  async mousedownOnItem(event, item) {
    const windows = process.platform === 'win32';

    if (event.ctrlKey && !windows) {
      return;
    } // simply open context menu


    if (event.button === 0) {
      this.mouseSelectionInProgress = true;
      event.persist();
      await new Promise(resolve => {
        if (event.metaKey || event.ctrlKey && windows) {
          this.setState(prevState => ({
            selection: prevState.selection.addOrSubtractSelection(item)
          }), resolve);
        } else {
          this.setState(prevState => ({
            selection: prevState.selection.selectItem(item, event.shiftKey)
          }), resolve);
        }
      });
    }
  }

  async mousemoveOnItem(event, item) {
    if (this.mouseSelectionInProgress) {
      await new Promise(resolve => {
        this.setState(prevState => ({
          selection: prevState.selection.selectItem(item, true)
        }), resolve);
      });
    }
  }

  async mouseup() {
    const hadSelectionInProgress = this.mouseSelectionInProgress;
    this.mouseSelectionInProgress = false;
    await new Promise(resolve => {
      this.setState(prevState => ({
        selection: prevState.selection.coalesce()
      }), resolve);
    });

    if (hadSelectionInProgress) {
      this.didChangeSelectedItems(true);
    }
  }

  undoLastDiscard({
    eventSource
  } = {}) {
    if (!this.props.hasUndoHistory) {
      return;
    }

    (0, _reporterProxy.addEvent)('undo-last-discard', {
      "package": 'github',
      component: 'StagingView',
      eventSource
    });
    this.props.undoLastDiscard();
  }

  getFocusClass(listKey) {
    return this.state.selection.getActiveListKey() === listKey ? 'is-focused' : '';
  }

  registerItemElement(item, element) {
    this.listElementsByItem.set(item, element);
  }

  getFocus(element) {
    return this.refRoot.map(root => root.contains(element)).getOr(false) ? StagingView.focus.STAGING : null;
  }

  setFocus(focus) {
    if (focus === this.constructor.focus.STAGING) {
      this.refRoot.map(root => root.focus());
      return true;
    }

    return false;
  }

  async advanceFocusFrom(focus) {
    if (focus === this.constructor.focus.STAGING) {
      if (await this.activateNextList()) {
        // There was a next list to activate.
        return this.constructor.focus.STAGING;
      } // We were already on the last list.


      return _commitView["default"].firstFocus;
    }

    return null;
  }

  async retreatFocusFrom(focus) {
    if (focus === _commitView["default"].firstFocus) {
      await this.activateLastList();
      return this.constructor.focus.STAGING;
    }

    if (focus === this.constructor.focus.STAGING) {
      await this.activatePreviousList();
      return this.constructor.focus.STAGING;
    }

    return false;
  }

  hasFocus() {
    return this.refRoot.map(root => root.contains(document.activeElement)).getOr(false);
  }

  isPopulated(props) {
    return props.workingDirectoryPath != null && (props.unstagedChanges.length > 0 || props.mergeConflicts.length > 0 || props.stagedChanges.length > 0);
  }

}

exports["default"] = StagingView;

_defineProperty(StagingView, "propTypes", {
  unstagedChanges: _propTypes["default"].arrayOf(_propTypes2.FilePatchItemPropType).isRequired,
  stagedChanges: _propTypes["default"].arrayOf(_propTypes2.FilePatchItemPropType).isRequired,
  mergeConflicts: _propTypes["default"].arrayOf(_propTypes2.MergeConflictItemPropType),
  workingDirectoryPath: _propTypes["default"].string,
  resolutionProgress: _propTypes["default"].object,
  hasUndoHistory: _propTypes["default"].bool.isRequired,
  commandRegistry: _propTypes["default"].object.isRequired,
  notificationManager: _propTypes["default"].object.isRequired,
  workspace: _propTypes["default"].object.isRequired,
  openFiles: _propTypes["default"].func.isRequired,
  attemptFileStageOperation: _propTypes["default"].func.isRequired,
  discardWorkDirChangesForPaths: _propTypes["default"].func.isRequired,
  undoLastDiscard: _propTypes["default"].func.isRequired,
  attemptStageAllOperation: _propTypes["default"].func.isRequired,
  resolveAsOurs: _propTypes["default"].func.isRequired,
  resolveAsTheirs: _propTypes["default"].func.isRequired
});

_defineProperty(StagingView, "defaultProps", {
  mergeConflicts: [],
  resolutionProgress: new _resolutionProgress["default"]()
});

_defineProperty(StagingView, "focus", {
  STAGING: Symbol('staging')
});

_defineProperty(StagingView, "firstFocus", StagingView.focus.STAGING);

_defineProperty(StagingView, "lastFocus", StagingView.focus.STAGING);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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