"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _compareSets = _interopRequireDefault(require("compare-sets"));

var _eventKit = require("event-kit");

var _workdirContext = _interopRequireDefault(require("./workdir-context"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Manage a WorkdirContext for each open directory.
 */
class WorkdirContextPool {
  /**
   * Options will be passed to each `WorkdirContext` as it is created.
   */
  constructor(options = {}) {
    this.options = options;
    this.contexts = new Map();
    this.emitter = new _eventKit.Emitter();
  }

  size() {
    return this.contexts.size;
  }
  /**
   * Access the context mapped to a known directory.
   */


  getContext(directory) {
    const {
      pipelineManager
    } = this.options;
    return this.contexts.get(directory) || _workdirContext["default"].absent({
      pipelineManager
    });
  }
  /**
   * Return a WorkdirContext whose Repository has at least one remote configured to push to the named GitHub repository.
   * Returns a null context if zero or more than one contexts match.
   */


  async getMatchingContext(host, owner, repo) {
    const matches = await Promise.all(this.withResidentContexts(async (_workdir, context) => {
      const match = await context.getRepository().hasGitHubRemote(host, owner, repo);
      return match ? context : null;
    }));
    const filtered = matches.filter(Boolean);
    return filtered.length === 1 ? filtered[0] : _workdirContext["default"].absent(_objectSpread({}, this.options));
  }

  add(directory, options = {}) {
    if (this.contexts.has(directory)) {
      return this.getContext(directory);
    }

    const context = new _workdirContext["default"](directory, _objectSpread({}, this.options, options));
    this.contexts.set(directory, context);
    const disposable = new _eventKit.CompositeDisposable();

    const forwardEvent = (subMethod, emitEventName) => {
      const emit = () => this.emitter.emit(emitEventName, context);

      disposable.add(context[subMethod](emit));
    };

    forwardEvent('onDidStartObserver', 'did-start-observer');
    forwardEvent('onDidChangeWorkdirOrHead', 'did-change-workdir-or-head');
    forwardEvent('onDidChangeRepositoryState', 'did-change-repository-state');
    forwardEvent('onDidUpdateRepository', 'did-update-repository');
    forwardEvent('onDidDestroyRepository', 'did-destroy-repository');
    disposable.add(this.onDidRemoveContext(removed => {
      if (removed === context) {
        disposable.dispose();
      }
    }));
    return context;
  }

  replace(directory, options = {}) {
    this.remove(directory);
    this.add(directory, options);
  }

  remove(directory) {
    const existing = this.contexts.get(directory);
    this.contexts["delete"](directory);

    if (existing) {
      this.emitter.emit('did-remove-context', existing);
      existing.destroy();
    }
  }

  set(directories, options = {}) {
    const previous = new Set(this.contexts.keys());
    const {
      added,
      removed
    } = (0, _compareSets["default"])(previous, directories);

    for (const directory of added) {
      this.add(directory, options);
    }

    for (const directory of removed) {
      this.remove(directory);
    }
  }

  withResidentContexts(callback) {
    const results = [];

    for (const [workdir, context] of this.contexts) {
      results.push(callback(workdir, context));
    }

    return results;
  }

  onDidStartObserver(callback) {
    return this.emitter.on('did-start-observer', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  onDidRemoveContext(callback) {
    return this.emitter.on('did-remove-context', callback);
  }

  onDidChangeRepositoryState(callback) {
    return this.emitter.on('did-change-repository-state', callback);
  }

  onDidUpdateRepository(callback) {
    return this.emitter.on('did-update-repository', callback);
  }

  onDidDestroyRepository(callback) {
    return this.emitter.on('did-destroy-repository', callback);
  }

  clear() {
    this.withResidentContexts(workdir => this.remove(workdir));

    _workdirContext["default"].destroyAbsent();
  }

}

exports["default"] = WorkdirContextPool;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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