/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/**
 * Type of the Inject decorator / constructor function.
 *
 * @publicApi
 */
export interface InjectDecorator {
    /**
     * A parameter decorator on a dependency parameter of a class constructor
     * that specifies a custom provider of the dependency.
     *
     * Learn more in the ["Dependency Injection Guide"](guide/dependency-injection).
     *
     * @usageNotes
     * The following example shows a class constructor that specifies a
     * custom provider of a dependency using the parameter decorator.
     *
     * {@example core/di/ts/metadata_spec.ts region='Inject'}
     *
     * When `@Inject()` is not present, the injector uses the type annotation of the
     * parameter as the provider.
     *
     * {@example core/di/ts/metadata_spec.ts region='InjectWithoutDecorator'}
     */
    (token: any): any;
    new (token: any): Inject;
}
/**
 * Type of the Inject metadata.
 *
 * @publicApi
 */
export interface Inject {
    /**
     * A [DI token](guide/glossary#di-token) that maps to the dependency to be injected.
     */
    token: any;
}
/**
 * Inject decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Inject: InjectDecorator;
/**
 * Type of the Optional decorator / constructor function.
 *
 * @publicApi
 */
export interface OptionalDecorator {
    /**
     * A parameter decorator to be used on constructor parameters,
     * which marks the parameter as being an optional dependency.
     * The DI framework provides null if the dependency is not found.
     *
     * Can be used together with other parameter decorators
     * that modify how dependency injection operates.
     *
     * Learn more in the ["Dependency Injection Guide"](guide/dependency-injection).
     *
     * @usageNotes
     *
     * The following code allows the possibility of a null result:
     *
     * {@example core/di/ts/metadata_spec.ts region='Optional'}
     *
     */
    (): any;
    new (): Optional;
}
/**
 * Type of the Optional metadata.
 *
 * @publicApi
 */
export interface Optional {
}
/**
 * Optional decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Optional: OptionalDecorator;
/**
 * Type of the Self decorator / constructor function.
 *
 * @publicApi
 */
export interface SelfDecorator {
    /**
     * A parameter decorator to be used on constructor parameters,
     * which tells the DI framework to start dependency resolution from the local injector.
     *
     * Resolution works upward through the injector hierarchy, so the children
     * of this class must configure their own providers or be prepared for a null result.
     *
     * @usageNotes
     *
     * In the following example, the dependency can be resolved
     * by the local injector when instantiating the class itself, but not
     * when instantiating a child.
     *
     * {@example core/di/ts/metadata_spec.ts region='Self'}
     *
     * @see `SkipSelf`
     * @see `Optional`
     *
     */
    (): any;
    new (): Self;
}
/**
 * Type of the Self metadata.
 *
 * @publicApi
 */
export interface Self {
}
/**
 * Self decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Self: SelfDecorator;
/**
 * Type of the SkipSelf decorator / constructor function.
 *
 * @publicApi
 */
export interface SkipSelfDecorator {
    /**
     * A parameter decorator to be used on constructor parameters,
     * which tells the DI framework to start dependency resolution from the parent injector.
     * Resolution works upward through the injector hierarchy, so the local injector
     * is not checked for a provider.
     *
     * @usageNotes
     *
     * In the following example, the dependency can be resolved when
     * instantiating a child, but not when instantiating the class itself.
     *
     * {@example core/di/ts/metadata_spec.ts region='SkipSelf'}
     *
     * Learn more in the
     * [Dependency Injection guide](guide/dependency-injection-in-action#skip).
     *
     * @see `Self`
     * @see `Optional`
     *
     */
    (): any;
    new (): SkipSelf;
}
/**
 * Type of the SkipSelf metadata.
 *
 * @publicApi
 */
export interface SkipSelf {
}
/**
 * SkipSelf decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const SkipSelf: SkipSelfDecorator;
/**
 * Type of the Host decorator / constructor function.
 *
 * @publicApi
 */
export interface HostDecorator {
    /**
     * A parameter decorator on a view-provider parameter of a class constructor
     * that tells the DI framework to resolve the view by checking injectors of child
     * elements, and stop when reaching the host element of the current component.
     *
     * For an extended example, see
     * ["Dependency Injection Guide"](guide/dependency-injection-in-action#optional).
     *
     * @usageNotes
     *
     * The following shows use with the `@Optional` decorator, and allows for a null result.
     *
     * {@example core/di/ts/metadata_spec.ts region='Host'}
     */
    (): any;
    new (): Host;
}
/**
 * Type of the Host metadata.
 *
 * @publicApi
 */
export interface Host {
}
/**
 * Host decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export declare const Host: HostDecorator;
