/*
 * Decompiled with CFR 0.152.
 */
package org.jgroups.protocols;

import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.jgroups.Address;
import org.jgroups.Event;
import org.jgroups.Header;
import org.jgroups.Message;
import org.jgroups.View;
import org.jgroups.annotations.MBean;
import org.jgroups.annotations.ManagedAttribute;
import org.jgroups.annotations.ManagedOperation;
import org.jgroups.annotations.Property;
import org.jgroups.protocols.FcHeader;
import org.jgroups.stack.Protocol;
import org.jgroups.util.Bits;
import org.jgroups.util.Credit;
import org.jgroups.util.MessageBatch;
import org.jgroups.util.Util;

@MBean(description="Simple flow control protocol based on a credit system")
public abstract class FlowControl
extends Protocol {
    @Property(description="Max number of bytes to send per receiver until an ack must be received to proceed")
    protected long max_credits = 500000L;
    @Property(description="Max time (in ms) to block")
    protected long max_block_time = 500L;
    protected Map<Long, Long> max_block_times;
    @Property(description="The threshold (as a percentage of max_credits) at which a receiver sends more credits to a sender. Example: if max_credits is 1'000'000, and min_threshold 0.25, then we send ca. 250'000 credits to P once we've got only 250'000 credits left for P (we've received 750'000 bytes from P)")
    protected double min_threshold = 0.4;
    @Property(description="Computed as max_credits x min_theshold unless explicitly set")
    protected long min_credits;
    protected int num_credit_requests_received;
    protected int num_credit_requests_sent;
    protected int num_credit_responses_sent;
    protected int num_credit_responses_received;
    protected final Map<Address, Credit> received = Util.createConcurrentMap();
    protected Address local_addr;
    protected volatile boolean running = true;
    protected int frag_size;

    @Override
    public void resetStats() {
        super.resetStats();
        this.num_credit_requests_sent = 0;
        this.num_credit_requests_received = 0;
        this.num_credit_responses_received = 0;
        this.num_credit_responses_sent = 0;
    }

    public long getMaxCredits() {
        return this.max_credits;
    }

    public void setMaxCredits(long max_credits) {
        this.max_credits = max_credits;
    }

    public double getMinThreshold() {
        return this.min_threshold;
    }

    public void setMinThreshold(double min_threshold) {
        this.min_threshold = min_threshold;
    }

    public long getMinCredits() {
        return this.min_credits;
    }

    public void setMinCredits(long min_credits) {
        this.min_credits = min_credits;
    }

    public long getMaxBlockTime() {
        return this.max_block_time;
    }

    public void setMaxBlockTime(long t) {
        this.max_block_time = t;
    }

    @Property(description="Max times to block for the listed messages sizes (Message.getLength()). Example: \"1000:10,5000:30,10000:500\"")
    public void setMaxBlockTimes(String str) {
        if (str == null) {
            return;
        }
        Long prev_key = null;
        Long prev_val = null;
        List<String> vals = Util.parseCommaDelimitedStrings(str);
        if (this.max_block_times == null) {
            this.max_block_times = new TreeMap<Long, Long>();
        }
        for (String tmp : vals) {
            int index = tmp.indexOf(58);
            if (index == -1) {
                throw new IllegalArgumentException("element '" + tmp + "'  is missing a ':' separator");
            }
            Long key = Long.parseLong(tmp.substring(0, index).trim());
            Long val = Long.parseLong(tmp.substring(index + 1).trim());
            if (key < 0L || val < 0L) {
                throw new IllegalArgumentException("keys and values must be >= 0");
            }
            if (prev_key != null && key <= prev_key) {
                throw new IllegalArgumentException("keys are not sorted: " + vals);
            }
            prev_key = key;
            if (prev_val != null && val <= prev_val) {
                throw new IllegalArgumentException("values are not sorted: " + vals);
            }
            prev_val = val;
            this.max_block_times.put(key, val);
        }
        this.log.debug("max_block_times: %s", this.max_block_times);
    }

    public String getMaxBlockTimes() {
        if (this.max_block_times == null) {
            return "n/a";
        }
        StringBuilder sb = new StringBuilder();
        boolean first = true;
        for (Map.Entry<Long, Long> entry : this.max_block_times.entrySet()) {
            if (!first) {
                sb.append(", ");
            } else {
                first = false;
            }
            sb.append(entry.getKey()).append(":").append(entry.getValue());
        }
        return sb.toString();
    }

    public abstract int getNumberOfBlockings();

    public abstract double getAverageTimeBlocked();

    @ManagedAttribute(description="Number of credit requests received")
    public int getNumberOfCreditRequestsReceived() {
        return this.num_credit_requests_received;
    }

    @ManagedAttribute(description="Number of credit requests sent")
    public int getNumberOfCreditRequestsSent() {
        return this.num_credit_requests_sent;
    }

    @ManagedAttribute(description="Number of credit responses received")
    public int getNumberOfCreditResponsesReceived() {
        return this.num_credit_responses_received;
    }

    @ManagedAttribute(description="Number of credit responses sent")
    public int getNumberOfCreditResponsesSent() {
        return this.num_credit_responses_sent;
    }

    public abstract String printSenderCredits();

    @ManagedOperation(description="Print receiver credits")
    public String printReceiverCredits() {
        return FlowControl.printMap(this.received);
    }

    public String printCredits() {
        return String.format("receivers:\n%s", FlowControl.printMap(this.received));
    }

    protected long getMaxBlockTime(long length) {
        if (this.max_block_times == null) {
            return 0L;
        }
        Long retval = null;
        for (Map.Entry<Long, Long> entry : this.max_block_times.entrySet()) {
            retval = entry.getValue();
            if (length > entry.getKey()) continue;
            break;
        }
        return retval != null ? retval : 0L;
    }

    protected abstract boolean handleMulticastMessage();

    protected abstract void handleCredit(Address var1, long var2);

    protected abstract Header getReplenishHeader();

    protected abstract Header getCreditRequestHeader();

    @ManagedOperation(description="Unblocks all senders")
    public void unblock() {
    }

    @Override
    public void init() throws Exception {
        boolean min_credits_set;
        boolean bl = min_credits_set = this.min_credits != 0L;
        if (!min_credits_set) {
            this.min_credits = (long)((double)this.max_credits * this.min_threshold);
        }
    }

    @Override
    public void start() throws Exception {
        super.start();
        boolean is_udp_transport = this.getTransport().isMulticastCapable();
        if (is_udp_transport && this.frag_size <= 0) {
            this.log.warn("No fragmentation protocol was found. When flow control is used, we recommend a fragmentation protocol, due to http://jira.jboss.com/jira/browse/JGRP-590");
        }
        if (this.frag_size > 0 && (long)this.frag_size >= this.min_credits) {
            this.log.warn("The fragmentation size of the fragmentation protocol is %d, which is greater than min_credits (%d). This can lead to blockings (https://issues.jboss.org/browse/JGRP-1659)", this.frag_size, this.min_credits);
        }
        this.running = true;
    }

    @Override
    public void stop() {
        super.stop();
        this.running = false;
    }

    @Override
    public Object down(Event evt) {
        switch (evt.getType()) {
            case 56: {
                this.handleConfigEvent((Map)evt.getArg());
                break;
            }
            case 6: {
                this.handleViewChange(((View)evt.getArg()).getMembers());
                break;
            }
            case 8: {
                this.local_addr = (Address)evt.getArg();
            }
        }
        return this.down_prot.down(evt);
    }

    @Override
    public Object down(Message msg) {
        long new_credits;
        boolean process;
        if (msg.isFlagSet(Message.Flag.NO_FC)) {
            return this.down_prot.down(msg);
        }
        Address dest = msg.getDest();
        boolean multicast = dest == null;
        boolean handle_multicasts = this.handleMulticastMessage();
        boolean bl = process = handle_multicasts && multicast || !handle_multicasts && !multicast;
        if (!process) {
            return this.down_prot.down(msg);
        }
        int length = msg.getLength();
        if (length == 0) {
            return this.down_prot.down(msg);
        }
        Object retval = this.handleDownMessage(msg);
        if (msg.isTransientFlagSet(Message.TransientFlag.DONT_LOOPBACK) && (new_credits = this.adjustCredit(this.received, this.local_addr, length)) > 0L) {
            this.sendCredit(this.local_addr, new_credits);
        }
        return retval;
    }

    @Override
    public Object up(Event evt) {
        switch (evt.getType()) {
            case 6: {
                this.handleViewChange(((View)evt.getArg()).getMembers());
            }
        }
        return this.up_prot.up(evt);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Object up(Message msg) {
        boolean process;
        if (msg.isFlagSet(Message.Flag.NO_FC)) {
            return this.up_prot.up(msg);
        }
        Address dest = msg.getDest();
        boolean multicast = dest == null;
        boolean handle_multicasts = this.handleMulticastMessage();
        FcHeader hdr = (FcHeader)msg.getHeader(this.id);
        boolean bl = process = handle_multicasts && multicast || !handle_multicasts && !multicast || hdr != null;
        if (!process) {
            return this.up_prot.up(msg);
        }
        if (hdr != null) {
            this.handleUpEvent(msg, hdr);
            return null;
        }
        try {
            Object object = this.up_prot.up(msg);
            return object;
        }
        finally {
            Address sender;
            long new_credits;
            int length = msg.getLength();
            if (length > 0 && (new_credits = this.adjustCredit(this.received, sender = msg.getSrc(), length)) > 0L) {
                this.sendCredit(sender, new_credits);
            }
        }
    }

    protected void handleUpEvent(Message msg, FcHeader hdr) {
        switch (hdr.type) {
            case 1: {
                ++this.num_credit_responses_received;
                this.handleCredit(msg.getSrc(), FlowControl.bufferToLong(msg.getRawBuffer(), msg.getOffset()));
                break;
            }
            case 2: {
                ++this.num_credit_requests_received;
                Address sender = msg.getSrc();
                Long requested_credits = FlowControl.bufferToLong(msg.getRawBuffer(), msg.getOffset());
                if (requested_credits == null) break;
                this.handleCreditRequest(this.received, sender, requested_credits);
                break;
            }
            default: {
                this.log.error(Util.getMessage("HeaderTypeNotKnown"), this.local_addr, hdr.type);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void up(MessageBatch batch) {
        int length = 0;
        for (Message msg : batch) {
            if (msg.isFlagSet(Message.Flag.NO_FC)) continue;
            Address dest = msg.getDest();
            boolean multicast = dest == null;
            boolean handle_multicasts = this.handleMulticastMessage();
            FcHeader hdr = (FcHeader)msg.getHeader(this.id);
            boolean process = handle_multicasts && multicast || !handle_multicasts && !multicast || hdr != null;
            if (!process) continue;
            if (hdr != null) {
                batch.remove(msg);
                this.handleUpEvent(msg, hdr);
                continue;
            }
            length += msg.getLength();
        }
        if (!batch.isEmpty()) {
            try {
                this.up_prot.up(batch);
            }
            finally {
                Address sender;
                long new_credits;
                if (length > 0 && (new_credits = this.adjustCredit(this.received, sender = batch.sender(), length)) > 0L) {
                    this.sendCredit(sender, new_credits);
                }
            }
        }
    }

    protected void handleConfigEvent(Map<String, Object> info) {
        Integer tmp;
        if (info != null && (tmp = (Integer)info.get("frag_size")) != null) {
            this.frag_size = tmp;
        }
    }

    protected abstract Object handleDownMessage(Message var1);

    protected long adjustCredit(Map<Address, Credit> map, Address sender, int length) {
        Credit cred;
        if (sender == null || length == 0 || (cred = map.get(sender)) == null) {
            return 0L;
        }
        if (this.log.isTraceEnabled()) {
            this.log.trace("%s used %d credits, %d remaining", sender, length, cred.get() - (long)length);
        }
        return cred.decrementAndGet(length, this.min_credits, this.max_credits);
    }

    protected void handleCreditRequest(Map<Address, Credit> map, Address sender, long requested_credits) {
        if (requested_credits > 0L && sender != null) {
            Credit cred = map.get(sender);
            if (cred == null) {
                return;
            }
            if (this.log.isTraceEnabled()) {
                this.log.trace("received credit request from %s: sending %d credits", sender, requested_credits);
            }
            cred.increment(requested_credits, this.max_credits);
            this.sendCredit(sender, requested_credits);
        }
    }

    protected void sendCredit(Address dest, long credits) {
        if (this.log.isTraceEnabled()) {
            this.log.trace("sending %d credits to %s", credits, dest);
        }
        Message msg = new Message(dest, FlowControl.longToBuffer(credits)).setFlag(Message.Flag.OOB, Message.Flag.INTERNAL, Message.Flag.DONT_BUNDLE).putHeader(this.id, this.getReplenishHeader());
        this.down_prot.down(msg);
        ++this.num_credit_responses_sent;
    }

    protected void sendCreditRequest(Address dest, long credits_needed) {
        if (this.log.isTraceEnabled()) {
            this.log.trace("sending request for %d credits to %s", credits_needed, dest);
        }
        Message msg = new Message(dest, FlowControl.longToBuffer(credits_needed)).setFlag(Message.Flag.OOB, Message.Flag.INTERNAL, Message.Flag.DONT_BUNDLE).putHeader(this.id, this.getCreditRequestHeader());
        this.down_prot.down(msg);
        ++this.num_credit_requests_sent;
    }

    protected void handleViewChange(List<Address> mbrs) {
        if (mbrs == null) {
            return;
        }
        if (this.log.isTraceEnabled()) {
            this.log.trace("new membership: %s", mbrs);
        }
        mbrs.stream().filter(addr -> !this.received.containsKey(addr)).forEach(addr -> this.received.put((Address)addr, new Credit(this.max_credits)));
        this.received.keySet().retainAll(mbrs);
    }

    protected static String printMap(Map<Address, ? extends Credit> m) {
        return m.entrySet().stream().collect(StringBuilder::new, (sb, entry) -> sb.append(entry.getKey()).append(": ").append(entry.getValue()).append("\n"), (l, r) -> {}).toString();
    }

    protected static byte[] longToBuffer(long num) {
        byte[] buf = new byte[8];
        Bits.writeLong(num, buf, 0);
        return buf;
    }

    protected static long bufferToLong(byte[] buf, int offset) {
        return Bits.readLong(buf, offset);
    }
}

