import { fromPredicate as eitherFromPredicate, getApplySemigroup as eitherGetApplySemigroup, getSemigroup as eitherGetSemigroup, left as eitherLeft, right as eitherRight } from './Either';
import * as eitherT from './EitherT';
import { constant, constIdentity, identity } from './function';
import { fromIO as taskFromIO, getSemigroup as taskGetSemigroup, Task, task, tryCatch as taskTryCatch } from './Task';
export const URI = 'TaskEither';
const T = eitherT.getEitherT2v(task);
const foldT = eitherT.fold(task);
/**
 * @since 1.0.0
 */
export class TaskEither {
    constructor(value) {
        this.value = value;
    }
    /** Runs the inner `Task` */
    run() {
        return this.value.run();
    }
    map(f) {
        return new TaskEither(T.map(this.value, f));
    }
    ap(fab) {
        return new TaskEither(T.ap(fab.value, this.value));
    }
    /**
     * Flipped version of `ap`
     */
    ap_(fb) {
        return fb.ap(this);
    }
    /**
     * Combine two (parallel) effectful actions, keeping only the result of the first
     * @since 1.6.0
     */
    applyFirst(fb) {
        return fb.ap(this.map(constant));
    }
    /**
     * Combine two (parallel) effectful actions, keeping only the result of the second
     * @since 1.5.0
     */
    applySecond(fb) {
        return fb.ap(this.map(constIdentity));
    }
    /**
     * Combine two (sequential) effectful actions, keeping only the result of the first
     * @since 1.12.0
     */
    chainFirst(fb) {
        return this.chain(a => fb.map(() => a));
    }
    /**
     * Combine two (sequential) effectful actions, keeping only the result of the second
     * @since 1.12.0
     */
    chainSecond(fb) {
        return this.chain(() => fb);
    }
    chain(f) {
        return new TaskEither(T.chain(this.value, a => f(a).value));
    }
    fold(onLeft, onRight) {
        return foldT(onLeft, onRight, this.value);
    }
    /**
     * Similar to `fold`, but the result is flattened.
     * @since 1.10.0
     */
    foldTask(onLeft, onRight) {
        return this.value.chain(e => e.fold(onLeft, onRight));
    }
    /**
     * Similar to `fold`, but the result is flattened.
     * @since 1.10.0
     */
    foldTaskEither(onLeft, onRight) {
        return new TaskEither(this.value.chain(e => e.fold(onLeft, onRight).value));
    }
    /**
     * Similar to `fold`, return the value from Right or the given argument if Left.
     * @since 1.17.0
     */
    getOrElse(a) {
        return this.getOrElseL(() => a);
    }
    /**
     * @since 1.17.0
     */
    getOrElseL(f) {
        return this.fold(f, identity);
    }
    mapLeft(f) {
        return new TaskEither(this.value.map(e => e.mapLeft(f)));
    }
    /**
     * Transforms the failure value of the `TaskEither` into a new `TaskEither`
     */
    orElse(f) {
        return new TaskEither(this.value.chain(e => e.fold(l => f(l).value, T.of)));
    }
    /**
     * @since 1.6.0
     */
    alt(fy) {
        return this.orElse(() => fy);
    }
    /**
     * @since 1.2.0
     */
    bimap(f, g) {
        return new TaskEither(this.value.map(e => e.bimap(f, g)));
    }
    /**
     * Return `Right` if the given action succeeds, `Left` if it throws
     * @since 1.10.0
     */
    attempt() {
        return new TaskEither(this.value.map(eitherRight));
    }
    filterOrElse(p, zero) {
        return new TaskEither(this.value.map(e => e.filterOrElse(p, zero)));
    }
    filterOrElseL(p, zero) {
        return new TaskEither(this.value.map(e => e.filterOrElseL(p, zero)));
    }
}
const map = (fa, f) => {
    return fa.map(f);
};
const of = (a) => {
    return new TaskEither(T.of(a));
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const chain = (fa, f) => {
    return fa.chain(f);
};
const alt = (fx, fy) => {
    return fx.alt(fy);
};
const bimap = (fa, f, g) => {
    return fa.bimap(f, g);
};
/**
 * @since 1.0.0
 */
export const right = (fa) => {
    return new TaskEither(fa.map(eitherRight));
};
/**
 * @since 1.0.0
 */
export const left = (fl) => {
    return new TaskEither(fl.map(eitherLeft));
};
/**
 * @since 1.0.0
 */
export const fromEither = (fa) => {
    return new TaskEither(task.of(fa));
};
/**
 * @since 1.5.0
 */
export const fromIO = (fa) => {
    return right(taskFromIO(fa));
};
/**
 * @since 1.3.0
 */
export const fromLeft = (l) => {
    return fromEither(eitherLeft(l));
};
/**
 * @since 1.6.0
 */
export const fromIOEither = (fa) => {
    return new TaskEither(taskFromIO(fa.value));
};
export function fromPredicate(predicate, onFalse) {
    const f = eitherFromPredicate(predicate, onFalse);
    return a => fromEither(f(a));
}
/**
 * @since 1.9.0
 */
export const getSemigroup = (S) => {
    const S2 = taskGetSemigroup(eitherGetSemigroup(S));
    return {
        concat: (x, y) => new TaskEither(S2.concat(x.value, y.value))
    };
};
/**
 * @since 1.9.0
 */
export const getApplySemigroup = (S) => {
    const S2 = taskGetSemigroup(eitherGetApplySemigroup(S));
    return {
        concat: (x, y) => new TaskEither(S2.concat(x.value, y.value))
    };
};
/**
 * @since 1.9.0
 */
export const getApplyMonoid = (M) => {
    return Object.assign({}, getApplySemigroup(M), { empty: of(M.empty) });
};
/**
 * Transforms a `Promise` into a `TaskEither`, catching the possible error.
 *
 * @example
 * import { createHash } from 'crypto'
 * import { TaskEither, tryCatch } from 'fp-ts/lib/TaskEither'
 * import { createReadStream } from 'fs'
 * import { left } from 'fp-ts/lib/Either'
 *
 * const md5 = (path: string): TaskEither<string, string> => {
 *   const mkHash = (p: string) =>
 *     new Promise<string>((resolve, reject) => {
 *       const hash = createHash('md5')
 *       const rs = createReadStream(p)
 *       rs.on('error', (error: Error) => reject(error.message))
 *       rs.on('data', (chunk: string) => hash.update(chunk))
 *       rs.on('end', () => {
 *         return resolve(hash.digest('hex'))
 *       })
 *     })
 *   return tryCatch(() => mkHash(path), message => `cannot create md5 hash: ${String(message)}`)
 * }
 *
 * md5('foo')
 *   .run()
 *   .then(x => {
 *     assert.deepStrictEqual(x, left(`cannot create md5 hash: ENOENT: no such file or directory, open 'foo'`))
 *   })
 *
 *
 * @since 1.0.0
 */
export const tryCatch = (f, onrejected) => {
    return new TaskEither(taskTryCatch(f, onrejected));
};
export function taskify(f) {
    return function () {
        const args = Array.prototype.slice.call(arguments);
        return new TaskEither(new Task(() => new Promise(resolve => {
            const cbResolver = (e, r) => e != null ? resolve(eitherLeft(e)) : resolve(eitherRight(r));
            f.apply(null, args.concat(cbResolver));
        })));
    };
}
const fromTask = right;
/**
 * Make sure that a resource is cleaned up in the event of an exception. The
 * release action is called regardless of whether the body action throws or
 * returns.
 *
 * @since 1.10.0
 */
export const bracket = (acquire, use, release) => {
    return acquire.chain(a => use(a)
        .attempt()
        .chain(e => release(a, e).chain(() => e.fold(fromLeft, taskEither.of))));
};
const throwError = fromLeft;
/**
 * @since 1.0.0
 */
export const taskEither = {
    URI,
    bimap,
    map,
    of,
    ap,
    chain,
    alt,
    fromIO,
    fromTask,
    throwError,
    fromEither,
    fromOption: (o, e) => (o.isNone() ? throwError(e) : of(o.value))
};
/**
 * Like `TaskEither` but `ap` is sequential
 *
 * @since 1.10.0
 */
export const taskEitherSeq = Object.assign({}, taskEither, { ap: (fab, fa) => fab.chain(f => fa.map(f)) });
