(function() {
  var RootDragAndDropHandler, ipcRenderer, ref, remote, url,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  url = require('url');

  ref = require('electron'), ipcRenderer = ref.ipcRenderer, remote = ref.remote;

  module.exports = RootDragAndDropHandler = (function() {
    function RootDragAndDropHandler(treeView) {
      this.treeView = treeView;
      this.onDrop = bind(this.onDrop, this);
      this.onDropOnOtherWindow = bind(this.onDropOnOtherWindow, this);
      this.onDragOver = bind(this.onDragOver, this);
      this.onDragEnd = bind(this.onDragEnd, this);
      this.onDragLeave = bind(this.onDragLeave, this);
      this.onDragStart = bind(this.onDragStart, this);
      ipcRenderer.on('tree-view:project-folder-dropped', this.onDropOnOtherWindow);
      this.handleEvents();
    }

    RootDragAndDropHandler.prototype.dispose = function() {
      return ipcRenderer.removeListener('tree-view:project-folder-dropped', this.onDropOnOtherWindow);
    };

    RootDragAndDropHandler.prototype.handleEvents = function() {
      this.treeView.element.addEventListener('dragenter', this.onDragEnter.bind(this));
      this.treeView.element.addEventListener('dragend', this.onDragEnd.bind(this));
      this.treeView.element.addEventListener('dragleave', this.onDragLeave.bind(this));
      this.treeView.element.addEventListener('dragover', this.onDragOver.bind(this));
      return this.treeView.element.addEventListener('drop', this.onDrop.bind(this));
    };

    RootDragAndDropHandler.prototype.onDragStart = function(e) {
      var directory, i, index, len, pathUri, projectRoot, ref1, ref2, root, rootIndex;
      if (!this.treeView.list.contains(e.target)) {
        return;
      }
      this.prevDropTargetIndex = null;
      e.dataTransfer.setData('atom-tree-view-root-event', 'true');
      projectRoot = e.target.closest('.project-root');
      directory = projectRoot.directory;
      e.dataTransfer.setData('project-root-index', Array.from(projectRoot.parentElement.children).indexOf(projectRoot));
      rootIndex = -1;
      ref1 = this.treeView.roots;
      for (index = i = 0, len = ref1.length; i < len; index = ++i) {
        root = ref1[index];
        if (root.directory === directory) {
          rootIndex = index;
          break;
        }
      }
      e.dataTransfer.setData('from-root-index', rootIndex);
      e.dataTransfer.setData('from-root-path', directory.path);
      e.dataTransfer.setData('from-window-id', this.getWindowId());
      e.dataTransfer.setData('text/plain', directory.path);
      if ((ref2 = process.platform) === 'darwin' || ref2 === 'linux') {
        if (!this.uriHasProtocol(directory.path)) {
          pathUri = "file://" + directory.path;
        }
        return e.dataTransfer.setData('text/uri-list', pathUri);
      }
    };

    RootDragAndDropHandler.prototype.uriHasProtocol = function(uri) {
      var error;
      try {
        return url.parse(uri).protocol != null;
      } catch (error1) {
        error = error1;
        return false;
      }
    };

    RootDragAndDropHandler.prototype.onDragEnter = function(e) {
      if (!this.treeView.list.contains(e.target)) {
        return;
      }
      if (!this.isAtomTreeViewEvent(e)) {
        return;
      }
      return e.stopPropagation();
    };

    RootDragAndDropHandler.prototype.onDragLeave = function(e) {
      if (!this.treeView.list.contains(e.target)) {
        return;
      }
      if (!this.isAtomTreeViewEvent(e)) {
        return;
      }
      e.stopPropagation();
      if (e.target === e.currentTarget) {
        return this.removePlaceholder();
      }
    };

    RootDragAndDropHandler.prototype.onDragEnd = function(e) {
      if (!e.target.matches('.project-root-header')) {
        return;
      }
      if (!this.isAtomTreeViewEvent(e)) {
        return;
      }
      e.stopPropagation();
      return this.clearDropTarget();
    };

    RootDragAndDropHandler.prototype.onDragOver = function(e) {
      var element, entry, newDropTargetIndex, projectRoots;
      if (!this.treeView.list.contains(e.target)) {
        return;
      }
      if (!this.isAtomTreeViewEvent(e)) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();
      entry = e.currentTarget;
      if (this.treeView.roots.length === 0) {
        this.treeView.list.appendChild(this.getPlaceholder());
        return;
      }
      newDropTargetIndex = this.getDropTargetIndex(e);
      if (newDropTargetIndex == null) {
        return;
      }
      if (this.prevDropTargetIndex === newDropTargetIndex) {
        return;
      }
      this.prevDropTargetIndex = newDropTargetIndex;
      projectRoots = this.treeView.roots;
      if (newDropTargetIndex < projectRoots.length) {
        element = projectRoots[newDropTargetIndex];
        element.classList.add('is-drop-target');
        return element.parentElement.insertBefore(this.getPlaceholder(), element);
      } else {
        element = projectRoots[newDropTargetIndex - 1];
        element.classList.add('drop-target-is-after');
        return element.parentElement.insertBefore(this.getPlaceholder(), element.nextSibling);
      }
    };

    RootDragAndDropHandler.prototype.onDropOnOtherWindow = function(e, fromItemIndex) {
      var paths;
      paths = atom.project.getPaths();
      paths.splice(fromItemIndex, 1);
      atom.project.setPaths(paths);
      return this.clearDropTarget();
    };

    RootDragAndDropHandler.prototype.clearDropTarget = function() {
      var element;
      element = this.treeView.element.querySelector(".is-dragging");
      if (element != null) {
        element.classList.remove('is-dragging');
      }
      if (element != null) {
        element.updateTooltip();
      }
      return this.removePlaceholder();
    };

    RootDragAndDropHandler.prototype.onDrop = function(e) {
      var browserWindow, dataTransfer, fromIndex, fromRootIndex, fromRootPath, fromWindowId, projectPaths, toIndex;
      if (!this.treeView.list.contains(e.target)) {
        return;
      }
      if (!this.isAtomTreeViewEvent(e)) {
        return;
      }
      e.preventDefault();
      e.stopPropagation();
      dataTransfer = e.dataTransfer;
      fromWindowId = parseInt(dataTransfer.getData('from-window-id'));
      fromRootPath = dataTransfer.getData('from-root-path');
      fromIndex = parseInt(dataTransfer.getData('project-root-index'));
      fromRootIndex = parseInt(dataTransfer.getData('from-root-index'));
      toIndex = this.getDropTargetIndex(e);
      this.clearDropTarget();
      if (fromWindowId === this.getWindowId()) {
        if (fromIndex !== toIndex) {
          projectPaths = atom.project.getPaths();
          projectPaths.splice(fromIndex, 1);
          if (toIndex > fromIndex) {
            toIndex -= 1;
          }
          projectPaths.splice(toIndex, 0, fromRootPath);
          return atom.project.setPaths(projectPaths);
        }
      } else {
        projectPaths = atom.project.getPaths();
        projectPaths.splice(toIndex, 0, fromRootPath);
        atom.project.setPaths(projectPaths);
        if (!isNaN(fromWindowId)) {
          browserWindow = remote.BrowserWindow.fromId(fromWindowId);
          return browserWindow != null ? browserWindow.webContents.send('tree-view:project-folder-dropped', fromIndex) : void 0;
        }
      }
    };

    RootDragAndDropHandler.prototype.getDropTargetIndex = function(e) {
      var center, projectRoot, projectRootIndex, projectRoots;
      if (this.isPlaceholder(e.target)) {
        return;
      }
      projectRoots = this.treeView.roots;
      projectRoot = e.target.closest('.project-root');
      if (!projectRoot) {
        projectRoot = projectRoots[projectRoots.length - 1];
      }
      if (!projectRoot) {
        return 0;
      }
      projectRootIndex = this.treeView.roots.indexOf(projectRoot);
      center = projectRoot.getBoundingClientRect().top + projectRoot.offsetHeight / 2;
      if (e.pageY < center) {
        return projectRootIndex;
      } else {
        return projectRootIndex + 1;
      }
    };

    RootDragAndDropHandler.prototype.canDragStart = function(e) {
      return e.target.closest('.project-root-header');
    };

    RootDragAndDropHandler.prototype.isDragging = function(e) {
      var i, item, len, ref1;
      ref1 = e.dataTransfer.items;
      for (i = 0, len = ref1.length; i < len; i++) {
        item = ref1[i];
        if (item.type === 'from-root-path') {
          return true;
        }
      }
      return false;
    };

    RootDragAndDropHandler.prototype.isAtomTreeViewEvent = function(e) {
      var i, item, len, ref1;
      ref1 = e.dataTransfer.items;
      for (i = 0, len = ref1.length; i < len; i++) {
        item = ref1[i];
        if (item.type === 'atom-tree-view-root-event') {
          return true;
        }
      }
      return false;
    };

    RootDragAndDropHandler.prototype.getPlaceholder = function() {
      if (!this.placeholderEl) {
        this.placeholderEl = document.createElement('li');
        this.placeholderEl.classList.add('placeholder');
      }
      return this.placeholderEl;
    };

    RootDragAndDropHandler.prototype.removePlaceholder = function() {
      var ref1;
      if ((ref1 = this.placeholderEl) != null) {
        ref1.remove();
      }
      return this.placeholderEl = null;
    };

    RootDragAndDropHandler.prototype.isPlaceholder = function(element) {
      return element.classList.contains('.placeholder');
    };

    RootDragAndDropHandler.prototype.getWindowId = function() {
      return this.processId != null ? this.processId : this.processId = atom.getCurrentWindow().id;
    };

    return RootDragAndDropHandler;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
