(function() {
  var CharacterPattern, _;

  _ = require('underscore-plus');

  CharacterPattern = /[^\s]/;

  module.exports = {
    activate: function() {
      return this.commandDisposable = atom.commands.add('atom-text-editor', {
        'autoflow:reflow-selection': (function(_this) {
          return function(event) {
            return _this.reflowSelection(event.currentTarget.getModel());
          };
        })(this)
      });
    },
    deactivate: function() {
      var ref;
      if ((ref = this.commandDisposable) != null) {
        ref.dispose();
      }
      return this.commandDisposable = null;
    },
    reflowSelection: function(editor) {
      var range, reflowOptions, reflowedText;
      range = editor.getSelectedBufferRange();
      if (range.isEmpty()) {
        range = editor.getCurrentParagraphBufferRange();
      }
      if (range == null) {
        return;
      }
      reflowOptions = {
        wrapColumn: this.getPreferredLineLength(editor),
        tabLength: this.getTabLength(editor)
      };
      reflowedText = this.reflow(editor.getTextInRange(range), reflowOptions);
      return editor.getBuffer().setTextInRange(range, reflowedText);
    },
    reflow: function(text, arg) {
      var beginningLinesToIgnore, block, blockLines, currentLine, currentLineLength, endingLinesToIgnore, escapedLinePrefix, firstLine, i, j, latexTagEndRegex, latexTagRegex, latexTagStartRegex, leadingVerticalSpace, len, len1, linePrefix, linePrefixTabExpanded, lines, paragraphBlocks, paragraphs, ref, segment, tabLength, tabLengthInSpaces, trailingVerticalSpace, wrapColumn, wrappedLinePrefix, wrappedLines;
      wrapColumn = arg.wrapColumn, tabLength = arg.tabLength;
      paragraphs = [];
      text = text.replace(/\r\n?/g, '\n');
      leadingVerticalSpace = text.match(/^\s*\n/);
      if (leadingVerticalSpace) {
        text = text.substr(leadingVerticalSpace.length);
      } else {
        leadingVerticalSpace = '';
      }
      trailingVerticalSpace = text.match(/\n\s*$/);
      if (trailingVerticalSpace) {
        text = text.substr(0, text.length - trailingVerticalSpace.length);
      } else {
        trailingVerticalSpace = '';
      }
      paragraphBlocks = text.split(/\n\s*\n/g);
      if (tabLength) {
        tabLengthInSpaces = Array(tabLength + 1).join(' ');
      } else {
        tabLengthInSpaces = '';
      }
      for (i = 0, len = paragraphBlocks.length; i < len; i++) {
        block = paragraphBlocks[i];
        blockLines = block.split('\n');
        beginningLinesToIgnore = [];
        endingLinesToIgnore = [];
        latexTagRegex = /^\s*\\\w+(\[.*\])?\{\w+\}(\[.*\])?\s*$/g;
        latexTagStartRegex = /^\s*\\\w+\s*\{\s*$/g;
        latexTagEndRegex = /^\s*\}\s*$/g;
        while (blockLines.length > 0 && (blockLines[0].match(latexTagRegex) || blockLines[0].match(latexTagStartRegex))) {
          beginningLinesToIgnore.push(blockLines[0]);
          blockLines.shift();
        }
        while (blockLines.length > 0 && (blockLines[blockLines.length - 1].match(latexTagRegex) || blockLines[blockLines.length - 1].match(latexTagEndRegex))) {
          endingLinesToIgnore.unshift(blockLines[blockLines.length - 1]);
          blockLines.pop();
        }
        if (!(blockLines.length > 0)) {
          paragraphs.push(block);
          continue;
        }
        linePrefix = blockLines[0].match(/^\s*(\/\/|\/\*|;;|#'|\|\|\||--|[#%*>-])?\s*/g)[0];
        linePrefixTabExpanded = linePrefix;
        if (tabLengthInSpaces) {
          linePrefixTabExpanded = linePrefix.replace(/\t/g, tabLengthInSpaces);
        }
        if (linePrefix) {
          escapedLinePrefix = _.escapeRegExp(linePrefix);
          blockLines = blockLines.map(function(blockLine) {
            return blockLine.replace(RegExp("^" + escapedLinePrefix), '');
          });
        }
        blockLines = blockLines.map(function(blockLine) {
          return blockLine.replace(/^\s+/, '');
        });
        lines = [];
        currentLine = [];
        currentLineLength = linePrefixTabExpanded.length;
        wrappedLinePrefix = linePrefix.replace(/^(\s*)\/\*/, '$1  ').replace(/^(\s*)-(?!-)/, '$1 ');
        firstLine = true;
        ref = this.segmentText(blockLines.join(' '));
        for (j = 0, len1 = ref.length; j < len1; j++) {
          segment = ref[j];
          if (this.wrapSegment(segment, currentLineLength, wrapColumn)) {
            if (firstLine !== true) {
              if (linePrefix.search(/^\s*\/\*/) !== -1 || linePrefix.search(/^\s*-(?!-)/) !== -1) {
                linePrefix = wrappedLinePrefix;
              }
            }
            lines.push(linePrefix + currentLine.join(''));
            currentLine = [];
            currentLineLength = linePrefixTabExpanded.length;
            firstLine = false;
          }
          currentLine.push(segment);
          currentLineLength += segment.length;
        }
        lines.push(linePrefix + currentLine.join(''));
        wrappedLines = beginningLinesToIgnore.concat(lines.concat(endingLinesToIgnore));
        paragraphs.push(wrappedLines.join('\n').replace(/\s+\n/g, '\n'));
      }
      return leadingVerticalSpace + paragraphs.join('\n\n') + trailingVerticalSpace;
    },
    getTabLength: function(editor) {
      var ref;
      return (ref = atom.config.get('editor.tabLength', {
        scope: editor.getRootScopeDescriptor()
      })) != null ? ref : 2;
    },
    getPreferredLineLength: function(editor) {
      return atom.config.get('editor.preferredLineLength', {
        scope: editor.getRootScopeDescriptor()
      });
    },
    wrapSegment: function(segment, currentLineLength, wrapColumn) {
      return CharacterPattern.test(segment) && (currentLineLength + segment.length > wrapColumn) && (currentLineLength > 0 || segment.length < wrapColumn);
    },
    segmentText: function(text) {
      var match, re, segments;
      segments = [];
      re = /[\s]+|[^\s]+/g;
      while (match = re.exec(text)) {
        segments.push(match[0]);
      }
      return segments;
    }
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
