import { phantom } from './function';
import { none, some } from './Option';
import { fromEquals } from './Setoid';
export const URI = 'Map';
/**
 * @since 1.17.0
 */
export const getShow = (SK, SA) => {
    return {
        show: m => {
            let elements = '';
            m.forEach((a, k) => {
                elements += `[${SK.show(k)}, ${SA.show(a)}], `;
            });
            if (elements !== '') {
                elements = elements.substring(0, elements.length - 2);
            }
            return `new Map([${elements}])`;
        }
    };
};
/**
 * Calculate the number of key/value pairs in a map
 *
 * @since 1.14.0
 */
export const size = (d) => d.size;
/**
 * Test whether or not a map is empty
 *
 * @since 1.14.0
 */
export const isEmpty = (d) => d.size === 0;
/**
 * Test whether or not a key exists in a map
 *
 * @since 1.14.0
 */
export const member = (S) => {
    const lookupS = lookup(S);
    return (k, m) => lookupS(k, m).isSome();
};
/**
 * Test whether or not a value is a member of a map
 *
 * @since 1.14.0
 */
export const elem = (S) => (a, m) => {
    const values = m.values();
    let e;
    while (!(e = values.next()).done) {
        const v = e.value;
        if (S.equals(a, v)) {
            return true;
        }
    }
    return false;
};
/**
 * Get a sorted array of the keys contained in a map
 *
 * @since 1.14.0
 */
export const keys = (O) => m => Array.from(m.keys()).sort(O.compare);
/**
 * Get a sorted array of the values contained in a map
 *
 * @since 1.14.0
 */
export const values = (O) => m => Array.from(m.values()).sort(O.compare);
/**
 * @since 1.14.0
 */
export const collect = (O) => {
    const keysO = keys(O);
    return (m, f) => {
        const out = [];
        const ks = keysO(m);
        for (let key of ks) {
            out.push(f(key, m.get(key)));
        }
        return out;
    };
};
/**
 * Get a sorted of the key/value pairs contained in a map
 *
 * @since 1.14.0
 */
export const toArray = (O) => {
    const collectO = collect(O);
    return m => collectO(m, (k, a) => [k, a]);
};
export function toUnfoldable(O, unfoldable) {
    const toArrayO = toArray(O);
    return d => {
        const arr = toArrayO(d);
        const len = arr.length;
        return unfoldable.unfoldr(0, b => (b < len ? some([arr[b], b + 1]) : none));
    };
}
/**
 * Insert or replace a key/value pair in a map
 *
 * @since 1.14.0
 */
export const insert = (S) => {
    const lookupS = lookupWithKey(S);
    return (k, a, m) => {
        const found = lookupS(k, m);
        if (found.isNone()) {
            const r = new Map(m);
            r.set(k, a);
            return r;
        }
        else if (found.value[1] !== a) {
            const r = new Map(m);
            r.set(found.value[0], a);
            return r;
        }
        return m;
    };
};
/**
 * Delete a key and value from a map
 *
 * @since 1.14.0
 */
export const remove = (S) => {
    const lookupS = lookupWithKey(S);
    return (k, m) => {
        const found = lookupS(k, m);
        if (found.isSome()) {
            const r = new Map(m);
            r.delete(found.value[0]);
            return r;
        }
        return m;
    };
};
/**
 * Delete a key and value from a map, returning the value as well as the subsequent map
 *
 * @since 1.14.0
 */
export const pop = (S) => {
    const lookupS = lookup(S);
    const removeS = remove(S);
    return (k, m) => lookupS(k, m).map(a => [a, removeS(k, m)]);
};
/**
 * Lookup the value for a key in a `Map`.
 * If the result is a `Some`, the existing key is also returned.
 *
 * @since 1.14.0
 */
export const lookupWithKey = (S) => (k, m) => {
    const entries = m.entries();
    let e;
    while (!(e = entries.next()).done) {
        const [ka, a] = e.value;
        if (S.equals(ka, k)) {
            return some([ka, a]);
        }
    }
    return none;
};
/**
 * Lookup the value for a key in a `Map`.
 *
 * @since 1.14.0
 */
export const lookup = (S) => {
    const lookupWithKeyS = lookupWithKey(S);
    return (k, m) => lookupWithKeyS(k, m).map(([_, a]) => a);
};
/**
 * Test whether or not one Map contains all of the keys and values contained in another Map
 *
 * @since 1.14.0
 */
export const isSubmap = (SK, SA) => {
    const lookupWithKeyS = lookupWithKey(SK);
    return (d1, d2) => {
        const entries = d1.entries();
        let e;
        while (!(e = entries.next()).done) {
            const [k, a] = e.value;
            const d2OptA = lookupWithKeyS(k, d2);
            if (d2OptA.isNone() || !SK.equals(k, d2OptA.value[0]) || !SA.equals(a, d2OptA.value[1])) {
                return false;
            }
        }
        return true;
    };
};
/**
 * @since 1.14.0
 */
export const empty = new Map();
/**
 * @since 1.14.0
 */
export const getSetoid = (SK, SA) => {
    const isSubmap_ = isSubmap(SK, SA);
    return fromEquals((x, y) => isSubmap_(x, y) && isSubmap_(y, x));
};
/**
 * Gets `Monoid` instance for Maps given `Semigroup` instance for their values
 *
 * @since 1.14.0
 */
export const getMonoid = (SK, SA) => {
    const lookupWithKeyS = lookupWithKey(SK);
    return {
        concat: (mx, my) => {
            const r = new Map(mx);
            const entries = my.entries();
            let e;
            while (!(e = entries.next()).done) {
                const [k, a] = e.value;
                const mxOptA = lookupWithKeyS(k, mx);
                if (mxOptA.isSome()) {
                    r.set(mxOptA.value[0], SA.concat(mxOptA.value[1], a));
                }
                else {
                    r.set(k, a);
                }
            }
            return r;
        },
        empty
    };
};
/**
 * @since 1.14.0
 */
const filter = (fa, p) => filterWithIndex(fa, (_, a) => p(a));
/**
 * @since 1.14.0
 */
const mapWithIndex = (fa, f) => {
    const m = new Map();
    const entries = fa.entries();
    let e;
    while (!(e = entries.next()).done) {
        const [key, a] = e.value;
        m.set(key, f(key, a));
    }
    return m;
};
/**
 * @since 1.14.0
 */
const _map = (fa, f) => mapWithIndex(fa, (_, a) => f(a));
/**
 * @since 1.14.0
 */
const reduce = (O) => {
    const reduceWithIndexO = reduceWithIndex(O);
    return (fa, b, f) => reduceWithIndexO(fa, b, (_, b, a) => f(b, a));
};
/**
 * @since 1.14.0
 */
const foldMap = (O) => M => {
    const foldMapWithIndexOM = foldMapWithIndex(O)(M);
    return (fa, f) => foldMapWithIndexOM(fa, (_, a) => f(a));
};
/**
 * @since 1.14.0
 */
const foldr = (O) => {
    const foldrWithIndexO = foldrWithIndex(O);
    return (fa, b, f) => foldrWithIndexO(fa, b, (_, a, b) => f(a, b));
};
/**
 * @since 1.14.0
 */
const reduceWithIndex = (O) => {
    const keysO = keys(O);
    return (fa, b, f) => {
        let out = b;
        const ks = keysO(fa);
        const len = ks.length;
        for (let i = 0; i < len; i++) {
            const k = ks[i];
            out = f(k, out, fa.get(k));
        }
        return out;
    };
};
/**
 * @since 1.14.0
 */
const foldMapWithIndex = (O) => {
    const keysO = keys(O);
    return (M) => (fa, f) => {
        let out = M.empty;
        const ks = keysO(fa);
        const len = ks.length;
        for (let i = 0; i < len; i++) {
            const k = ks[i];
            out = M.concat(out, f(k, fa.get(k)));
        }
        return out;
    };
};
/**
 * @since 1.14.0
 */
const foldrWithIndex = (O) => {
    const keysO = keys(O);
    return (fa, b, f) => {
        let out = b;
        const ks = keysO(fa);
        const len = ks.length;
        for (let i = len - 1; i >= 0; i--) {
            const k = ks[i];
            out = f(k, fa.get(k), out);
        }
        return out;
    };
};
/**
 * Create a map with one key/value pair
 *
 * @since 1.14.0
 */
export const singleton = (k, a) => {
    return new Map([[k, a]]);
};
/**
 * @since 1.14.0
 */
const traverseWithIndex = (F) => {
    return (ta, f) => {
        let fm = F.of(empty);
        const entries = ta.entries();
        let e;
        while (!(e = entries.next()).done) {
            const [key, a] = e.value;
            fm = F.ap(F.map(fm, m => (b) => new Map(m).set(key, b)), f(key, a));
        }
        return fm;
    };
};
/**
 * @since 1.14.0
 */
const traverse = (F) => {
    const traverseWithIndexF = traverseWithIndex(F);
    return (ta, f) => traverseWithIndexF(ta, (_, a) => f(a));
};
/**
 * @since 1.14.0
 */
const sequence = (F) => {
    const traverseWithIndexF = traverseWithIndex(F);
    return ta => traverseWithIndexF(ta, (_, a) => a);
};
/**
 * @since 1.14.0
 */
const compact = (fa) => {
    const m = new Map();
    const entries = fa.entries();
    let e;
    while (!(e = entries.next()).done) {
        const [k, oa] = e.value;
        if (oa.isSome()) {
            m.set(k, oa.value);
        }
    }
    return m;
};
/**
 * @since 1.14.0
 */
const partitionMap = (fa, f) => partitionMapWithIndex(fa, (_, a) => f(a));
/**
 * @since 1.14.0
 */
const partition = (fa, p) => partitionWithIndex(fa, (_, a) => p(a));
/**
 * @since 1.14.0
 */
const separate = (fa) => {
    const left = new Map();
    const right = new Map();
    const entries = fa.entries();
    let e;
    while (!(e = entries.next()).done) {
        const [k, ei] = e.value;
        if (ei.isLeft()) {
            left.set(k, ei.value);
        }
        else {
            right.set(k, ei.value);
        }
    }
    return {
        left,
        right
    };
};
/**
 * @since 1.14.0
 */
const wither = (F) => {
    const traverseF = traverse(F);
    return (wa, f) => F.map(traverseF(wa, f), compact);
};
/**
 * @since 1.14.0
 */
const wilt = (F) => {
    const traverseF = traverse(F);
    return (wa, f) => F.map(traverseF(wa, f), separate);
};
/**
 * @since 1.14.0
 */
const filterMap = (fa, f) => {
    return filterMapWithIndex(fa, (_, a) => f(a));
};
/**
 * @since 1.14.0
 */
const partitionMapWithIndex = (fa, f) => {
    const left = new Map();
    const right = new Map();
    const entries = fa.entries();
    let e;
    while (!(e = entries.next()).done) {
        const [k, a] = e.value;
        const ei = f(k, a);
        if (ei.isLeft()) {
            left.set(k, ei.value);
        }
        else {
            right.set(k, ei.value);
        }
    }
    return {
        left,
        right
    };
};
/**
 * @since 1.14.0
 */
const partitionWithIndex = (fa, p) => {
    const left = new Map();
    const right = new Map();
    const entries = fa.entries();
    let e;
    while (!(e = entries.next()).done) {
        const [k, a] = e.value;
        if (p(k, a)) {
            right.set(k, a);
        }
        else {
            left.set(k, a);
        }
    }
    return {
        left,
        right
    };
};
/**
 * @since 1.14.0
 */
const filterMapWithIndex = (fa, f) => {
    const m = new Map();
    const entries = fa.entries();
    let e;
    while (!(e = entries.next()).done) {
        const [k, a] = e.value;
        const o = f(k, a);
        if (o.isSome()) {
            m.set(k, o.value);
        }
    }
    return m;
};
/**
 * @since 1.14.0
 */
const filterWithIndex = (fa, p) => {
    const m = new Map();
    const entries = fa.entries();
    let e;
    while (!(e = entries.next()).done) {
        const [k, a] = e.value;
        if (p(k, a)) {
            m.set(k, a);
        }
    }
    return m;
};
export function fromFoldable(S, F) {
    return (ta, onConflict) => {
        const lookupWithKeyS = lookupWithKey(S);
        return F.reduce(ta, new Map(), (b, [k, a]) => {
            const bOpt = lookupWithKeyS(k, b);
            if (bOpt.isSome()) {
                b.set(bOpt.value[0], onConflict(bOpt.value[1], a));
            }
            else {
                b.set(k, a);
            }
            return b;
        });
    };
}
/**
 * @since 1.14.0
 */
const compactable = {
    URI,
    compact,
    separate
};
/**
 * @since 1.14.0
 */
const functor = {
    URI,
    map: _map
};
/**
 * @since 1.14.0
 */
const getFunctorWithIndex = () => {
    return Object.assign({ _L: phantom }, functor, { mapWithIndex: mapWithIndex });
};
/**
 * @since 1.14.0
 */
const getFoldable = (O) => {
    return {
        URI,
        _L: phantom,
        reduce: reduce(O),
        foldMap: foldMap(O),
        foldr: foldr(O)
    };
};
/**
 * @since 1.14.0
 */
const getFoldableWithIndex = (O) => {
    return Object.assign({}, getFoldable(O), { reduceWithIndex: reduceWithIndex(O), foldMapWithIndex: foldMapWithIndex(O), foldrWithIndex: foldrWithIndex(O) });
};
/**
 * @since 1.14.0
 */
const filterable = Object.assign({}, compactable, functor, { filter,
    filterMap,
    partition,
    partitionMap });
/**
 * @since 1.14.0
 */
export const getFilterableWithIndex = () => {
    return Object.assign({}, filterable, getFunctorWithIndex(), { partitionMapWithIndex,
        partitionWithIndex,
        filterMapWithIndex,
        filterWithIndex });
};
/**
 * @since 1.14.0
 */
const getTraversable = (O) => {
    return Object.assign({ _L: phantom }, getFoldable(O), functor, { traverse,
        sequence });
};
/**
 * @since 1.14.0
 */
export const getWitherable = (O) => {
    return Object.assign({}, filterable, getTraversable(O), { wilt,
        wither });
};
/**
 * @since 1.14.0
 */
export const getTraversableWithIndex = (O) => {
    return Object.assign({}, getFunctorWithIndex(), getFoldableWithIndex(O), getTraversable(O), { traverseWithIndex });
};
/**
 * @since 1.14.0
 */
export const map = Object.assign({ URI }, compactable, functor, filterable);
