"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const electron_1 = require("electron");
const electron_util_1 = require("electron-util");
const config_1 = require("./config");
const menu_bar_mode_1 = require("./menu-bar-mode");
let tray = null;
let previousMessageCount = 0;
let contextMenu;
exports.default = {
    create: (win) => {
        if (tray) {
            return;
        }
        function toggleWin() {
            if (win.isVisible()) {
                win.hide();
            }
            else {
                win.show();
            }
        }
        const macosMenuItems = electron_util_1.is.macos
            ? [
                {
                    label: 'Disable Menu Bar Mode',
                    click() {
                        config_1.default.set('menuBarMode', false);
                        menu_bar_mode_1.toggleMenuBarMode(win);
                    }
                },
                {
                    label: 'Show Dock Icon',
                    type: 'checkbox',
                    checked: config_1.default.get('showDockIcon'),
                    click(menuItem) {
                        config_1.default.set('showDockIcon', menuItem.checked);
                        if (menuItem.checked) {
                            electron_1.app.dock.show();
                        }
                        else {
                            electron_1.app.dock.hide();
                        }
                        const dockMenuItem = contextMenu.getMenuItemById('dockMenu');
                        dockMenuItem.visible = !menuItem.checked;
                    }
                },
                {
                    type: 'separator'
                },
                {
                    id: 'dockMenu',
                    label: 'Menu',
                    visible: !config_1.default.get('showDockIcon'),
                    submenu: electron_1.Menu.getApplicationMenu()
                }
            ]
            : [];
        contextMenu = electron_1.Menu.buildFromTemplate([
            {
                label: 'Toggle',
                visible: !electron_util_1.is.macos,
                click() {
                    toggleWin();
                }
            },
            ...macosMenuItems,
            {
                type: 'separator'
            },
            {
                role: 'quit'
            }
        ]);
        tray = new electron_1.Tray(getIconPath(false));
        tray.setContextMenu(contextMenu);
        updateToolTip(0);
        const trayClickHandler = () => {
            if (!win.isFullScreen()) {
                toggleWin();
            }
        };
        tray.on('click', trayClickHandler);
        tray.on('double-click', trayClickHandler);
        tray.on('right-click', () => {
            if (tray) {
                tray.popUpContextMenu(contextMenu);
            }
        });
    },
    destroy: () => {
        // Workaround for https://github.com/electron/electron/issues/14036
        setTimeout(() => {
            if (tray) {
                tray.destroy();
                tray = null;
            }
        }, 500);
    },
    update: (messageCount) => {
        if (!tray || previousMessageCount === messageCount) {
            return;
        }
        previousMessageCount = messageCount;
        tray.setImage(getIconPath(messageCount > 0));
        updateToolTip(messageCount);
    },
    setBadge: (shouldDisplayUnread) => {
        if (electron_util_1.is.macos || !tray) {
            return;
        }
        const icon = shouldDisplayUnread ? 'IconTrayUnread.png' : 'IconTray.png';
        const iconPath = path.join(__dirname, '..', 'static', icon);
        tray.setImage(iconPath);
    }
};
function updateToolTip(counter) {
    if (!tray) {
        return;
    }
    let tooltip = electron_1.app.getName();
    if (counter > 0) {
        tooltip += `- ${counter} unread ${counter === 1 ? 'message' : 'messages'}`;
    }
    tray.setToolTip(tooltip);
}
function getIconPath(hasUnreadMessages) {
    const icon = electron_util_1.is.macos
        ? getMacOSIconName(hasUnreadMessages)
        : getNonMacOSIconName(hasUnreadMessages);
    return path.join(__dirname, '..', `static/${icon}`);
}
function getNonMacOSIconName(hasUnreadMessages) {
    return hasUnreadMessages ? 'IconTrayUnread.png' : 'IconTray.png';
}
function getMacOSIconName(hasUnreadMessages) {
    return hasUnreadMessages ? 'IconMenuBarUnreadTemplate.png' : 'IconMenuBarTemplate.png';
}
//# sourceMappingURL=tray.js.map