"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

class ModelObserver {
  constructor({
    fetchData,
    didUpdate
  }) {
    this.fetchData = fetchData || (() => {});

    this.didUpdate = didUpdate || (() => {});

    this.activeModel = null;
    this.activeModelData = null;
    this.activeModelUpdateSubscription = null;
    this.inProgress = false;
    this.pending = false;
  }

  setActiveModel(model) {
    if (model !== this.activeModel) {
      if (this.activeModelUpdateSubscription) {
        this.activeModelUpdateSubscription.dispose();
        this.activeModelUpdateSubscription = null;
      }

      this.activeModel = model;
      this.activeModelData = null;
      this.inProgress = false;
      this.pending = false;
      this.didUpdate(model);

      if (model) {
        this.activeModelUpdateSubscription = model.onDidUpdate(() => this.refreshModelData(model));
        return this.refreshModelData(model);
      }
    }

    return null;
  }

  refreshModelData(model = this.activeModel) {
    if (this.inProgress) {
      this.pending = true;
      return null;
    }

    this.lastModelDataRefreshPromise = this._refreshModelData(model);
    return this.lastModelDataRefreshPromise;
  }

  async _refreshModelData(model) {
    try {
      this.inProgress = true;
      const fetchDataPromise = this.fetchData(model);
      this.lastFetchDataPromise = fetchDataPromise;
      const modelData = await fetchDataPromise; // Since we re-fetch immediately when the model changes,
      // we need to ensure a fetch for an old active model
      // does not trample the newer fetch for the newer active model.

      if (fetchDataPromise === this.lastFetchDataPromise) {
        this.activeModel = model;
        this.activeModelData = modelData;
        this.didUpdate(model);
      }
    } finally {
      this.inProgress = false;

      if (this.pending) {
        this.pending = false;
        this.refreshModelData();
      }
    }
  }

  getActiveModel() {
    return this.activeModel;
  }

  getActiveModelData() {
    return this.activeModelData;
  }

  getLastModelDataRefreshPromise() {
    return this.lastModelDataRefreshPromise;
  }

  hasPendingUpdate() {
    return this.pending;
  }

  destroy() {
    if (this.activeModelUpdateSubscription) {
      this.activeModelUpdateSubscription.dispose();
    }
  }

}

exports["default"] = ModelObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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