#ifndef _EFL_UI_ANIMATION_VIEW_EO_H_
#define _EFL_UI_ANIMATION_VIEW_EO_H_

#ifndef _EFL_UI_ANIMATION_VIEW_EO_CLASS_TYPE
#define _EFL_UI_ANIMATION_VIEW_EO_CLASS_TYPE

typedef Eo Efl_Ui_Animation_View;

#endif

#ifndef _EFL_UI_ANIMATION_VIEW_EO_TYPES
#define _EFL_UI_ANIMATION_VIEW_EO_TYPES

/** State of animation view
 *
 * @ingroup Efl_Ui
 */
typedef enum
{
  EFL_UI_ANIMATION_VIEW_STATE_NOT_READY = 0, /**< Animation is not ready to
                                              * play. (Probably, it didn't file
                                              * set yet or failed to read file.
                                              */
  EFL_UI_ANIMATION_VIEW_STATE_PLAY, /**< Animation is playing. See
                                     * @ref efl_ui_animation_view_play. */
  EFL_UI_ANIMATION_VIEW_STATE_PLAY_BACK, /**< Animation is playing back
                                          * (rewinding). See
                                          * @ref efl_ui_animation_view_play_back.
                                          */
  EFL_UI_ANIMATION_VIEW_STATE_PAUSE, /**< Animation has been paused. To continue
                                      * animation call
                                      * @ref efl_ui_animation_view_resume. See
                                      * @ref efl_ui_animation_view_pause. */
  EFL_UI_ANIMATION_VIEW_STATE_STOP /**< Animation view successfully loaded a
                                    * file then readied for playing. Otherwise
                                    * after finished animation or stopped
                                    * forcibly by request. See
                                    * @ref efl_ui_animation_view_stop. */
} Efl_Ui_Animation_View_State;


#endif
#ifdef EFL_BETA_API_SUPPORT
/**
 * @brief Elementary Animation view class. Animation view is designed to show
 * and play animation of vector graphics based content. It hides all
 * @ref Efl_Canvas_Vg_Object details but just open an API to read vector data
 * from file. Also, it implements details of animation control methods of
 * Vector.
 *
 * Vector data could contain static or animatable vector elements including
 * animation information. Available vector data file formats are SVG, JSON and
 * EET. @ref Efl_Ui_Animation_View currently only supports the animation
 * information contained in JSON (known as Lottie file as well) and EET files.
 *
 * @ingroup Efl_Ui_Animation_View
 */
#define EFL_UI_ANIMATION_VIEW_CLASS efl_ui_animation_view_class_get()

EWAPI const Efl_Class *efl_ui_animation_view_class_get(void);

/**
 * @brief Animation will be started automatically when it's possible.
 *
 * If @ref efl_ui_animation_view_auto_play_get is @c true, animation will be
 * started when it's readied. The condition of @c auto play is when animation
 * view opened file successfully, yet to play it plus when the object is
 * visible. If animation view is disabled, invisible, it turns to pause state
 * then resume animation when it's visible again.
 *
 * @c true Enable auto play mode, disable otherwise.
 *
 * @warning This auto play will be only affected to the next animation source.
 * So must be called before setting animation file.
 *
 * @param[in] obj The object.
 * @param[in] auto_play Auto play mode.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI void efl_ui_animation_view_auto_play_set(Eo *obj, Eina_Bool auto_play);

/**
 * @brief Animation will be started automatically when it's possible.
 *
 * If @ref efl_ui_animation_view_auto_play_get is @c true, animation will be
 * started when it's readied. The condition of @c auto play is when animation
 * view opened file successfully, yet to play it plus when the object is
 * visible. If animation view is disabled, invisible, it turns to pause state
 * then resume animation when it's visible again.
 *
 * @c true Enable auto play mode, disable otherwise.
 *
 * @warning This auto play will be only affected to the next animation source.
 * So must be called before setting animation file.
 *
 * @param[in] obj The object.
 *
 * @return Auto play mode.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Bool efl_ui_animation_view_auto_play_get(const Eo *obj);

/**
 * @brief Turn on/off animation looping.
 *
 * If @ref efl_ui_animation_view_auto_repeat_get is @c true, it repeats
 * animation when animation frame is reached to end. This auto repeat mode is
 * valid to both play and play_back cases.
 *
 * @c true Enable auto play mode, disable otherwise.
 *
 * @param[in] obj The object.
 * @param[in] auto_repeat Loop mode, Default is @c false.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI void efl_ui_animation_view_auto_repeat_set(Eo *obj, Eina_Bool auto_repeat);

/**
 * @brief Turn on/off animation looping.
 *
 * If @ref efl_ui_animation_view_auto_repeat_get is @c true, it repeats
 * animation when animation frame is reached to end. This auto repeat mode is
 * valid to both play and play_back cases.
 *
 * @c true Enable auto play mode, disable otherwise.
 *
 * @param[in] obj The object.
 *
 * @return Loop mode, Default is @c false.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Bool efl_ui_animation_view_auto_repeat_get(const Eo *obj);

/**
 * @brief Control animation speed.
 *
 * Control animation speed by multiplying @c speed value. If you want to play
 * animation double-time faster, you can give @c speed 2. If you want to play
 * animation double-time slower, you can give @c speed 0.5.
 *
 * @warning speed must be greater than zero.
 *
 * @param[in] obj The object.
 * @param[in] speed Speed factor.
 *
 * @return @c true when it's successful. @c false otherwise.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Bool efl_ui_animation_view_speed_set(Eo *obj, double speed);

/**
 * @brief Control animation speed.
 *
 * Control animation speed by multiplying @c speed value. If you want to play
 * animation double-time faster, you can give @c speed 2. If you want to play
 * animation double-time slower, you can give @c speed 0.5.
 *
 * @warning speed must be greater than zero.
 *
 * @param[in] obj The object.
 *
 * @return Speed factor.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI double efl_ui_animation_view_speed_get(const Eo *obj);

/**
 * @brief Get the duration of animation in seconds.
 *
 * This API returns total duration time of current animation in the seconds. If
 * current animation source isn't animatable, it returns zero.
 *
 * @param[in] obj The object.
 *
 * @return duration time in the seconds
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI double efl_ui_animation_view_duration_time_get(const Eo *obj);

/**
 * @brief Set current progress position of animation view object.
 *
 * When you required to jump on a certain frame instantly, you can change
 * current position by using this API.
 *
 * @warning The range of progress is 0 ~ 1.
 *
 * @param[in] obj The object.
 * @param[in] progress Progress position. Value must be 0 ~ 1.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI void efl_ui_animation_view_progress_set(Eo *obj, double progress);

/**
 * @brief Set current progress position of animation view object.
 *
 * When you required to jump on a certain frame instantly, you can change
 * current position by using this API.
 *
 * @warning The range of progress is 0 ~ 1.
 *
 * @param[in] obj The object.
 *
 * @return Progress position. Value must be 0 ~ 1.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI double efl_ui_animation_view_progress_get(const Eo *obj);

/**
 * @brief Number of current frame.
 *
 * Ranges from 0 to @ref efl_ui_animation_view_frame_count_get - 1.
 *
 * @param[in] obj The object.
 * @param[in] frame_num Current frame number.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI void efl_ui_animation_view_frame_set(Eo *obj, int frame_num);

/**
 * @brief Number of current frame.
 *
 * Ranges from 0 to @ref efl_ui_animation_view_frame_count_get - 1.
 *
 * @param[in] obj The object.
 *
 * @return Current frame number.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI int efl_ui_animation_view_frame_get(const Eo *obj);

/**
 * @brief Play animation one time instantly when it's available.
 *
 * If current keyframe is on a certain position by playing back, this will play
 * forward from there.
 *
 * @warning Play request will be ignored if animation source is not set yet or
 * animation is paused state or it's already on playing.
 *
 * @param[in] obj The object.
 *
 * @return @c true when it's successful. @c false otherwise.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Bool efl_ui_animation_view_play(Eo *obj);

/**
 * @brief Play back animation one time instantly when it's available.
 *
 * If current keyframe is on a certain position by playing, this will play
 * backward from there.
 *
 * @warning Play back request will be ignored if animation source is not set
 * yet or animation is paused state or it's already on playing back.
 *
 * @param[in] obj The object.
 *
 * @return @c true when it's successful. @c false otherwise.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Bool efl_ui_animation_view_play_back(Eo *obj);

/**
 * @brief Pause current animation instantly.
 *
 * Once animation is paused, animation view must get resume to play continue
 * again.
 *
 * @warning Animation must be on playing or playing back status.
 *
 * @param[in] obj The object.
 *
 * @return @c true when it's successful. @c false otherwise.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Bool efl_ui_animation_view_pause(Eo *obj);

/**
 * @brief Resume paused animation to continue animation.
 *
 * @warning This resume must be called on animation paused status.
 *
 * @param[in] obj The object.
 *
 * @return @c true when it's successful. @c false otherwise.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Bool efl_ui_animation_view_resume(Eo *obj);

/**
 * @brief Stop playing animation.
 *
 * Stop animation instantly regardless of its status and reset to show first
 * frame of animation. Even though current animation is paused, the animation
 * status will be stopped.
 *
 * @param[in] obj The object.
 *
 * @return @c true when it's successful. @c false otherwise.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Bool efl_ui_animation_view_stop(Eo *obj);

/** The default view size that specified from vector resource.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Size2D efl_ui_animation_view_default_view_size_get(const Eo *obj);

/**
 * @brief Current animation view state. See @ref Efl_Ui_Animation_View_State
 *
 * @param[in] obj The object.
 *
 * @return Current animation view state
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Efl_Ui_Animation_View_State efl_ui_animation_view_state_get(const Eo *obj);

/**
 * @brief Returns the status whether current animation is on playing forward or
 * backward. Note: If animation view is not on playing, it will return
 * @c false.
 *
 * @param[in] obj The object.
 *
 * @return @c true, if animation on playing back, @c false otherwise.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI Eina_Bool efl_ui_animation_view_is_playing_back(Eo *obj);

/**
 * @brief The index of end frame of the animation view, if it's animated. Note
 * : frame number starts with 0.
 *
 * @param[in] obj The object.
 *
 * @return The number of frames. 0, if it's not animated.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI int efl_ui_animation_view_frame_count_get(const Eo *obj);

/**
 * @brief The start progress of the play.
 *
 * @param[in] obj The object.
 * @param[in] min_progress The minimum progress. Value must be 0 ~ 1.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI void efl_ui_animation_view_min_progress_set(Eo *obj, double min_progress);

/**
 * @brief The start progress of the play.
 *
 * @param[in] obj The object.
 *
 * @return The minimum progress. Value must be 0 ~ 1.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI double efl_ui_animation_view_min_progress_get(const Eo *obj);

/**
 * @brief The last progress of the play.
 *
 * @param[in] obj The object.
 * @param[in] max_progress The maximum progress. Value must be 0 ~ 1.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI void efl_ui_animation_view_max_progress_set(Eo *obj, double max_progress);

/**
 * @brief The last progress of the play.
 *
 * @param[in] obj The object.
 *
 * @return The maximum progress. Value must be 0 ~ 1.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI double efl_ui_animation_view_max_progress_get(const Eo *obj);

/**
 * @brief The start frame of the play.
 *
 * @param[in] obj The object.
 * @param[in] min_frame The minimum frame for play. Value must be 0 ~
 * @ref efl_ui_animation_view_max_frame_get
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI void efl_ui_animation_view_min_frame_set(Eo *obj, int min_frame);

/**
 * @brief The start frame of the play.
 *
 * @param[in] obj The object.
 *
 * @return The minimum frame for play. Value must be 0 ~
 * @ref efl_ui_animation_view_max_frame_get
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI int efl_ui_animation_view_min_frame_get(const Eo *obj);

/**
 * @brief The last frame of the play.
 *
 * @param[in] obj The object.
 * @param[in] max_frame The maximum frame for play. Value must be
 * @ref efl_ui_animation_view_min_frame_get ~
 * (@Efl.Ui.Animation_View.frame_count - 1). The default value is
 * @ref efl_ui_animation_view_frame_count_get - 1.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI void efl_ui_animation_view_max_frame_set(Eo *obj, int max_frame);

/**
 * @brief The last frame of the play.
 *
 * @param[in] obj The object.
 *
 * @return The maximum frame for play. Value must be
 * @ref efl_ui_animation_view_min_frame_get ~
 * (@Efl.Ui.Animation_View.frame_count - 1). The default value is
 * @ref efl_ui_animation_view_frame_count_get - 1.
 *
 * @ingroup Efl_Ui_Animation_View
 */
EOAPI int efl_ui_animation_view_max_frame_get(const Eo *obj);

EWAPI extern const Efl_Event_Description _EFL_UI_ANIMATION_VIEW_EVENT_PLAY_START;

/** Called when animation is just started
 *
 * @ingroup Efl_Ui_Animation_View
 */
#define EFL_UI_ANIMATION_VIEW_EVENT_PLAY_START (&(_EFL_UI_ANIMATION_VIEW_EVENT_PLAY_START))

EWAPI extern const Efl_Event_Description _EFL_UI_ANIMATION_VIEW_EVENT_PLAY_REPEAT;

/** Called when animation is just repeated
 *
 * @ingroup Efl_Ui_Animation_View
 */
#define EFL_UI_ANIMATION_VIEW_EVENT_PLAY_REPEAT (&(_EFL_UI_ANIMATION_VIEW_EVENT_PLAY_REPEAT))

EWAPI extern const Efl_Event_Description _EFL_UI_ANIMATION_VIEW_EVENT_PLAY_DONE;

/** Called when animation is just finished
 *
 * @ingroup Efl_Ui_Animation_View
 */
#define EFL_UI_ANIMATION_VIEW_EVENT_PLAY_DONE (&(_EFL_UI_ANIMATION_VIEW_EVENT_PLAY_DONE))

EWAPI extern const Efl_Event_Description _EFL_UI_ANIMATION_VIEW_EVENT_PLAY_PAUSE;

/** Called when animation is just paused
 *
 * @ingroup Efl_Ui_Animation_View
 */
#define EFL_UI_ANIMATION_VIEW_EVENT_PLAY_PAUSE (&(_EFL_UI_ANIMATION_VIEW_EVENT_PLAY_PAUSE))

EWAPI extern const Efl_Event_Description _EFL_UI_ANIMATION_VIEW_EVENT_PLAY_RESUME;

/** Called when animation is just resumed
 *
 * @ingroup Efl_Ui_Animation_View
 */
#define EFL_UI_ANIMATION_VIEW_EVENT_PLAY_RESUME (&(_EFL_UI_ANIMATION_VIEW_EVENT_PLAY_RESUME))

EWAPI extern const Efl_Event_Description _EFL_UI_ANIMATION_VIEW_EVENT_PLAY_STOP;

/** Called when animation is just stopped
 *
 * @ingroup Efl_Ui_Animation_View
 */
#define EFL_UI_ANIMATION_VIEW_EVENT_PLAY_STOP (&(_EFL_UI_ANIMATION_VIEW_EVENT_PLAY_STOP))

EWAPI extern const Efl_Event_Description _EFL_UI_ANIMATION_VIEW_EVENT_PLAY_UPDATE;

/** Called when animation is just updated
 *
 * @ingroup Efl_Ui_Animation_View
 */
#define EFL_UI_ANIMATION_VIEW_EVENT_PLAY_UPDATE (&(_EFL_UI_ANIMATION_VIEW_EVENT_PLAY_UPDATE))
#endif /* EFL_BETA_API_SUPPORT */

#endif
