"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _path = _interopRequireDefault(require("path"));

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _yubikiri = _interopRequireDefault(require("yubikiri"));

var _eventKit = require("event-kit");

var _observeModel = _interopRequireDefault(require("../views/observe-model"));

var _resolutionProgress = _interopRequireDefault(require("../models/conflicts/resolution-progress"));

var _editorConflictController = _interopRequireDefault(require("./editor-conflict-controller"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const DEFAULT_REPO_DATA = {
  mergeConflictPaths: [],
  isRebasing: false
};
/**
 * Render an `EditorConflictController` for each `TextEditor` open on a file that contains git conflict markers.
 */

class RepositoryConflictController extends _react["default"].Component {
  constructor(props, context) {
    super(props, context);

    _defineProperty(this, "fetchData", repository => {
      return (0, _yubikiri["default"])({
        workingDirectoryPath: repository.getWorkingDirectoryPath(),
        mergeConflictPaths: repository.getMergeConflicts().then(conflicts => {
          return conflicts.map(conflict => conflict.filePath);
        }),
        isRebasing: repository.isRebasing()
      });
    });

    this.state = {
      openEditors: this.props.workspace.getTextEditors()
    };
    this.subscriptions = new _eventKit.CompositeDisposable();
  }

  componentDidMount() {
    const updateState = () => {
      this.setState({
        openEditors: this.props.workspace.getTextEditors()
      });
    };

    this.subscriptions.add(this.props.workspace.observeTextEditors(updateState), this.props.workspace.onDidDestroyPaneItem(updateState), this.props.config.observe('github.graphicalConflictResolution', () => this.forceUpdate()));
  }

  render() {
    return _react["default"].createElement(_observeModel["default"], {
      model: this.props.repository,
      fetchData: this.fetchData
    }, data => this.renderWithData(data || DEFAULT_REPO_DATA));
  }

  renderWithData(repoData) {
    const conflictingEditors = this.getConflictingEditors(repoData);
    return _react["default"].createElement("div", null, conflictingEditors.map(editor => _react["default"].createElement(_editorConflictController["default"], {
      key: editor.id,
      commandRegistry: this.props.commandRegistry,
      resolutionProgress: this.props.resolutionProgress,
      editor: editor,
      isRebase: repoData.isRebasing,
      refreshResolutionProgress: this.props.refreshResolutionProgress
    })));
  }

  getConflictingEditors(repoData) {
    if (repoData.mergeConflictPaths.length === 0 || this.state.openEditors.length === 0 || !this.props.config.get('github.graphicalConflictResolution')) {
      return [];
    }

    const commonBasePath = this.props.repository.getWorkingDirectoryPath();
    const fullMergeConflictPaths = new Set(repoData.mergeConflictPaths.map(relativePath => _path["default"].join(commonBasePath, relativePath)));
    return this.state.openEditors.filter(editor => fullMergeConflictPaths.has(editor.getPath()));
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

}

exports["default"] = RepositoryConflictController;

_defineProperty(RepositoryConflictController, "propTypes", {
  workspace: _propTypes["default"].object.isRequired,
  commandRegistry: _propTypes["default"].object.isRequired,
  config: _propTypes["default"].object.isRequired,
  resolutionProgress: _propTypes["default"].object.isRequired,
  repository: _propTypes["default"].object.isRequired,
  refreshResolutionProgress: _propTypes["default"].func
});

_defineProperty(RepositoryConflictController, "defaultProps", {
  refreshResolutionProgress: () => {},
  resolutionProgress: new _resolutionProgress["default"]()
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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